/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "areainterface.h"
#include <unistd.h>
#include <QDBusConnection>

AreaInterface::AreaInterface()
{
    ukccPanelGsettings = UniversalInterface::self()->ukccPanelGsettings();
    accountInterface   = UniversalInterface::self()->accountUserDbus();
    fcitxInterface     = UniversalInterface::self()->fcitxInputMethodDbus();
    styleGsettings     = UniversalInterface::self()->ukuiStyleGsettings();
    connect(ukccPanelGsettings, &QGSettings::changed, this, [=](QString key) {
        if (key == CALENDAR_KEY) {
            mCalendar.clear();
            getCalendar();
            Q_EMIT changed("calendar");
        } else if (key == FIRST_DAY_KEY) {
            mFirstDay.clear();
            getFirstDay();
            Q_EMIT changed("firstDay");
        } else if (key == DATE_FORMATE_KEY) {
            mDateFormat.clear();
            getDateFormat();
            Q_EMIT changed("dateFormat");
        } else if (key == TIME_FORMAT_KEY) {
            mTimeFormat.clear();
            getTimeFormat();
            Q_EMIT changed("timeFormat");
        } else if (key == SHOW_LANGUAGE_KEY) {
            mShowLanguageList.clear();
            getShowLanguageList();
            Q_EMIT changed("showLanguageList");
        }
    });

    connect(styleGsettings, &QGSettings::changed, this, [=](QString key) {
        if (key == "iconThemeName") {
            Q_EMIT changed("iconThemeName");
        }
    });
}

AreaInterface::~AreaInterface()
{

}

const QString AreaInterface::gsettingGet(const QString &key) const
{
    if (ukccPanelGsettings->keys().contains(key)) {
        return ukccPanelGsettings->get(key).toString();
    } else {
        return "";
    }
}

void AreaInterface::gsettingSet(const QString &key, const QString &value)
{
    if (ukccPanelGsettings->keys().contains(key)) {
        ukccPanelGsettings->set(key, value);
    }
}

const QStringList AreaInterface::getFormatCountryList()
{
    if (mFormatCountryList.isEmpty()) {
        mFormatCountryList.append(QString("en_US.UTF-8"));
        mFormatCountryList.append(QString("zh_CN.UTF-8"));
        mFormatCountryList.append(QString("bo_CN.UTF-8"));
    }
    return mFormatCountryList;
}

const QStringList AreaInterface::getCalendarList()
{
    if (mCalendarList.isEmpty()) {
        mCalendarList.append(QString("solarlunar"));
        QString locale = QLocale::system().name();
        if ("zh_CN" == locale) {
            mCalendarList.append(QString("lunar"));
        }
    }
    return mCalendarList;
}

const QStringList AreaInterface::getFirstDayList()
{
    if (mFirstDayList.isEmpty()) {
        mFirstDayList.append("monday");
        mFirstDayList.append("sunday");
    }
    return mFirstDayList;
}

const QStringList AreaInterface::getDateFormatList()
{
    if (mDateFormatList.isEmpty()) {
        mDateFormatList.append("cn");
        mDateFormatList.append("en");
    }
    return mDateFormatList;
}

const QStringList AreaInterface::getTimeFormatList()
{
    if (mtimeFormatList.isEmpty()) {
        mtimeFormatList.append("12");
        mtimeFormatList.append("24");
    }
    return mtimeFormatList;
}

const QStringList AreaInterface::getShowLanguageList()
{
    if (mShowLanguageList.isEmpty()) {
        if (ukccPanelGsettings->keys().contains(SHOW_LANGUAGE_KEY)) {
            mShowLanguageList = ukccPanelGsettings->get(SHOW_LANGUAGE_KEY).toStringList();
        } else {
            qCritical() << "get SHOW_LANGUAGE_KEY error.";
        }
    }
    return mShowLanguageList;
}

const QStringList AreaInterface::getLanguageList()
{
    if (mLanguageList.isEmpty()) {
        mLanguageList.append("zh_CN");
        mLanguageList.append("en");
        mLanguageList.append("bo_CN");
    }
    return mLanguageList;
}

const QString AreaInterface::getFormatCountry()
{
    if (mFormatCountry.isEmpty()) {
        mFormatCountry = accountInterface->property("FormatsLocale").toString();
    }
    return mFormatCountry;
}

const QString AreaInterface::getCalendar()
{
    if (mCalendar.isEmpty()) {
        mCalendar = gsettingGet(CALENDAR_KEY);
    }
    return mCalendar;
}

const QString AreaInterface::getFirstDay()
{
    if (mFirstDay.isEmpty()) {
        mFirstDay = gsettingGet(FIRST_DAY_KEY);
    }
    return mFirstDay;
}

const QString AreaInterface::getDateFormat()
{
    if (mDateFormat.isEmpty()) {
        mDateFormat = gsettingGet(DATE_FORMATE_KEY);
    }
    return mDateFormat;
}

const QString AreaInterface::getTimeFormat()
{
    if (mTimeFormat.isEmpty()) {
        mTimeFormat = gsettingGet(TIME_FORMAT_KEY);
    }
    return mTimeFormat;
}

const QString AreaInterface::getLanguage()
{
    mLanguage = accountInterface->property("Language").toString();
    return mLanguage;
}

void AreaInterface::setFormatCountry(const QString &formatCountry)
{
    mFormatCountry = formatCountry;
    accountInterface->call("SetFormatsLocale", formatCountry);
    Q_EMIT changed("formatCountry");
}

void AreaInterface::setCalendar(const QString &calendar)
{
    gsettingSet(CALENDAR_KEY, calendar);
}

void AreaInterface::setFirstDay(const QString &firstDay)
{
    gsettingSet(FIRST_DAY_KEY, firstDay);
}

void AreaInterface::setDateFormat(const QString &dateFormat)
{
    gsettingSet(DATE_FORMATE_KEY, dateFormat);
}

void AreaInterface::setTimeFormat(const QString &timeFormat)
{
    gsettingSet(TIME_FORMAT_KEY, timeFormat);
}

void AreaInterface::setShowLanguageList(const QStringList &languageList)
{
    if (ukccPanelGsettings->keys().contains(SHOW_LANGUAGE_KEY)) {
        ukccPanelGsettings->set(SHOW_LANGUAGE_KEY, languageList);
    } else {
        qCritical() << "set SHOW_LANGUAGE_KEY error:" << languageList;
    }
}

void AreaInterface::setLanguage(const QString &languageCode)
{
    accountInterface->call("SetLanguage", languageCode);
    Q_EMIT changed("language");
}

void AreaInterface::reloadInputMethodConfig()
{
    fcitxInterface->call("ReloadConfig");
}

void AreaInterface::changeInputMethod(const QString &inputMethod, const bool &add)
{
    QString fileTextString;
    QFile configFile(FCITX_CONFIG_FILE);
    if (configFile.open(QIODevice::ReadWrite | QIODevice::Text)) {
        //把文件所有信息读出来
        QTextStream stream(&configFile);
        fileTextString = stream.readAll();

        if (!fileTextString.contains(inputMethod)) {
            return ;
        }
        QString addString    = inputMethod  + ":True";
        QString removeString =  inputMethod + ":False";
        if (add) {
            fileTextString.replace(removeString, addString);
        } else {
            fileTextString.replace(addString, removeString);
        }
        stream << fileTextString;
        configFile.close();
        reloadInputMethodConfig();
    }
}

bool AreaInterface::isAddedInputMethod(const QString &inputMethod)
{
    QString fileTextString;
    QFile mFile(FCITX_CONFIG_FILE);
    if (mFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QTextStream stream(&mFile);
        fileTextString = stream.readAll();
        QString string = inputMethod  + ":True";
        if (fileTextString.contains(string)) {
           return true;
        } else {
            return false;
        }
    }
    qCritical()<<"open "<<FCITX_CONFIG_FILE<<" error.";
    return false;
}
