/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNotNull;
import static org.testng.AssertJUnit.assertNull;

import org.jboss.cache.Fqn;
import org.jboss.cache.Region;
import org.jboss.cache.RegionManager;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;
/**
 * Unit tests for MRUAlgorithm.
 *
 * @author Daniel Huang (dhuang@jboss.org)
 * @version $Revision: 4444 $
 */
@Test(groups = {"functional"})
public class MRUAlgorithmTest
{
   MRUAlgorithm algorithm;
   RegionManager regionManager;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      algorithm = new MRUAlgorithm();
      MRUConfiguration config = new MRUConfiguration();
      // We have to setCache maxNodes!!
      config.setMaxNodes(0);
      config.setEvictionPolicyClass(DummyEvictionPolicy.class.getName());
      regionManager = new RegionManager();
      regionManager.getRegion("/a/b", true).setEvictionPolicy(config);
   }

   public void testMaxNodes() throws Exception
   {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager.getRegion("/a/b", true);
      MRUConfiguration config = (MRUConfiguration) region.getEvictionPolicyConfig();
      config.setMaxNodes(1);
      region.putNodeEvent(new EvictedEventNode(fqn1, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn2, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn3, NodeEventType.ADD_NODE_EVENT));
      algorithm.process(region);

      assertEquals(1, algorithm.getEvictionQueue().getNumberOfNodes());

      config.setMaxNodes(100);
      for (int i = 0; i < 150; i++)
      {
         Fqn fqn = Fqn.fromString("/a/b/c/" + Integer.toString(i));
         region.putNodeEvent(new EvictedEventNode(fqn, NodeEventType.ADD_NODE_EVENT));
      }

      algorithm.process(region);

      assertEquals(100, algorithm.getEvictionQueue().getNumberOfNodes());
   }

   public void testMRU() throws Exception
   {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Fqn fqn4 = Fqn.fromString("/a/b/f");
      Fqn fqn5 = Fqn.fromString("/a/b/g");
      Fqn fqn6 = Fqn.fromString("/a/b/h");
      Fqn fqn7 = Fqn.fromString("/a/b/i");
      Fqn fqn8 = Fqn.fromString("/a/b/j");
      Fqn fqn9 = Fqn.fromString("/a/b/k");
      Fqn fqn10 = Fqn.fromString("/a/b/l");
      Region region = regionManager.getRegion("/a/b", true);
      MRUConfiguration config = (MRUConfiguration) region.getEvictionPolicyConfig();
      config.setMaxNodes(8);
      region.putNodeEvent(new EvictedEventNode(fqn1, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn2, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn3, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn4, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn5, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn6, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn7, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn8, NodeEventType.ADD_NODE_EVENT));

      algorithm.process(region);

      assertEquals(8, algorithm.getEvictionQueue().getNumberOfNodes());

      region.putNodeEvent(new EvictedEventNode(fqn9, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn10, NodeEventType.ADD_NODE_EVENT));

      Thread.sleep(5000);
      assertEquals(8, algorithm.getEvictionQueue().getNumberOfNodes());

      region.putNodeEvent(new EvictedEventNode(fqn2, NodeEventType.ADD_NODE_EVENT));
      region.putNodeEvent(new EvictedEventNode(fqn4, NodeEventType.ADD_NODE_EVENT));

      algorithm.process(region);

      assertEquals(8, algorithm.getEvictionQueue().getNumberOfNodes());

      assertNull(algorithm.getEvictionQueue().getNodeEntry(fqn2));
      assertNull("No FQN4 " + algorithm.getEvictionQueue(),
            algorithm.getEvictionQueue().getNodeEntry(fqn4));

      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn1));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn3));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn5));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn6));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn7));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn8));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn9));
      assertNotNull(algorithm.getEvictionQueue().getNodeEntry(fqn10));
   }

}
