/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.cache.eviction;

import static org.testng.AssertJUnit.fail;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheSPI;
import org.jboss.cache.Fqn;
import org.jboss.cache.Region;
import org.jboss.cache.RegionManager;
import org.jboss.cache.config.ConfigurationException;
import org.jboss.cache.config.EvictionPolicyConfig;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

/**
 * Tests BaseEvictionAlgorithm class.
 * 
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 */
@Test(groups = "functional")
public class BaseEvictionAlgorithmTest
{
   private static final Log log = LogFactory.getLog(BaseEvictionAlgorithmTest.class);  
   
   private RegionManager regionManager;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      regionManager = new RegionManager();
   }
   
   public void testFillUpRecycleQueue() throws Exception
   {
      final int recycleQueueCapacity = 10;

      /* override recycle queue capacity to make the test shorter */
      BaseEvictionAlgorithm algorithm = new MockEvictionAlgorithm(recycleQueueCapacity);
      
      Region region = regionManager.getRegion("/a/b/c", true);
      region.setEvictionPolicy(new MockEvictionPolicyConfig());
      
      for (int i = 0; i < (recycleQueueCapacity + 1); i ++)
      {
         Fqn<String> fqn = Fqn.fromString("/a/b/c/" + Integer.toString(i + 1));
         region.putNodeEvent(new EvictedEventNode(fqn, NodeEventType.ADD_NODE_EVENT));         
      }
      
      ExecutorService executor = Executors.newSingleThreadExecutor();
      Future<Void> future = executor.submit(new ProcessEvictionRegion(region, algorithm));
      
      try
      {
         future.get(20, TimeUnit.SECONDS);
      }
      catch(TimeoutException te)
      {
         log.error("Region eviction processing did not finish on time", te);
         fail("Region eviction processing should have finished by now, something is wrong. Recycle queue may have filled up.");
      }
      finally
      {
         log.info("recycle queue size: " + algorithm.recycleQueue.size());
      }
   }
   
   /** Classes **/
   
   public static class MockEvictionAlgorithm extends BaseEvictionAlgorithm
   {
      public MockEvictionAlgorithm(int recycleQueueCapacity)
      {
         recycleQueue = new LinkedBlockingQueue<Fqn>(recycleQueueCapacity);
      }
      
      @Override
      protected EvictionQueue setupEvictionQueue(Region region) throws EvictionException
      {
         return new LRUQueue();
      }

      @Override
      protected boolean shouldEvictNode(NodeEntry ne)
      {
         /* all node entries need evicting */
         return true;
      }
      
   }

   public static class MockEvictionPolicy extends BaseEvictionPolicy
   {
      
      @Override
      public void evict(Fqn fqn) throws Exception
      {
         throw new Exception("Unable to evict");
      }

      @Override
      public void setCache(CacheSPI cache)
      {
         /* no op */
      }

      public EvictionAlgorithm getEvictionAlgorithm()
      {
         return null;
      }

      public Class<? extends EvictionPolicyConfig> getEvictionConfigurationClass()
      {
         return MockEvictionPolicyConfig.class;
      }
   }
   
   public static class MockEvictionPolicyConfig implements EvictionPolicyConfig
   {

      public String getEvictionPolicyClass()
      {
         return MockEvictionPolicy.class.getName();
      }

      public void reset()
      {
         /* no op */
      }

      public void validate() throws ConfigurationException
      {
         /* no op */
      }      
   }
   
   public class ProcessEvictionRegion implements Callable<Void>
   {
      private Region region;
      
      private EvictionAlgorithm algorithm;
      
      public ProcessEvictionRegion(Region region, EvictionAlgorithm algorithm)
      {
         this.region = region;
         this.algorithm = algorithm;
      }

      public Void call() throws Exception
      {
         try
         {
            algorithm.process(region);
         }
         catch(EvictionException e)
         {
            log.error("Eviction exception reported", e);
            fail("Eviction exception reported" + e);
         }
         
         return null;
      }
   }
}
