package org.jboss.cache.interceptors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.commands.VisitableCommand;
import org.jboss.cache.factories.annotations.Inject;
import org.jboss.cache.factories.annotations.Start;
import org.jboss.cache.interceptors.base.CommandInterceptor;
import org.jboss.cache.invocation.InvocationContextContainer;
import org.jboss.cache.util.CachePrinter;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.ArrayList;

/**
 * Knows how to build and manage an chain of interceptors. Also in charge with invoking methods on the chain.
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class InterceptorChain
{
   /**
    * reference to the first interceptor in the chain
    */
   private CommandInterceptor firstInChain;

   /**
    * used for invoking commands on the chain
    */
   private InvocationContextContainer invocationContextContainer;
   private static final Log log = LogFactory.getLog(InterceptorChain.class);

   /**
    * Constructs an interceptor chain having the supplied interceptor as first.
    */
   public InterceptorChain(CommandInterceptor first)
   {
      this.firstInChain = first;
   }

   @Inject
   public void initialize(InvocationContextContainer invocationContextContainer)
   {
      this.invocationContextContainer = invocationContextContainer;
   }

   @Start
   private void printChainInfo()
   {
      if (log.isDebugEnabled()) log.debug("Interceptor chain is: " + toString());
   }

   /**
    * Inserts the given interceptor at the specified position in the chain (o based indexing).
    *
    * @throws IllegalArgumentException if the position is invalid (e.g. 5 and there are only 2 interceptors in the chain)
    */
   public synchronized void addInterceptor(CommandInterceptor interceptor, int position)
   {
      if (position == 0)
      {
         interceptor.setNext(firstInChain);
         firstInChain = interceptor;
         return;
      }
      if (firstInChain == null) return;
      CommandInterceptor it = firstInChain;
      int index = 0;
      while (it != null)
      {
         if (++index == position)
         {
            interceptor.setNext(it.getNext());
            it.setNext(interceptor);
            return;
         }
         it = it.getNext();
      }
      throw new IllegalArgumentException("Invalid index: " + index + " !");
   }

   /**
    * Removes the interceptor at the given postion.
    *
    * @throws IllegalArgumentException if the position is invalid (e.g. 5 and there are only 2 interceptors in the chain)
    */
   public synchronized void removeInterceptor(int position)
   {
      if (firstInChain == null) return;
      if (position == 0)
      {
         firstInChain = firstInChain.getNext();
         return;
      }
      CommandInterceptor it = firstInChain;
      int index = 0;
      while (it != null)
      {
         if (++index == position)
         {
            if (it.getNext() == null) return; //nothing to remove
            it.setNext(it.getNext().getNext());
            return;
         }
         it = it.getNext();
      }
      throw new IllegalArgumentException("Invalid position: " + position + " !");
   }

   /**
    * Returns the number of interceptors in the chain.
    */
   public int size()
   {
      int size = 0;
      CommandInterceptor it = firstInChain;
      while (it != null)
      {
         size++;
         it = it.getNext();
      }
      return size;

   }

   /**
    * Returns an unmofiable list with all the interceptors in sequence.
    * If first in chain is null an empty list is returned.
    */
   public List<CommandInterceptor> asList()
   {
      if (firstInChain == null) return Collections.emptyList();

      List<CommandInterceptor> retval = new LinkedList<CommandInterceptor>();
      CommandInterceptor tmp = firstInChain;
      do
      {
         retval.add(tmp);
         tmp = tmp.getNext();
      }
      while (tmp != null);
      return Collections.unmodifiableList(retval);
   }


   /**
    * Removes all the occurences of supplied interceptor type from the chain.
    */
   public synchronized void removeInterceptor(Class<? extends CommandInterceptor> clazz)
   {
      if (firstInChain.getClass() == clazz)
      {
         firstInChain = firstInChain.getNext();
      }
      CommandInterceptor it = firstInChain.getNext();
      CommandInterceptor prevIt = firstInChain;
      while (it != null)
      {
         if (it.getClass() == clazz)
         {
            prevIt.setNext(it.getNext());
         }
         prevIt = it;
         it = it.getNext();
      }
   }

   /**
    * Adds a new interceptor in list after an interceptor of a given type.
    *
    * @return true if the interceptor was added; i.e. the afterInterceptor exists
    */
   public synchronized boolean addInterceptor(CommandInterceptor toAdd, Class<? extends CommandInterceptor> afterInterceptor)
   {
      CommandInterceptor it = firstInChain;
      while (it != null)
      {
         if (it.getClass().equals(afterInterceptor))
         {
            toAdd.setNext(it.getNext());
            it.setNext(toAdd);
            return true;
         }
         it = it.getNext();
      }
      return false;
   }

   /**
    * Appends at the end.
    */
   public void appendIntereceptor(CommandInterceptor ci)
   {
      CommandInterceptor it = firstInChain;
      while (it.hasNext()) it = it.getNext();
      it.setNext(ci);
      // make sure we nullify the "next" pointer in the last interceptor.
      ci.setNext(null);
   }

   /**
    * Walks the command through the interceptor chain. The received ctx is being passed in.
    */
   @SuppressWarnings("deprecation")
   public Object invoke(InvocationContext ctx, VisitableCommand command)
   {
      ctx.setCommand(command);
      try
      {
         return command.acceptVisitor(ctx, firstInChain);
      }
      catch (CacheException e)
      {
         throw e;
      }
      catch (RuntimeException e)
      {
         throw e;
      }
      catch (Throwable t)
      {
         throw new CacheException(t);
      }
   }

   /**
    * Similar to {@link #invoke(org.jboss.cache.InvocationContext, org.jboss.cache.commands.VisitableCommand)}, but
    * constructs a invocation context on the fly, using {@link InvocationContextContainer#get()}
    */
   public Object invokeRemote(VisitableCommand cacheCommand) throws Throwable
   {
      InvocationContext ctxt = invocationContextContainer.get();
      ctxt.setOriginLocal(false);
      return cacheCommand.acceptVisitor(ctxt, firstInChain);
   }

   /**
    * Similar to {@link #invoke(org.jboss.cache.InvocationContext, org.jboss.cache.commands.VisitableCommand)}, but
    * constructs a invocation context on the fly, using {@link InvocationContextContainer#get()} and setting the origin local flag to it's default value.
    */
   public Object invoke(VisitableCommand cacheCommand) throws Throwable
   {
      InvocationContext ctxt = invocationContextContainer.get();
      return cacheCommand.acceptVisitor(ctxt, firstInChain);
   }

   /**
    * @return the first interceptor in the chain.
    */
   public CommandInterceptor getFirstInChain()
   {
      return firstInChain;
   }

   /**
    * Mainly used by unit tests to replace the interceptor chain with the starting point passed in.
    *
    * @param interceptor interceptor to be used as the first interceptor in the chain.
    */
   public void setFirstInChain(CommandInterceptor interceptor)
   {
      this.firstInChain = interceptor;
   }

   public InvocationContext getInvocationContext()
   {
      return invocationContextContainer.get();
   }

   /**
    * Returns all interceptors which extend the given command interceptor.
    */
   public List<CommandInterceptor> getInterceptorsWhichExtend(Class<? extends CommandInterceptor> interceptorClass)
   {
      List<CommandInterceptor> result = new ArrayList<CommandInterceptor>();
      for (CommandInterceptor interceptor : asList())
      {
         boolean isSubclass = interceptorClass.isAssignableFrom(interceptor.getClass());
         if (isSubclass)
         {
            result.add(interceptor);
         }
      }
      return result;
   }

   public String toString()
   {
      return "InterceptorChain{\n" +
            CachePrinter.printInterceptorChain(firstInChain) +
            "\n}";
   }
}
