------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S W I T C H - C                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--          Copyright (C) 2001-2026, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT; see file COPYING3.  If not, go to --
-- http://www.gnu.org/licenses for a complete copy of the license.          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  This package is for switch processing and should not depend on higher level
--  packages such as those for the scanner, parser, etc. Doing so may cause
--  circularities, especially for back ends using Adabkend.

with Debug;    use Debug;
with Lib;      use Lib;
with Osint;    use Osint;
with Opt;      use Opt;
with Stylesw;  use Stylesw;
with Targparm; use Targparm;
with Ttypes;   use Ttypes;
with Validsw;  use Validsw;
with Warnsw;   use Warnsw;

with Ada.Unchecked_Deallocation;

with System.WCh_Con; use System.WCh_Con;
with System.OS_Lib;

package body Switch.C is

   RTS_Specified : String_Access := null;
   --  Used to detect multiple use of --RTS= flag

   procedure Add_Symbol_Definition (Def : String);
   --  Add a symbol definition from the command line

   procedure Free is
      new Ada.Unchecked_Deallocation (String_List, String_List_Access);
   --  Avoid using System.Strings.Free, which also frees the designated strings

   function Get_Overflow_Mode (C : Character) return Overflow_Mode_Type;
   --  Given a digit in the range 0 .. 3, returns the corresponding value of
   --  Overflow_Mode_Type. Raises Program_Error if C is outside this range.

   function Switch_Subsequently_Cancelled
     (C        : String;
      Args     : String_List;
      Arg_Rank : Positive) return Boolean;
   --  This function is called from Scan_Front_End_Switches. It determines if
   --  the switch currently being scanned is followed by a switch of the form
   --  "-gnat-" & C, where C is the argument. If so, then True is returned,
   --  and Scan_Front_End_Switches will cancel the effect of the switch. If
   --  no such switch is found, False is returned.

   ---------------------------
   -- Add_Symbol_Definition --
   ---------------------------

   procedure Add_Symbol_Definition (Def : String) is
   begin
      --  If Preprocessor_Symbol_Defs is not large enough, double its size

      if Preprocessing_Symbol_Last = Preprocessing_Symbol_Defs'Last then
         declare
            New_Symbol_Definitions : constant String_List_Access :=
              new String_List (1 .. 2 * Preprocessing_Symbol_Last);
         begin
            New_Symbol_Definitions (Preprocessing_Symbol_Defs'Range) :=
              Preprocessing_Symbol_Defs.all;
            Free (Preprocessing_Symbol_Defs);
            Preprocessing_Symbol_Defs := New_Symbol_Definitions;
         end;
      end if;

      Preprocessing_Symbol_Last := Preprocessing_Symbol_Last + 1;
      Preprocessing_Symbol_Defs (Preprocessing_Symbol_Last) :=
        new String'(Def);
   end Add_Symbol_Definition;

   -----------------------
   -- Get_Overflow_Mode --
   -----------------------

   function Get_Overflow_Mode (C : Character) return Overflow_Mode_Type is
   begin
      case C is
         when '1' =>
            return Strict;

         when '2' =>
            return Minimized;

         --  Eliminated allowed only if Long_Long_Integer is 64 bits (since
         --  the current implementation of System.Bignums assumes this).

         when '3' =>
            if Standard_Long_Long_Integer_Size /= 64 then
               Bad_Switch ("-gnato3 requires Long_Long_Integer'Size = 64");
            else
               return Eliminated;
            end if;

         when others =>
            raise Program_Error;
      end case;
   end Get_Overflow_Mode;

   -----------------------------
   -- Scan_Front_End_Switches --
   -----------------------------

   procedure Scan_Front_End_Switches
     (Switch_Chars : String;
      Args         : String_List;
      Arg_Rank     : Positive)
   is
      Max : constant Natural := Switch_Chars'Last;
      C   : Character := ' ';
      Ptr : Natural;

      Dot : Boolean;
      --  This flag is set upon encountering a dot in a debug switch

      First_Char : Positive;
      --  Marks start of switch to be stored

      First_Ptr : Positive;
      --  Save position of first character after -gnatd (for checking that
      --  debug flags that must come first are first, in particular -gnatd.b).

      First_Switch : Boolean := True;
      --  False for all but first switch

      Store_Switch : Boolean;
      --  For -gnatxx switches, the normal processing, signalled by this flag
      --  being set to True, is to store the switch on exit from the case
      --  statement, the switch stored is -gnat followed by the characters
      --  from First_Char to Ptr-1. For cases like -gnaty, where the switch
      --  is stored in separate pieces, this flag is set to False, and the
      --  appropriate calls to Store_Compilation_Switch are made from within
      --  the case branch.

      Underscore : Boolean;
      --  This flag is set upon encountering an underscode in a debug switch

   begin
      Ptr := Switch_Chars'First;

      --  Skip past the initial character (must be the switch character)

      if Ptr = Max then
         Bad_Switch (C);
      else
         Ptr := Ptr + 1;
      end if;

      --  Handle switches that do not start with -gnat

      if Ptr + 3 > Max or else Switch_Chars (Ptr .. Ptr + 3) /= "gnat" then

         --  There are two front-end switches that do not start with -gnat:
         --  -I, --RTS

         if Switch_Chars (Ptr) = 'I' then

            --  Set flag Search_Directory_Present if switch is "-I" only:
            --  the directory will be the next argument.

            if Ptr = Max then
               Search_Directory_Present := True;
               return;
            end if;

            Ptr := Ptr + 1;

            --  Find out whether this is a -I- or regular -Ixxx switch

            --  Note: -I switches are not recorded in the ALI file, since the
            --  meaning of the program depends on the source files compiled,
            --  not where they came from.

            if Ptr = Max and then Switch_Chars (Ptr) = '-' then
               Look_In_Primary_Dir := False;
            else
               Add_Src_Search_Dir (Switch_Chars (Ptr .. Max));
            end if;

         --  Processing of the --RTS switch. --RTS may have been modified by
         --  gcc into -fRTS (for GCC targets).

         elsif Ptr + 3 <= Max
           and then (Switch_Chars (Ptr .. Ptr + 3) = "fRTS"
                       or else
                     Switch_Chars (Ptr .. Ptr + 3) = "-RTS")
         then
            Ptr := Ptr + 1;

            if Ptr + 4 > Max
              or else Switch_Chars (Ptr + 3) /= '='
            then
               Osint.Fail ("missing path for --RTS");

            else
               declare
                  Runtime_Dir : String_Access;
               begin
                  if System.OS_Lib.Is_Absolute_Path
                       (Switch_Chars (Ptr + 4 .. Max))
                  then
                     Runtime_Dir :=
                       new String'(System.OS_Lib.Normalize_Pathname
                                      (Switch_Chars (Ptr + 4 .. Max)));
                  else
                     Runtime_Dir :=
                       new String'(Switch_Chars (Ptr + 4 .. Max));
                  end if;

                  --  Valid --RTS switch

                  Opt.No_Stdinc := True;
                  Opt.RTS_Switch := True;

                  RTS_Src_Path_Name :=
                    Get_RTS_Search_Dir (Runtime_Dir.all, Include);

                  RTS_Lib_Path_Name :=
                    Get_RTS_Search_Dir (Runtime_Dir.all, Objects);

                  if RTS_Specified /= null then
                     if RTS_Src_Path_Name = null
                       or else RTS_Lib_Path_Name = null
                       or else
                         System.OS_Lib.Normalize_Pathname
                           (RTS_Specified.all) /=
                         System.OS_Lib.Normalize_Pathname
                           (RTS_Lib_Path_Name.all)
                     then
                        Osint.Fail
                          ("--RTS cannot be specified multiple times");
                     end if;

                  elsif RTS_Src_Path_Name /= null
                    and then RTS_Lib_Path_Name /= null
                  then
                     --  Store the -fRTS switch (Note: Store_Compilation_Switch
                     --  changes -fRTS back into --RTS for the actual output).

                     Store_Compilation_Switch (Switch_Chars);
                     RTS_Specified := new String'(RTS_Lib_Path_Name.all);

                  elsif RTS_Src_Path_Name = null
                    and then RTS_Lib_Path_Name = null
                  then
                     Osint.Fail ("RTS path not valid: missing "
                                 & "adainclude and adalib directories");

                  elsif RTS_Src_Path_Name = null then
                     Osint.Fail ("RTS path not valid: missing "
                                 & "adainclude directory");

                  else pragma Assert (RTS_Lib_Path_Name = null);
                     Osint.Fail ("RTS path not valid: missing "
                                 & "adalib directory");
                  end if;
               end;
            end if;

            --  There are no other switches not starting with -gnat

         else
            Bad_Switch (Switch_Chars);
         end if;

      --  Case of switch starting with -gnat

      else
         Ptr := Ptr + 4;

         --  Loop to scan through switches given in switch string

         while Ptr <= Max loop
            First_Char := Ptr;
            Store_Switch := True;

            C := Switch_Chars (Ptr);

            case C is

            --  -gnata (assertions enabled)

            when 'a' =>
               Ptr := Ptr + 1;
               Assertions_Enabled := True;

            --  -gnatA (disregard gnat.adc)

            when 'A' =>
               Ptr := Ptr + 1;
               Config_File := False;

            --  -gnatb (brief messages to stderr)

            when 'b' =>
               Ptr := Ptr + 1;
               Brief_Output := True;

            --  -gnatB (assume no invalid values)

            when 'B' =>
               Ptr := Ptr + 1;
               Assume_No_Invalid_Values := True;

            --  -gnatc (check syntax and semantics only)

            when 'c' =>
               if not First_Switch then
                  Osint.Fail
                    ("-gnatc must be first if combined with other switches");
               end if;

               Ptr := Ptr + 1;
               Check_Semantics_Only_Mode := True;
               Operating_Mode := Check_Semantics;

            --  -gnatC (Generate CodePeer information)

            when 'C' =>
               Ptr := Ptr + 1;
               CodePeer_Mode := True;

            --  -gnatd (compiler debug options)

            when 'd' =>
               Dot          := False;
               Store_Switch := False;
               Underscore   := False;

               First_Ptr := Ptr + 1;

               --  Note: for the debug switch, the remaining characters in this
               --  switch field must all be debug flags, since all valid switch
               --  characters are also valid debug characters.

               --  Loop to scan out debug flags

               while Ptr < Max loop
                  Ptr := Ptr + 1;
                  C := Switch_Chars (Ptr);
                  exit when C = ASCII.NUL or else C = '/' or else C = '-';

                  if C in '1' .. '9' or else
                     C in 'a' .. 'z' or else
                     C in 'A' .. 'Z'
                  then
                     --  Case of dotted flag

                     if Dot then
                        Set_Dotted_Debug_Flag (C);
                        Store_Compilation_Switch ("-gnatd." & C);

                        --  Special check, -gnatd.b must come first

                        if C = 'b'
                          and then (Ptr /= First_Ptr + 1
                                     or else not First_Switch)
                        then
                           Osint.Fail
                             ("-gnatd.b must be first if combined with other "
                              & "switches");
                        end if;

                     --  Case of an underscored flag

                     elsif Underscore then
                        Set_Underscored_Debug_Flag (C);
                        Store_Compilation_Switch ("-gnatd_" & C);
                        if Debug_Flag_Underscore_C then
                           Enable_CUDA_Expansion := True;
                        end if;

                     --  Normal flag

                     else
                        Set_Debug_Flag (C);
                        Store_Compilation_Switch ("-gnatd" & C);
                     end if;

                  elsif C = '.' then
                     Dot := True;

                  elsif C = '_' then
                     Underscore := True;

                  elsif Dot then
                     Bad_Switch ("-gnatd." & Switch_Chars (Ptr .. Max));

                  elsif Underscore then
                     Bad_Switch ("-gnatd_" & Switch_Chars (Ptr .. Max));

                  else
                     Bad_Switch ("-gnatd" & Switch_Chars (Ptr .. Max));
                  end if;
               end loop;

               return;

            --  -gnatD (debug expanded code)

            when 'D' =>
               Ptr := Ptr + 1;

               --  Not allowed if previous -gnatR given

               --  The reason for this prohibition is that the rewriting of
               --  Sloc values causes strange malfunctions in the tests of
               --  whether units belong to the main source. This is really a
               --  bug, but too hard to fix for a marginal capability.

               --  The proper fix is to completely redo -gnatD processing so
               --  that the tree is not messed with, and instead a separate
               --  table is built on the side for debug information generation.

               if List_Representation_Info /= 0 then
                  Osint.Fail
                    ("-gnatD not permitted since -gnatR given previously");
               end if;

               --  Scan optional integer line limit value

               if Nat_Present (Switch_Chars, Max, Ptr) then
                  Scan_Nat (Switch_Chars, Max, Ptr, Sprint_Line_Limit, 'D');
                  Sprint_Line_Limit := Nat'Max (Sprint_Line_Limit, 40);
               end if;

               --  Note: -gnatD also sets -gnatx (to turn off cross-reference
               --  generation in the ali file) since otherwise this generation
               --  gets confused by the "wrong" Sloc values put in the tree.

               Debug_Generated_Code := True;
               Xref_Active := False;

            --  -gnate? (extended switches)

            when 'e' =>
               Ptr := Ptr + 1;

               --  The -gnate? switches are all double character switches
               --  so we must always have a character after the e.

               if Ptr > Max then
                  Bad_Switch ("-gnate");
               end if;

               case Switch_Chars (Ptr) is

                  --  -gnatea (initial delimiter of explicit switches)

                  --  This is an internal switch

                  --  All switches that come before -gnatea have been added by
                  --  the GCC driver and are not stored in the ALI file.
                  --  See also -gnatez below.

                  when 'a' =>
                     Store_Switch := False;
                     Enable_Switch_Storing;
                     Ptr := Ptr + 1;

                  --  -gnateA (aliasing checks on parameters)

                  when 'A' =>
                     Ptr := Ptr + 1;
                     Check_Aliasing_Of_Parameters := True;

                  --  -gnateb (config file basenames and checksums in ALI)

                  when 'b' =>
                     Ptr := Ptr + 1;
                     Config_Files_Store_Basename := True;

                  --  -gnatec (configuration pragmas)

                  when 'c' =>
                     Store_Switch := False;
                     Ptr := Ptr + 1;

                     --  There may be an equal sign between -gnatec and
                     --  the path name of the config file.

                     if Ptr <= Max and then Switch_Chars (Ptr) = '=' then
                        Ptr := Ptr + 1;
                     end if;

                     if Ptr > Max then
                        Bad_Switch ("-gnatec");
                     end if;

                     declare
                        Config_File_Name : constant String_Access :=
                                             new String'
                                                  (Switch_Chars (Ptr .. Max));

                     begin
                        if Config_File_Names = null then
                           Config_File_Names :=
                             new String_List'(1 => Config_File_Name);

                        else
                           declare
                              New_Names : constant String_List_Access :=
                                            new String_List
                                              (1 ..
                                               Config_File_Names'Length + 1);

                           begin
                              for Index in Config_File_Names'Range loop
                                 New_Names (Index) :=
                                   Config_File_Names (Index);
                                 Config_File_Names (Index) := null;
                              end loop;

                              New_Names (New_Names'Last) := Config_File_Name;
                              Free (Config_File_Names);
                              Config_File_Names := New_Names;
                           end;
                        end if;
                     end;

                     return;

                  --  -gnateC switch (generate CodePeer messages)

                  when 'C' =>
                     Ptr := Ptr + 1;

                     if not Generate_CodePeer_Messages then
                        Generate_CodePeer_Messages := True;
                        CodePeer_Mode              := True;
                        Warning_Mode               := Normal;
                        Warning_Doc_Switch         := True;  -- -gnatw.d

                        --  Enable warnings potentially useful for non GNAT
                        --  users.

                        Constant_Condition_Warnings      := True; -- -gnatwc
                        Warn_On_Assertion_Failure        := True; -- -gnatw.a
                        Warn_On_Assumed_Low_Bound        := True; -- -gnatww
                        Warn_On_Bad_Fixed_Value          := True; -- -gnatwb
                        Warn_On_Biased_Representation    := True; -- -gnatw.b
                        Warn_On_Export_Import            := True; -- -gnatwx
                        Warn_On_No_Value_Assigned        := True; -- -gnatwv
                        Warn_On_Object_Renames_Function  := True; -- -gnatw.r
                        Warn_On_Overlap                  := True; -- -gnatw.i
                        Warn_On_Parameter_Order          := True; -- -gnatw.p
                        Warn_On_Questionable_Missing_Parens := True; -- -gnatwq
                        Warn_On_Redundant_Constructs     := True; -- -gnatwr
                        Warn_On_Suspicious_Modulus_Value := True; -- -gnatw.m
                     end if;

                  --  -gnated switch (disable atomic synchronization)

                  when 'd' =>
                     Suppress_Options.Suppress (Atomic_Synchronization) :=
                       True;

                  --  -gnateD switch (preprocessing symbol definition)

                  when 'D' =>
                     Store_Switch := False;
                     Ptr := Ptr + 1;

                     if Ptr > Max then
                        Bad_Switch ("-gnateD");
                     end if;

                     Add_Symbol_Definition (Switch_Chars (Ptr .. Max));

                     --  Store the switch

                     Store_Compilation_Switch
                       ("-gnateD" & Switch_Chars (Ptr .. Max));
                     Ptr := Max + 1;

                  --  -gnateE (extra exception information)

                  when 'E' =>
                     Exception_Extra_Info := True;
                     Ptr := Ptr + 1;

                  --  -gnatef (full source path for brief error messages and
                  --  absolute paths for -fdiagnostics-format=json)

                  when 'f' =>
                     Store_Switch := False;
                     Ptr := Ptr + 1;
                     Full_Path_Name_For_Brief_Errors := True;

                  --  -gnateF (Check_Float_Overflow)

                  when 'F' =>
                     Ptr := Ptr + 1;
                     Check_Float_Overflow := not Machine_Overflows_On_Target;

                  --  -gnateg (generate C header)

                  when 'g' =>
                     --  Special check, -gnateg must occur after -gnatc

                     if Operating_Mode /= Check_Semantics then
                        Osint.Fail
                          ("gnateg requires previous occurrence of -gnatc");
                     end if;

                     Generate_C_Header := True;
                     Ptr := Ptr + 1;

                  --  -gnateG (save preprocessor output)

                  when 'G' =>
                     Generate_Processed_File := True;
                     Ptr := Ptr + 1;

                     if Ptr <= Max
                       and then Switch_Chars (Ptr) in 'b' | 'c' | 'e'
                     then
                        case Switch_Chars (Ptr) is
                           when 'b' =>
                              Opt.Blank_Deleted_Lines         := True;
                              Opt.Comment_Deleted_Lines       := False;
                              Opt.Empty_Comment_Deleted_Lines := False;

                           when 'c' =>
                              Opt.Blank_Deleted_Lines         := False;
                              Opt.Comment_Deleted_Lines       := True;
                              Opt.Empty_Comment_Deleted_Lines := False;

                           when 'e' =>
                              Opt.Blank_Deleted_Lines         := False;
                              Opt.Comment_Deleted_Lines       := False;
                              Opt.Empty_Comment_Deleted_Lines := True;

                           when others =>
                              raise Program_Error;
                        end case;

                        Ptr := Ptr + 1;

                     --  Default to emitting blank lines for deleted lines
                     --  when generating a preprocessor output file. This is
                     --  despite the fact that when the file isn't being
                     --  generated, we emit empty comment lines for the
                     --  internally generated output (to avoid conflicts
                     --  with style switches -gnatyu and -gnatyM), but is
                     --  done for compatibility with the behavior of -gnateG
                     --  prior to adding support for empty comment lines.

                     else
                        Opt.Blank_Deleted_Lines         := True;
                        Opt.Comment_Deleted_Lines       := False;
                        Opt.Empty_Comment_Deleted_Lines := False;
                     end if;

                  --  -gnateH (set reverse Bit_Order threshold to 64)

                  when 'H' =>
                     Reverse_Bit_Order_Threshold := 64;
                     Ptr := Ptr + 1;

                  --  -gnatei (max number of instantiations)

                  when 'i' =>
                     Ptr := Ptr + 1;
                     Scan_Pos
                       (Switch_Chars, Max, Ptr, Maximum_Instantiations, C);

                  --  -gnateI (index of unit in multi-unit source)

                  when 'I' =>
                     Ptr := Ptr + 1;
                     Scan_Pos (Switch_Chars, Max, Ptr, Multiple_Unit_Index, C);

                  --  -gnatel

                  when 'l' =>
                     Ptr := Ptr + 1;
                     Elab_Info_Messages := True;

                  --  -gnateL

                  when 'L' =>
                     Ptr := Ptr + 1;
                     Elab_Info_Messages := False;

                  --  -gnatem (mapping file)

                  when 'm' =>
                     Store_Switch := False;
                     Ptr := Ptr + 1