#!/usr/bin/env python3
# encoding: utf-8

import codecs
import json
import os.path
import sys

# This script collects translator credits and locale information
# from the JSON files in ../data/i18n/locales/ (created by transifex)
# and writes the translator credits to ./data/txts/translators.lua.
# The locale information is written to ../data/i18n/locales.lua.
# It then collects all other contributors from ../data/txts/developers.json,
# and writes the developer credits to ./data/txts/developers.lua

INDENT = '   '  # one indentation level in lua


base_path = os.path.abspath(os.path.join(
    os.path.dirname(__file__), os.path.pardir))

print('Reading locales from JSON:')

source_path = os.path.normpath(base_path + '/data/i18n/locales')

if (not os.path.isdir(source_path)):
    print('Error: Path ' + source_path + ' not found.')
    sys.exit(1)

# Each language's translators live in a separate file, so we list the dir
source_files = sorted(os.listdir(source_path), key=str.lower)

lua_locales = '''-- This file is generated by utils/update_authors.py.
-- The locale data is managed in Transifex.

return {
\t-- Locales are identified by their ISO code.
\ten = {
\t\t-- Used to display the locale in the Options menu.
\t\tname = "English",

\t\t-- Defines the language's position on the list in the Options menu.
\t\tsort_name = "English",

\t\t-- The font set used, including the script's direction. See i18n/fonts.lua
\t\tfont = "default"
\t},
'''

lua_translators = '''-- This file is generated by utils/update_authors.py.
function translators() return {
'''  # translators

for source_filename in source_files:
    # Only json files, and not the template file please
    if source_filename.endswith('.json') and source_filename != 'locales_translators.json':
        source_file = open(source_path + '/' + source_filename, 'r')
        translators = json.load(source_file)
        locale_message = '- Added'

        # Parsing translator credits
        # Make sure we don't pick up untranslated stuff
        if translators['translator-list'] != 'translator-credits':
            locale_message += ' translators and'
            lua_translators += '\t{\n'  # entry
            lua_translators += f'''\t\theading = "{ translators['your-language-name'] }'''
            if translators['your-language-name-in-english'] != 'English' and \
               translators['your-language-name-in-english'] != translators['your-language-name']:
                lua_translators += f''' ({ translators['your-language-name-in-english'] })'''
            lua_translators += '",\n'
            lua_translators += '\t\tentries = {\n'  # entries
            lua_translators += '\t\t\t{\n'  # entry
            lua_translators += '\t\t\t\tmembers = {\n'  # members
            for transl_name in translators['translator-list'].split('\n'):
                lua_translators += f'\t\t\t\t\t"{ transl_name }",\n'
            lua_translators += '\t\t\t\t},\n'  # members
            lua_translators += '\t\t\t},\n'  # entry
            lua_translators += '\t\t},\n'  # entries
            lua_translators += '\t},\n'  # entry

        # Parsing locale info
        # Make sure we don't pick up untranslated stuff
        locale_code = source_filename.split('.json')[0]
        locale_message += ' locale info for ' + locale_code
        lua_locales += '\n\t' + locale_code + \
            ' = {\n'  # entry with locale code

        if translators['your-language-name'] != 'English' or locale_code == 'en':
            lua_locales += f'''\t\tname = "{ translators['your-language-name'] }",\n'''
        else:
            lua_locales += f'\t\tname = "{ locale_code }",\n'

        if translators['language-sort-name'] != 'English' or locale_code == 'en':
            lua_locales += f'''\t\tsort_name = "{ translators['language-sort-name'] }",\n'''
        else:
            lua_locales += f'\t\tsort_name = "{ locale_code }",\n'

        lua_locales += f'''\t\tfont = "{ translators['font-set'] }"\n'''
        lua_locales += '\t},'  # entry
        print(locale_message)
lua_locales += '\n}\n'

lua_translators += '} end\n'  # translators

print('Writing locales\n')
dest_filename = 'locales.lua'
dest_filepath = os.path.normpath(
    base_path + '/data/i18n') + '/' + dest_filename
dest_file = codecs.open(dest_filepath, encoding='utf-8', mode='w')
dest_file.write(lua_locales.replace('\t', INDENT))  # replace indentation
dest_file.close()

print('Writing translators\n')
dest_filename = 'translators_data.lua'
dest_filepath = os.path.normpath(
    base_path + '/data/txts') + '/' + dest_filename
dest_file = codecs.open(dest_filepath, encoding='utf-8', mode='w')
dest_file.write(lua_translators.replace('\t', INDENT))
dest_file.close()

print('Reading developers from JSON')
source_path = os.path.normpath(base_path + '/data/txts')

if (not os.path.isdir(source_path)):
    print('Error: Path ' + source_path + ' not found.')
    sys.exit(1)

source_file = open(source_path + '/developers.json', 'r')
developers = json.load(source_file)['developers']
source_file.close()

lua_string = """-- Do not edit this file - it is automatically generated
-- by utils/update_authors.py from developers.json.
"""
lua_string += 'function developers() return {'  # developers

for category in developers:
    # Unused hook for adding translators
    if 'heading' in category and category['heading'] != 'Translators':
        print('- Adding ' + category['heading'])
        lua_string += '\n\t{'  # category
        lua_string += f'''\n\t\theading = _("{ category['heading'] }"),'''
        lua_string += f'''\n\t\timage = "{ category['image'] }",'''

        lua_string += '\n\t\tentries = {'  # entries
        for subcategory in category['entries']:
            lua_string += '\n\t\t\t{'  # entry
            if 'subheading' in subcategory:
                lua_string += f'''\n\t\t\t\tsubheading = _("{ subcategory['subheading'] }"),'''
            lua_string += '\n\t\t\t\tmembers = {'
            if 'members' in subcategory:
                for m in subcategory['members']:
                    lua_string += f'\n\t\t\t\t\t"{ m }",'
            if 'translate' in subcategory:
                for m in subcategory['translate']:
                    lua_string += f'\n\t\t\t\t\t_("{ m }"),'
            lua_string += '\n\t\t\t\t},'

            lua_string += '\n\t\t\t},'  # entry
        lua_string += '\n\t\t},'  # entries
        lua_string += '\n\t},'  # category

lua_string += '\n} end\n'  # developers

print('Writing developers')
dest_filename = 'developers.lua'
dest_filepath = source_path + '/' + dest_filename
dest_file = codecs.open(dest_filepath, encoding='utf-8', mode='w')
dest_file.write(lua_string.replace('\t', INDENT))
dest_file.close()
print('Done.')
