# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the blhc workflow."""

from collections.abc import Sequence
from datetime import datetime
from typing import Any, ClassVar, NewType

from django.utils import timezone

from debusine.artifacts.models import (
    ArtifactCategory,
    BareDataCategory,
    CollectionCategory,
    DebianBlhc,
    DebianSourcePackage,
    TaskTypes,
)
from debusine.client.models import LookupChildType
from debusine.db.models import (
    Artifact,
    Collection,
    CollectionItem,
    TaskDatabase,
    WorkRequest,
    Workspace,
)
from debusine.db.models.work_requests import SkipWorkRequest
from debusine.server.collections.lookup import LookupResult, lookup_single
from debusine.server.scheduler import schedule
from debusine.server.workflows import BlhcWorkflow, WorkflowValidationError
from debusine.server.workflows.base import orchestrate_workflow
from debusine.server.workflows.models import (
    BaseWorkflowData,
    BlhcWorkflowData,
    SbuildWorkflowData,
    WorkRequestWorkflowData,
)
from debusine.server.workflows.tests.helpers import (
    SampleWorkflow,
    WorkflowTestBase,
)
from debusine.tasks.models import (
    ActionSkipIfLookupResultChanged,
    ActionUpdateCollectionWithArtifacts,
    BaseDynamicTaskData,
    BlhcFlags,
    LookupMultiple,
    LookupSingle,
    OutputData,
    RegressionAnalysis,
    RegressionAnalysisStatus,
    SbuildData,
    SbuildInput,
)
from debusine.test.test_utils import preserve_task_registry

Architecture = NewType("Architecture", str)


class BlhcWorkflowTests(WorkflowTestBase[BlhcWorkflow]):
    """Unit tests for :py:class:`BlhcWorkflow`."""

    package_build_log_artifacts: ClassVar[dict[Architecture, Artifact]]
    source_artifact: ClassVar[Artifact]

    workspace: ClassVar[Workspace]

    @classmethod
    def setUpTestData(cls) -> None:
        """Set up common data."""
        super().setUpTestData()

        cls.workspace = cls.playground.get_default_workspace()

        cls.source_artifact = cls.playground.create_source_artifact(
            name="hello", version="1.0-1"
        )

        cls.package_build_log_artifacts = {}
        for arch in ["amd64", "arm64", "all"]:
            artifact = cls.playground.create_build_log_artifact(
                source="hello", version="1.0-1", build_arch=arch
            )

            cls.package_build_log_artifacts[Architecture(arch)] = artifact

    def orchestrate(
        self,
        *,
        task_data: BlhcWorkflowData,
        source_artifact: Artifact,
        architectures: Sequence[str] | None = None,
        parent: WorkRequest | None = None,
        pipeline_task_name: str = "examplepipeline",
    ) -> WorkRequest:
        """Create and orchestrate a BlhcWorkflow."""
        if architectures is None:
            architectures = ["amd64", "arm64", "all"]

        class ExamplePipeline(
            SampleWorkflow[BaseWorkflowData, BaseDynamicTaskData]
        ):
            """Pipeline workflow."""

            TASK_NAME = pipeline_task_name

            def populate(self) -> None:
                """Populate the pipeline."""
                source_data = source_artifact.create_data()
                assert isinstance(source_data, DebianSourcePackage)
                assert architectures is not None

                sbuild = self.work_request_ensure_child_workflow(
                    task_name="sbuild",
                    task_data=SbuildWorkflowData(
                        input=SbuildInput(source_artifact=source_artifact.id),
                        target_distribution="debian:trixie",
                        architectures=list(architectures),
                    ),
                    workflow_data=WorkRequestWorkflowData(
                        display_name="sbuild", step="sbuild"
                    ),
                )
                sbuild.mark_running()
                for arch in architectures:
                    child = sbuild.create_child_worker(
                        task_name="sbuild",
                        task_data=SbuildData(
                            input=SbuildInput(
                                source_artifact=source_artifact.id
                            ),
                            host_architecture=arch,
                            environment="debian/match:codename=trixie",
                        ),
                    )
                    self.provides_artifact(
                        child,
                        ArtifactCategory.PACKAGE_BUILD_LOG,
                        name=f"buildlog-{arch}",
                        data={
                            "source_package_name": source_data.name,
                            "source_package_version": source_data.version,
                            "architecture": arch,
                        },
                    )
                sbuild.unblock_workflow_children()

                blhc = self.work_request_ensure_child_workflow(
                    task_name="blhc",
                    task_data=task_data,
                    workflow_data=WorkRequestWorkflowData(
                        display_name="blhc",
                        step="blhc",
                    ),
                )
                self.orchestrate_child(blhc)

        root = self.playground.create_workflow(
            task_name=pipeline_task_name, parent=parent
        )
        orchestrate_workflow(root)

        return root

    def create_blhc_workflow(
        self,
        *,
        package_build_logs: list[LookupSingle] | None = None,
        extra_task_data: dict[str, Any] | None = None,
    ) -> BlhcWorkflow:
        """
        Create and schedule a Blhc workflow.

        :param package_build_logs: build logs for the workflow. If None
          use all the expected promises by setUpTestData()
        :param extra_task_data: any extra task data to pass to the workflow
        """
        if package_build_logs is None:
            package_build_logs = []
            for package_build_log in self.package_build_log_artifacts.values():
                package_build_logs.append(package_build_log.id)

        task_data: dict[str, Any] = {
            "source_artifact": self.source_artifact.id,
            "package_build_logs": package_build_logs,
            "vendor": "debian",
            "codename": "bookworm",
        }
        if extra_task_data is not None:
            task_data.update(extra_task_data)

        example_pipeline = self.orchestrate(
            task_data=BlhcWorkflowData(**task_data),
            source_artifact=self.source_artifact,
        )
        work_request = example_pipeline.children.get(
            task_name="blhc", task_type=TaskTypes.WORKFLOW
        )
        return self.get_workflow(work_request)

    def add_qa_result(
        self,
        qa_results: Collection,
        package: str,
        version: str,
        architecture: str,
        *,
        timestamp: datetime | None = None,
    ) -> CollectionItem:
        """Add a blhc result to a ``debian:qa-results`` collection."""
        work_request = self.playground.create_worker_task(
            task_name="blhc", result=WorkRequest.Results.SUCCESS, validate=False
        )
        return qa_results.manager.add_artifact(
            self.playground.create_artifact(
                category=ArtifactCategory.BLHC,
                data=DebianBlhc(
                    source=package, version=version, architecture=architecture
                ),
            )[0],
            user=self.playground.get_default_user(),
            variables={
                "package": package,
                "version": version,
                "architecture": architecture,
                "timestamp": int((timestamp or timezone.now()).timestamp()),
                "work_request_id": work_request.id,
            },
        )

    @preserve_task_registry()
    def test_validate_input(self) -> None:
        """validate_input passes a valid case."""
        # validate_input() is called and does not raise
        self.create_blhc_workflow()

    @preserve_task_registry()
    def test_validate_input_bad_qa_suite(self) -> None:
        """validate_input raises errors in looking up a suite."""
        w = self.create_blhc_workflow(
            extra_task_data={"qa_suite": "nonexistent@debian:suite"}
        )

        with self.assertRaisesRegex(
            WorkflowValidationError,
            "'nonexistent@debian:suite' does not exist or is hidden",
        ):
            w.validate_input()

    @preserve_task_registry()
    def test_validate_input_bad_reference_qa_results(self) -> None:
        """validate_input raises errors in looking up reference QA results."""
        w = self.create_blhc_workflow(
            extra_task_data={
                "reference_qa_results": "nonexistent@debian:qa-results"
            }
        )

        with self.assertRaisesRegex(
            WorkflowValidationError,
            "'nonexistent@debian:qa-results' does not exist or is hidden",
        ):
            w.validate_input()

    def test_has_current_reference_qa_result_no_match(self) -> None:
        """_has_current_reference_qa_result: no matching result."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "other", "1.0-1", "amd64")
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-1"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.assertFalse(
            self.get_workflow(wr)._has_current_reference_qa_result(
                LookupResult(
                    result_type=CollectionItem.Types.ARTIFACT,
                    artifact=build_log,
                ),
                "amd64",
            )
        )

    def test_has_current_reference_qa_result_different_version(self) -> None:
        """_has_current_reference_qa_result: result for different version."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "hello", "1.0-1", "amd64")
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-2"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.assertFalse(
            self.get_workflow(wr)._has_current_reference_qa_result(
                LookupResult(
                    result_type=CollectionItem.Types.ARTIFACT,
                    artifact=build_log,
                ),
                "amd64",
            )
        )

    def test_has_current_reference_qa_result_unexpected(self) -> None:
        """_has_current_reference_qa_result: unexpected result."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "hello", "1.0-1", "amd64")
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-1"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        with self.assertRaisesRegex(ValueError, r"^Unexpected result"):
            self.get_workflow(wr)._has_current_reference_qa_result(
                LookupResult(
                    result_type=CollectionItem.Types.ARTIFACT,
                    artifact=self.playground.create_artifact()[0],
                ),
                "amd64",
            )

    def test_has_current_reference_qa_result_current(self) -> None:
        """_has_current_reference_qa_result: current result."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "hello", "1.0-1", "amd64")
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-1"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.assertTrue(
            self.get_workflow(wr)._has_current_reference_qa_result(
                LookupResult(
                    result_type=CollectionItem.Types.ARTIFACT,
                    artifact=build_log,
                ),
                "amd64",
            )
        )

    def test_has_current_reference_qa_result_current_promise(self) -> None:
        """_has_current_reference_qa_result: current result matching promise."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "hello", "1.0-1", "amd64")
        collection = self.playground.create_collection(
            "test", CollectionCategory.TEST
        )
        build_log_promise = self.playground.create_bare_data_item(
            collection,
            "test",
            category=BareDataCategory.PROMISE,
            data={
                "promise_work_request_id": 2,
                "promise_workflow_id": 1,
                "promise_category": ArtifactCategory.TEST,
                "source_package_name": "hello",
                "source_package_version": "1.0-1",
            },
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj(
                    [f"test@{CollectionCategory.TEST}/name:test"]
                ),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.assertTrue(
            self.get_workflow(wr)._has_current_reference_qa_result(
                LookupResult(
                    result_type=CollectionItem.Types.BARE,
                    collection_item=build_log_promise,
                ),
                "amd64",
            )
        )

    @preserve_task_registry()
    def test_populate(self) -> None:
        blhc_workflow = self.create_blhc_workflow(
            extra_task_data={
                "extra_flags": [BlhcFlags.BINDNOW, BlhcFlags.LINE_NUMBERS]
            }
        )

        # One work request per architecture
        self.assertQuerySetEqual(
            blhc_workflow.work_request.children.order_by("id").values_list(
                "task_type", "task_name", "task_data", "workflow_data_json"
            ),
            [
                (
                    TaskTypes.WORKER,
                    "blhc",
                    {
                        "environment": "debian/match:codename=bookworm",
                        "host_architecture": (
                            "amd64" if architecture == "all" else architecture
                        ),
                        "input": {"artifact": f"{artifact.id}@artifacts"},
                        "extra_flags": [
                            BlhcFlags.BINDNOW,
                            BlhcFlags.LINE_NUMBERS,
                        ],
                    },
                    {
                        "display_name": (
                            f"build log hardening check for {architecture}"
                        ),
                        "step": f"blhc-{architecture}",
                    },
                )
                for architecture, artifact in sorted(
                    self.package_build_log_artifacts.items()
                )
            ],
        )

    def test_populate_has_current_reference_qa_result(self) -> None:
        """The workflow does nothing with a current reference QA result."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.add_qa_result(sid_qa_results, "hello", "1.0-1", "amd64")
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-1"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.get_workflow(wr).populate()

        self.assertQuerySetEqual(wr.children.all(), [])

    def test_populate_previous_reference_qa_result_outdated(self) -> None:
        """The workflow produces a reference QA result if it is outdated."""
        sid_suite = self.playground.create_collection(
            "sid", CollectionCategory.SUITE
        )
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        old_qa_result = self.add_qa_result(
            sid_qa_results, "hello", "1.0-1", "amd64"
        )
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-2"
        )
        source_item = sid_suite.manager.add_artifact(
            self.playground.create_source_artifact(
                name="hello", version="1.0-2"
            ),
            user=self.playground.get_default_user(),
            variables={"component": "main", "section": "devel"},
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.get_workflow(wr).populate()

        [child] = wr.children.all()
        self.assertEqual(child.status, WorkRequest.Statuses.BLOCKED)
        self.assertEqual(child.task_type, TaskTypes.WORKER)
        self.assertEqual(child.task_name, "blhc")
        self.assertEqual(
            child.task_data,
            {
                "environment": "debian/match:codename=sid",
                "host_architecture": "amd64",
                "input": {"artifact": f"{build_log.id}@artifacts"},
                "extra_flags": [],
            },
        )
        qa_result_action = ActionUpdateCollectionWithArtifacts(
            collection=f"sid@{CollectionCategory.QA_RESULTS}",
            variables={
                "package": "hello",
                "version": "1.0-2",
                "architecture": "amd64",
                "timestamp": int(source_item.created_at.timestamp()),
                "work_request_id": child.id,
            },
            artifact_filters={"category": ArtifactCategory.BLHC},
        )
        self.assert_work_request_event_reactions(
            child,
            on_assignment=[
                ActionSkipIfLookupResultChanged(
                    lookup=(
                        f"sid@{CollectionCategory.QA_RESULTS}/"
                        f"latest:blhc_hello_amd64"
                    ),
                    collection_item_id=old_qa_result.id,
                    promise_name="reference-qa-result|blhc-amd64",
                )
            ],
            on_failure=[qa_result_action],
            on_success=[
                ActionUpdateCollectionWithArtifacts(
                    collection="internal@collections",
                    name_template=("reference-qa-result|blhc-amd64"),
                    artifact_filters={"category": ArtifactCategory.BLHC},
                ),
                qa_result_action,
            ],
        )

        # Completing the work request stores the QA result.
        self.assertTrue(child.mark_pending())
        result, _ = self.playground.create_artifact(
            category=ArtifactCategory.BLHC,
            data=DebianBlhc(
                source="hello", version="1.0-2", architecture="amd64"
            ),
            work_request=child,
        )
        self.assertTrue(child.mark_completed(WorkRequest.Results.SUCCESS))
        self.assertEqual(
            lookup_single(
                f"sid@{CollectionCategory.QA_RESULTS}/"
                f"latest:blhc_hello_amd64",
                child.workspace,
                user=child.created_by,
                expect_type=LookupChildType.ARTIFACT,
            ).artifact,
            result,
        )

    def test_populate_reference_qa_result_backs_off(self) -> None:
        """Reference tasks are skipped if another workflow got there first."""
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        sid_qa_results = self.playground.create_collection(
            "sid", CollectionCategory.QA_RESULTS
        )
        self.playground.create_debian_environment(
            codename="sid", variant="blhc"
        )
        build_log = self.playground.create_build_log_artifact(
            source="hello", version="1.0-1"
        )

        wr = self.playground.create_workflow(
            task_name="blhc",
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                package_build_logs=LookupMultiple.parse_obj([build_log.id]),
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                vendor="debian",
                codename="sid",
            ),
        )

        self.get_workflow(wr).populate()

        racing_qa_result = self.add_qa_result(
            sid_qa_results, "hello", "1.0-1", "amd64"
        )
        [child] = wr.children.all()
        self.assertTrue(child.mark_pending())

        with self.assertRaises(SkipWorkRequest):
            child.assign_worker(self.playground.create_worker())

        self.assertEqual(child.status, WorkRequest.Statuses.COMPLETED)
        self.assertEqual(child.result, WorkRequest.Results.SKIPPED)
        self.assertEqual(
            child.output_data,
            OutputData(
                skip_reason=(
                    f"Result of lookup "
                    f"'sid@{CollectionCategory.QA_RESULTS}/"
                    f"latest:blhc_hello_amd64' changed"
                )
            ),
        )
        self.assertEqual(
            lookup_single(
                f"sid@{CollectionCategory.QA_RESULTS}/"
                f"latest:blhc_hello_amd64",
                child.workspace,
                user=child.created_by,
                expect_type=LookupChildType.ARTIFACT,
            ).collection_item,
            racing_qa_result,
        )

    @preserve_task_registry()
    def test_callback_regression_analysis(self) -> None:
        self.playground.create_collection("sid", CollectionCategory.SUITE)
        self.playground.create_collection("sid", CollectionCategory.QA_RESULTS)
        architectures = ["amd64", "arm64", "i386", "ppc64el", "s390x"]

        root = self.playground.create_workflow()
        root.mark_running()
        reference = self.orchestrate(
            task_data=BlhcWorkflowData(
                prefix="reference-qa-result|",
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                update_qa_results=True,
                source_artifact=self.source_artifact.id,
                package_build_logs=LookupMultiple.parse_obj(
                    [
                        f"internal@collections/name:buildlog-{arch}"
                        for arch in architectures
                        if arch != "ppc64el"
                    ]
                ),
                vendor="debian",
                codename="sid",
            ),
            source_artifact=self.source_artifact,
            architectures=architectures,
            parent=root,
            pipeline_task_name="referencepipeline",
        )
        new = self.orchestrate(
            task_data=BlhcWorkflowData(
                reference_prefix="reference-qa-result|",
                qa_suite=f"sid@{CollectionCategory.SUITE}",
                reference_qa_results=f"sid@{CollectionCategory.QA_RESULTS}",
                enable_regression_tracking=True,
                source_artifact=self.source_artifact.id,
                package_build_logs=LookupMultiple.parse_obj(
                    [
                        f"internal@collections/name:buildlog-{arch}"
                        for arch in architectures
                    ]
                ),
                vendor="debian",
                codename="sid",
            ),
            source_artifact=self.source_artifact,
            architectures=architectures,
            parent=root,
        )

        # Unblock the blhc tasks.
        for sbuild in reference.children.get(
            task_type=TaskTypes.WORKFLOW, task_name="sbuild"
        ).children.filter(task_type=TaskTypes.WORKER, task_name="sbuild"):
            self.assertTrue(sbuild.mark_completed(WorkRequest.Results.SUCCESS))
        for sbuild in new.children.get(
            task_type=TaskTypes.WORKFLOW, task_name="sbuild"
        ).children.filter(task_type=TaskTypes.WORKER, task_name="sbuild"):
            self.assertTrue(sbuild.mark_completed(WorkRequest.Results.SUCCESS))

        reference_blhc_workflow = reference.children.get(
            task_type=TaskTypes.WORKFLOW, task_name="blhc"
        )
        reference_tasks = {
            wr.task_data["host_architecture"]: wr
            for wr in reference_blhc_workflow.children.filter(
                task_type=TaskTypes.WORKER, task_name="blhc"
            )
        }
        reference_results = {
            architecture: self.playground.create_artifact(
                category=ArtifactCategory.BLHC,
                data=DebianBlhc(
                    source="hello", version="1.0-1", architecture=architecture
                ),
                workspace=reference.workspace,
                work_request=reference_tasks[architecture],
            )[0]
            for architecture in architectures
            if architecture != "ppc64el"
        }
        new_blhc_workflow = new.children.get(
            task_type=TaskTypes.WORKFLOW, task_name="blhc"
        )
        new_tasks = {
            wr.task_data["host_architecture"]: wr
            for wr in new_blhc_workflow.children.filter(
                task_type=TaskTypes.WORKER, task_name="blhc"
            )
        }
        new_results = {
            architecture: self.playground.create_artifact(
                category=ArtifactCategory.BLHC,
                data=DebianBlhc(
                    source="hello", version="1.0-1", architecture=architecture
                ),
                workspace=new.workspace,
                work_request=new_tasks[architecture],
            )[0]
            for architecture in architectures
        }

        self.assertIsNone(new_blhc_workflow.output_data)
        self.assertEqual(schedule(), [])
        self.assertTrue(
            reference_tasks["amd64"].mark_completed(WorkRequest.Results.SUCCESS)
        )
        self.assertEqual(schedule(), [])
        self.assertTrue(
            new_tasks["amd64"].mark_completed(WorkRequest.Results.SUCCESS)
        )
        self.schedule_and_run_workflow_callback(new_blhc_workflow)

        expected_analysis: list[tuple[str, RegressionAnalysisStatus]] = [
            ("amd64", RegressionAnalysisStatus.STABLE),
        ]

        new_blhc_workflow.refresh_from_db()
        assert new_blhc_workflow.output_data is not None
        self.assertEqual(
            new_blhc_workflow.output_data.regression_analysis,
            {
                architecture: RegressionAnalysis(
                    original_url=(
                        reference_results[architecture].get_absolute_url()
                        if architecture != "ppc64el"
                        else None
                    ),
                    new_url=new_results[architecture].get_absolute_url(),
                    status=status,
                )
                for architecture, status in expected_analysis
            },
        )

        for architecture, reference_result, new_result in (
            ("arm64", WorkRequest.Results.SUCCESS, WorkRequest.Results.FAILURE),
            ("i386", WorkRequest.Results.FAILURE, WorkRequest.Results.FAILURE),
            ("ppc64el", None, WorkRequest.Results.SUCCESS),
            ("s390x", WorkRequest.Results.FAILURE, WorkRequest.Results.SUCCESS),
        ):
            if reference_result is not None:
                self.assertTrue(
                    reference_tasks[architecture].mark_completed(
                        reference_result
                    )
                )
            self.assertTrue(new_tasks[architecture].mark_completed(new_result))

        self.schedule_and_run_workflow_callback(new_blhc_workflow)

        expected_analysis += [
            ("arm64", RegressionAnalysisStatus.REGRESSION),
            ("i386", RegressionAnalysisStatus.STABLE),
            ("ppc64el", RegressionAnalysisStatus.NO_RESULT),
            ("s390x", RegressionAnalysisStatus.IMPROVEMENT),
        ]

        new_blhc_workflow.refresh_from_db()
        assert new_blhc_workflow.output_data is not None
        self.assertEqual(
            new_blhc_workflow.output_data.regression_analysis,
            {
                architecture: RegressionAnalysis(
                    original_url=(
                        reference_results[architecture].get_absolute_url()
                        if architecture != "ppc64el"
                        else None
                    ),
                    new_url=new_results[architecture].get_absolute_url(),
                    status=status,
                )
                for architecture, status in expected_analysis
            },
        )

    @preserve_task_registry()
    def test_compute_dynamic_data(self) -> None:
        package_build_log_1 = self.playground.create_build_log_artifact(
            source="hello"
        )
        package_build_log_2 = self.playground.create_build_log_artifact(
            source="hello-traditional"
        )

        workflow = self.create_blhc_workflow(
            package_build_logs=[
                f"{package_build_log_1.id}@artifacts",
                f"{package_build_log_2.id}@artifacts",
            ]
        )

        self.assertEqual(
            workflow.compute_dynamic_data(TaskDatabase(workflow.work_request)),
            BaseDynamicTaskData(subject="hello hello-traditional"),
        )
