# -*- coding: utf-8 -*-
from zope.interface import Attribute
from zope.interface import Interface


class IQuickInstallerTool(Interface):
    ''' the QuickInstaller Tool
        contains 'InstalledProduct' instances
    '''
    id = Attribute('id', 'Must be set to "portal_quickinstaller"')

    def listInstallableProducts(skipInstalled=True):
        ''' list candidate CMF products for installation -> list of dicts with
            keys:(id,hasError,status)'''

    def listInstalledProducts(showHidden=False):
        ''' returns a list of products that are installed -> list of dicts with
            keys:(id,hasError,status,,isLocked,isHidden)'''

    def isProductInstallable(productname):
        ''' is the product directory present and ready for installation '''

    def isProductAvailable(productname):
        ''' is the product directory present (to check if it has been deleted
            from the Filesystem

        Deprecated: use isProductInstallable instead.
        '''

    def installProduct(productname, locked=False, hidden=False,
                       swallowExceptions=False, forceProfile=False,
                       blacklistedSteps=None):
        ''' installs a product by name
            throws AlreadyInstalled exception, if components of the product are
            already installed

            if swallowExceptions is true, exceptions are caught and logged

            'blacklistedSteps' can be a list of step-names that will be handed
            to GenericSetup and skipped on install.
        '''

    def installProducts(products=None, stoponerror=False, REQUEST=None,
                        forceProfile=False):
        ''' installs the products specified in the products list'''

    def getProductFile(productname, fname='readme.txt'):
        ''' returns the content of a file of the product case-insensitive, if it
             does not exist -> None '''

    def getProductReadme(productname):
        ''' returns the readme file of the product case-insensitive '''

    def getProductVersion(productname):
        ''' returns the version string stored in version.txt'''

    def isProductInstalled(productname):
        ''' checks wether a product is installed (by name) '''

    def notifyInstalled(productname, locked=True, hidden=False, **kw):
        ''' marks a product that has been installed without QuickInstaller
         as installed
         if locked is set -> the prod cannot be uninstalled
         if hidden is set -> the prod is not listed in the UI
         the **kw param is passed to the constructor of InstalledProduct
         '''

    def uninstallProducts(products,
                          cascade=['types', 'skins', 'actions',
                                   'portalobjects', 'workflows', 'slots',
                                   'registrypredicates'], REQUEST=None):
        ''' removes a list of products, cascade defines which items created by
            the install shall be uninstalled '''

    def reinstallProducts(products, REQUEST=None):
        ''' reinstalls a list of products, the main difference to
            ininstall/reinstall is that it does not remove portal objects
            created  during install (e.g. tools, etc.)'''


class IInstalledProduct(Interface):
    ''' represents the installed product
        is contained inside the QuickInstallerTool '''

    id = Attribute(
        'id', 'Must be set to the same name as the product directory')

    types = Attribute('types', 'default: []')
    skins = Attribute('types', 'default: []')
    actions = Attribute('types', 'default: []')
    portalobjects = Attribute('types', 'default: []')
    workflows = Attribute('types', 'default: []')
    leftslots = Attribute('types', 'default: []')
    rightslots = Attribute('types', 'default: []')

    def __init__(id):
        ''' constructor '''

    def update(settings, installedversion='', logmsg='', status='installed',
               error=False, locked=False, hidden=False):
        ''' updates the product attributes '''

    def log(logmsg):
        ''' adds a log to the transcript '''

    def hasError():
        ''' returns if the prod is in error state '''

    def isLocked():
        ''' is the product locked for uninstall '''

    def isHidden():
        ''' is the product hidden'''

    def isInstalled():
        ''' determines if the product is in already installed '''

    def getTranscriptAsText():
        ''' return the product's install log as plain text '''

    def uninstall(cascade=['types', 'skins', 'actions', 'portalobjects',
                           'workflows', 'slots', 'registrypredicates'],
                  REQUEST=None):
        '''uninstalls the prod and removes its deps
           the parameter 'cascade' specifies what should be deleted while
           uninstalling the product

           if the Product has an uninstall() method in its Install.py it gets
           called automatically
        '''

    def getInstalledVersion():
        ''' returns the version of the prod in the moment of installation '''

    def getValue(name):
        ''' returns a value that is tracked by QuickInstaller '''
