<?php
/**
 * @package     Joomla.UnitTest
 * @subpackage  Twitter
 *
 * @copyright   (C) 2013 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

include_once __DIR__ . '/stubs/JTwitterObjectMock.php';

/**
 * Test class for JTwitterObject.
 *
 * @package     Joomla.UnitTest
 * @subpackage  Twitter
 *
 * @since       3.1.4
 */
class JTwitterObjectTest extends TestCase
{
	/**
	 * @var    JRegistry  Options for the Twitter object.
	 * @since  3.1.4
	 */
	protected $options;

	/**
	 * @var    JHttp  Mock client object.
	 * @since  3.1.4
	 */
	protected $client;

	/**
	 * @var    JTwitterObjectMock  Object under test.
	 * @since  3.1.4
	 */
	protected $object;

	/**
	 * @var    JTwitterOauth  Authentication object for the Twitter object.
	 * @since  3.1.4
	 */
	protected $oauth;

	/**
	 * @var    string  Sample JSON string.
	 * @since  3.1.4
	 */
	protected $sampleString = '{"a":1,"b":2,"c":3,"d":4,"e":5}';

	/**
	 * @var    string  Sample JSON string.
	 * @since  3.1.4
	 */
	protected $rateLimit = '{"remaining_hits":150, "reset_time":"Mon Jun 25 17:20:53 +0000 2012"}';

	/**
	 * @var    string  Sample JSON error message.
	 * @since  3.1.4
	 */
	protected $errorString = '{"errors":[{"message":"Sorry, that page does not exist","code":34}]}';

	/**
	 * Backup of the SERVER superglobal
	 *
	 * @var  array
	 * @since  3.6
	 */
	protected $backupServer;

	/**
	 * Sets up the fixture, for example, opens a network connection.
	 * This method is called before a test is executed.
	 *
	 * @access protected
	 *
	 * @return void
	 */
	protected function setUp()
	{
		$this->backupServer = $_SERVER;
		$_SERVER['HTTP_HOST'] = 'example.com';
		$_SERVER['HTTP_USER_AGENT'] = 'Mozilla/5.0';
		$_SERVER['REQUEST_URI'] = '/index.php';
		$_SERVER['SCRIPT_NAME'] = '/index.php';

		$key = "app_key";
		$secret = "app_secret";
		$my_url = "http://127.0.0.1/gsoc/joomla-platform/twitter_test.php";

		$access_token = array('key' => 'token_key', 'secret' => 'token_secret');

		$this->options = new JRegistry;
		$this->input = new JInput;
		$this->client = $this->getMockBuilder('JHttp')->setMethods(array('get', 'post', 'delete', 'put'))->getMock();
		$this->oauth = new JTwitterOAuth($this->options, $this->client, $this->input);
		$this->oauth->setToken($access_token);

		$this->object = new JTwitterObjectMock($this->options, $this->client, $this->oauth);

		$this->options->set('consumer_key', $key);
		$this->options->set('consumer_secret', $secret);
		$this->options->set('callback', $my_url);
		$this->options->set('sendheaders', true);
	}

	/**
	 * Tears down the fixture, for example, closes a network connection.
	 * This method is called after a test is executed.
	 *
	 * @access protected
	 *
	 * @return void
	 */
	protected function tearDown()
	{
		$_SERVER = $this->backupServer;
		unset($this->backupServer, $this->options, $this->input, $this->client, $this->oauth, $this->object);
	}

	/**
	 * Tests the checkRateLimit method
	 *
	 * @return void
	 *
	 * @since 3.1.4
	 * @expectedException RuntimeException
	 */
	public function testCheckRateLimit()
	{
		$resource = 'statuses';
		$action = 'show';

		$returnData = new stdClass;
		$returnData->code = 200;
		$returnData->body = '{"resources":{"statuses":{"/statuses/show":{"remaining":0, "reset":"Mon Jun 25 17:20:53 +0000 2012"}}}}';

		$path = $this->object->fetchUrl('/application/rate_limit_status.json', array('resources' => $resource));

		$this->client->expects($this->once())
		->method('get')
		->with($path)
		->will($this->returnValue($returnData));

		$this->object->checkRateLimit($resource, $action);
	}

	/**
	 * Tests the fetchUrl method
	 *
	 * @return void
	 *
	 * @since 3.1.4
	 */
	public function testFetchUrl()
	{
		// Method tested via requesting classes
		$this->markTestSkipped('This method is tested via requesting classes.');
	}

	/**
	 * Tests the getRateLimit method
	 *
	 * @return  void
	 *
	 * @since   3.1.4
	 */
	public function testGetRateLimit()
	{
		$resource = 'statuses';

		$returnData = new stdClass;
		$returnData->code = 200;
		$returnData->body = $this->sampleString;

		$path = $this->object->fetchUrl('/application/rate_limit_status.json', array('resources' => $resource));

		$this->client->expects($this->once())
		->method('get')
		->with($path)
		->will($this->returnValue($returnData));

		$this->assertThat(
			$this->object->getRateLimit($resource),
			$this->equalTo(json_decode($this->sampleString))
		);
	}

	/**
	 * Tests the getRateLimit method - failure
	 *
	 * @return  void
	 *
	 * @since   3.1.4
	 *
	 * @expectedException  DomainException
	 */
	public function testGetRateLimitFailure()
	{
		$resource = 'statuses';

		$returnData = new stdClass;
		$returnData->code = 500;
		$returnData->body = $this->errorString;

		$path = $this->object->fetchUrl('/application/rate_limit_status.json', array('resources' => $resource));

		$this->client->expects($this->once())
		->method('get')
		->with($path)
		->will($this->returnValue($returnData));

		$this->object->getRateLimit($resource);
	}

	/**
	 * Tests the getSendRequest method
	 *
	 * @return  void
	 *
	 * @since   3.1.4
	 */
	public function testSendRequest()
	{
		// Method tested via requesting classes
		$this->markTestSkipped('This method is tested via requesting classes.');
	}

	/**
	 * Tests the setOption method
	 *
	 * @return void
	 *
	 * @since 3.1.4
	 */
	public function testSetOption()
	{
		$this->object->setOption('api.url', 'https://example.com/settest');

		$this->assertThat(
			$this->options->get('api.url'),
			$this->equalTo('https://example.com/settest')
		);
	}

	/**
	 * Tests the getOption method
	 *
	 * @return  void
	 *
	 * @since   3.1.4
	 */
	public function testGetOption()
	{
		$this->options->set('api.url', 'https://example.com/settest');

		$this->assertThat(
				$this->object->getOption('api.url'),
				$this->equalTo('https://example.com/settest')
		);
	}
}
