/* -------------------------------------------------------------

   actions.cpp (part of The KDE Dictionary Client)

   Copyright (C) 2000-2001 Christian Gebauer <gebauer@kde.org>

   This file is distributed under the Artistic License.
   See LICENSE for details.

   -------------------------------------------------------------

   DictComboAction,            special TDEAction subclasses used
   DictLabelAction,            in the toolbar
   DictButtonAction

 ------------------------------------------------------------- */

#include "actions.h"

#include <tqlabel.h>
#include <tqpushbutton.h>

#include <kcombobox.h>
#include <tdetoolbar.h>


DictComboAction::DictComboAction( const TQString &text, TQObject *parent, const char *name,
                                  bool editable, bool autoSized )
  : TDEAction( text, 0, parent, name ), m_editable(editable), m_autoSized(autoSized), m_compMode(TDEGlobalSettings::completionMode())
{
}


DictComboAction::~DictComboAction()
{
}


int DictComboAction::plug( TQWidget *widget, int index )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar* bar = static_cast<TDEToolBar*>( widget );
    int id_ = TDEAction::getToolButtonID();

    m_combo = new KComboBox(m_editable,bar);
    m_combo->setCompletionMode(m_compMode);

    bar->insertWidget( id_, m_combo->sizeHint().width(), m_combo, index );
    bar->setItemAutoSized(id_,m_autoSized);

    if ( m_combo ) {
      connect(bar->getCombo(id_), TQ_SIGNAL(activated(const TQString&)), TQ_SLOT(slotComboActivated(const TQString&)));
      connect(bar->getCombo(id_), TQ_SIGNAL(activated(int)), TQ_SLOT(slotComboActivated(int)));

      if (m_editable)
        m_combo->setInsertionPolicy( TQComboBox::NoInsertion );
    }

    addContainer( bar, id_ );
    connect( bar, TQ_SIGNAL( destroyed() ), this, TQ_SLOT( slotDestroyed() ) );
    return containerCount() - 1;
  }

  return -1;
}


void DictComboAction::unplug( TQWidget *widget )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar *bar = (TDEToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( itemId( idx ) );
      removeContainer( idx );
    }

    return;
  }
}


TQWidget* DictComboAction::widget()
{
  return m_combo;
}


void DictComboAction::setFocus()
{
  if (m_combo)
   m_combo->setFocus();
}


TQString DictComboAction::currentText() const
{
  if (m_combo)
    return m_combo->currentText();
  else
    return TQString();
}

void DictComboAction::selectAll()
{
  if (m_combo)
  {
    m_combo->lineEdit()->selectAll();
    m_combo->lineEdit()->setFocus();
  }
}


void DictComboAction::setEditText(const TQString &s)
{
  if (m_combo && m_editable)
    m_combo->setEditText(s);
}


void DictComboAction::setCurrentItem(int index)
{
  if (m_combo)
    m_combo->setCurrentItem(index);
}


void DictComboAction::clearEdit()
{
  if (m_combo && m_editable)
    m_combo->clearEdit();
}


void DictComboAction::clear()
{
  if (m_combo) {
    m_combo->clear();
    if (m_editable && m_combo->completionObject())
      m_combo->completionObject()->clear();
  }
}


void DictComboAction::setList(TQStringList items)
{
  if (m_combo) {
    m_combo->clear();
    m_combo->insertStringList(items);
    if (m_editable && m_combo->completionObject())
      m_combo->completionObject()->setItems(items);
    if (!m_autoSized)
      m_combo->setFixedWidth(m_combo->sizeHint().width());
  }
}


TDEGlobalSettings::Completion DictComboAction::completionMode()
{
  if (m_combo)
    return m_combo->completionMode();
  else
    return m_compMode;
 }


void DictComboAction::setCompletionMode(TDEGlobalSettings::Completion mode)
{
  if (m_combo)
    m_combo->setCompletionMode(mode);
  else
    m_compMode = mode;
}


void DictComboAction::slotComboActivated(int i)
{
  emit(activated(i));
}


void DictComboAction::slotComboActivated(const TQString &s)
{
  emit(activated(s));
}


//*********************************************************************************


DictLabelAction::DictLabelAction( const TQString &text, TQObject *parent, const char *name )
  : TDEAction( text, 0, parent, name )
{
}


DictLabelAction::~DictLabelAction()
{
}


int DictLabelAction::plug( TQWidget *widget, int index )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar *tb = (TDEToolBar *)widget;

    int id = TDEAction::getToolButtonID();

    TQLabel *label = new TQLabel( text(), widget,  "tde toolbar widget" );
    label->setMinimumWidth(label->sizeHint().width());
    label->setBackgroundMode( TQt::PaletteButton );
    label->setAlignment(AlignCenter | AlignVCenter);
    label->adjustSize();

    tb->insertWidget( id, label->width(), label, index );

    addContainer( tb, id );

    connect( tb, TQ_SIGNAL( destroyed() ), this, TQ_SLOT( slotDestroyed() ) );

    m_label = label;

    return containerCount() - 1;
  }

  return -1;
}


void DictLabelAction::unplug( TQWidget *widget )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar *bar = (TDEToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( itemId( idx ) );
      removeContainer( idx );
    }

    return;
  }
}


void DictLabelAction::setBuddy(TQWidget *buddy)
{
  if (m_label && buddy)
    m_label->setBuddy(buddy);
}


//*********************************************************************************


DictButtonAction::DictButtonAction( const TQString& text, TQObject* receiver,
                                    const char* slot, TQObject* parent, const char* name )
  : TDEAction( text, 0, receiver, slot, parent, name )
{
}


DictButtonAction::~DictButtonAction()
{
}


int DictButtonAction::plug( TQWidget *widget, int index )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar *tb = (TDEToolBar *)widget;

    int id = TDEAction::getToolButtonID();

    TQPushButton *button = new TQPushButton( text(), widget );
    button->adjustSize();
    connect(button,TQ_SIGNAL(clicked()),this,TQ_SLOT(activate()));
    tb->insertWidget( id, button->width(), button, index );

    addContainer( tb, id );

    connect( tb, TQ_SIGNAL( destroyed() ), this, TQ_SLOT( slotDestroyed() ) );

    m_button = button;

    return containerCount() - 1;
  }

  return -1;
}


void DictButtonAction::unplug( TQWidget *widget )
{
  if ( widget->inherits( "TDEToolBar" ) )
  {
    TDEToolBar *bar = (TDEToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( itemId( idx ) );
      removeContainer( idx );
    }
  }
}


int DictButtonAction::widthHint()
{
  if (m_button)
    return m_button->sizeHint().width();
  else
    return 0;
}


void DictButtonAction::setWidth(int width)
{
  if (m_button)
    m_button->setFixedWidth(width);
}

#include "actions.moc"
