//////////////////////////////////////////////////////////////////////////////
// polyesterconfig.cc
// -------------------
// Config module for Polyester window decoration
// -------------------
// Copyright (c) 2005 Marco Martin
// -------------------
// derived from Smooth Blend
// Copyright (c) 2005 Ryan Nickell
// Please see the header file for copyright and license information.
//////////////////////////////////////////////////////////////////////////////

#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdeglobal.h>
#include <tqbuttongroup.h>
#include <tqgroupbox.h>
#include <tqradiobutton.h>
#include <tqcheckbox.h>
#include <tqspinbox.h>
#include <tqwhatsthis.h>
#include <tqcombobox.h>

#include "polyesterconfig.h"
#include "configdialog.h"

//////////////////////////////////////////////////////////////////////////////
// polyesterConfig()
// -------------
// Constructor

polyesterConfig::polyesterConfig(TDEConfig* config, TQWidget* parent)
        : TQObject(parent), config_(0), dialog_(0) {
    // create the configuration object
    config_ = new TDEConfig("twinpolyesterrc");
    TDEGlobal::locale()->insertCatalogue("polyester");

    // create and show the configuration dialog
    dialog_ = new ConfigDialog(parent);

    dialog_->show();
    // load the configuration
    load(config_);

    // setup the connections for title align
    connect(dialog_->titlealign, TQ_SIGNAL(clicked(int)),this, TQ_SLOT(selectionChanged(int)));
    // setup the connections for corner rounding
    connect(dialog_->roundCorners, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->titleBarStyle, TQ_SIGNAL(activated(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->buttonStyle, TQ_SIGNAL(activated(int)),this,TQ_SLOT(selectionChanged(int)));
    // setup title shadow
    connect(dialog_->titleshadow, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    // setup button actions
    connect(dialog_->animatebuttons, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->nomodalbuttons, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->btnComboBox, TQ_SIGNAL(activated(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->lightBorder, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    // setup the connections for spin boxes
    connect(dialog_->titlesize, TQ_SIGNAL(valueChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->buttonsize, TQ_SIGNAL(valueChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->framesize, TQ_SIGNAL(valueChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    connect(dialog_->squareButton, TQ_SIGNAL(stateChanged(int)),this,TQ_SLOT(selectionChanged(int)));
    // double click the menu 
    connect(dialog_->menuClose, TQ_SIGNAL(stateChanged(int)),this, TQ_SLOT(selectionChanged(int)));
}

//////////////////////////////////////////////////////////////////////////////
// ~polyesterConfig()
// --------------
// Destructor

polyesterConfig::~polyesterConfig() {
    if (dialog_)
    {
        delete dialog_;
    }
    if (config_)
    {
        delete config_;
    }
}

//////////////////////////////////////////////////////////////////////////////
// selectionChanged()
// ------------------
// Selection has changed

void polyesterConfig::selectionChanged(int) {

    if(dialog_->buttonsize->value() + dialog_->framesize->value() > dialog_->titlesize->value())
    {
        dialog_->buttonsize->setValue(dialog_->titlesize->value()- dialog_->framesize->value());
    }
    // setting the framesize to less than 2 will lose the top gradient and look flat
   /* if(dialog_->framesize->value() < 2)
    {
        dialog_->framesize->setValue(2);
    }*/
    emit changed();
}

//////////////////////////////////////////////////////////////////////////////
// load()
// ------
// Load configuration data

void polyesterConfig::load(TDEConfig*) {
    config_->setGroup("General");

    TQString value = config_->readEntry("TitleAlignment", "AlignHCenter");
    TQRadioButton *button = (TQRadioButton*)dialog_->titlealign->child(value.local8Bit());
    if (button)
    {
        button->setChecked(true);
    }

    dialog_->titlesize->setValue( config_->readNumEntry("TitleSize",20 ) );
    dialog_->buttonsize->setValue( config_->readNumEntry("ButtonSize",18 ) );
    dialog_->squareButton->setChecked( config_->readBoolEntry("SquareButton", false ) );
    dialog_->framesize->setValue( config_->readNumEntry("FrameSize",2 ) );

    bool cornersFlag = config_->readBoolEntry("RoundCorners", true);
    dialog_->roundCorners->setChecked( cornersFlag );
    bool titleshadow = config_->readBoolEntry("TitleShadow", true);
    dialog_->titleshadow->setChecked(titleshadow);
    bool animatebuttons = config_->readBoolEntry("AnimateButtons", true);
    dialog_->animatebuttons->setChecked(animatebuttons);
    bool lightBorder = config_->readBoolEntry("LightBorder", true);
    dialog_->lightBorder->setChecked(lightBorder);
    dialog_->titleBarStyle->setCurrentItem(config_->readNumEntry("TitleBarStyle",0));
    dialog_->buttonStyle->setCurrentItem(config_->readNumEntry("ButtonStyle",1));
    bool nomodalbuttons = config_->readBoolEntry("NoModalButtons", false);
    dialog_->nomodalbuttons->setChecked(nomodalbuttons);
    dialog_->btnComboBox->setCurrentItem(config_->readNumEntry("ButtonComboBox",0));
    bool menuClose = config_->readBoolEntry("CloseOnMenuDoubleClick");
    dialog_->menuClose->setChecked(menuClose);
}

//////////////////////////////////////////////////////////////////////////////
// save()
// ------
// Save configuration data

void polyesterConfig::save(TDEConfig*) {
    config_->setGroup("General");

    TQRadioButton *button = (TQRadioButton*)dialog_->titlealign->selected();
    if (button)
    {
        config_->writeEntry("TitleAlignment", TQString(button->name()));
    }
    config_->writeEntry("RoundCorners", dialog_->roundCorners->isChecked() );
    config_->writeEntry("TitleSize", dialog_->titlesize->value() );
    config_->writeEntry("ButtonSize", dialog_->buttonsize->value() );
    config_->writeEntry("SquareButton", dialog_->squareButton->isChecked() );
    config_->writeEntry("FrameSize", dialog_->framesize->value() );
    config_->writeEntry("TitleShadow", dialog_->titleshadow->isChecked() );
    config_->writeEntry("TitleBarStyle", dialog_->titleBarStyle->currentItem());
    config_->writeEntry("ButtonStyle", dialog_->buttonStyle->currentItem());
    config_->writeEntry("AnimateButtons", dialog_->animatebuttons->isChecked() );
    config_->writeEntry("LightBorder", dialog_->lightBorder->isChecked() );
    config_->writeEntry("NoModalButtons", dialog_->nomodalbuttons->isChecked() );
    config_->writeEntry("ButtonComboBox", dialog_->btnComboBox->currentItem());
    config_->writeEntry("CloseOnMenuDoubleClick", dialog_->menuClose->isChecked() );

    config_->sync();
}

//////////////////////////////////////////////////////////////////////////////
// defaults()
// ----------
// Set configuration defaults

void polyesterConfig::defaults() {
    TQRadioButton *button = (TQRadioButton*)dialog_->titlealign->child("AlignHCenter");
    if (button)
    {
        button->setChecked(true);
    }
    dialog_->roundCorners->setChecked( true );
    dialog_->titlesize->setValue( 20 );
    dialog_->squareButton->setChecked( false );
    dialog_->buttonsize->setValue( 18 );
    dialog_->framesize->setValue( 2 );
    dialog_->titleBarStyle->setCurrentItem( 0 );
    dialog_->buttonStyle->setCurrentItem( 1 );
    dialog_->titleshadow->setChecked( true );
    dialog_->animatebuttons->setChecked( true );
    dialog_->nomodalbuttons->setChecked( true );
    dialog_->lightBorder->setChecked( true );
    dialog_->btnComboBox->setCurrentItem( 0 );
    dialog_->menuClose->setChecked( false );
}

//////////////////////////////////////////////////////////////////////////////
// Plugin Stuff                                                             //
//////////////////////////////////////////////////////////////////////////////

extern "C" {
    TDE_EXPORT TQObject* allocate_config(TDEConfig* config, TQWidget* parent) {
        return (new polyesterConfig(config, parent));
    }
}

#include "polyesterconfig.moc"
