/*********
*
* This file is part of BibleTime's source code, http://www.bibletime.info/.
*
* Copyright 1999-2006 by the BibleTime developers.
* The BibleTime source code is licensed under the GNU General Public License version 2.0.
*
**********/



#ifndef CSWORDKEY_H
#define CSWORDKEY_H

//Own includes


//TQt includes
#include <tqstring.h>
//forward declarations

class CSwordModuleInfo;

/** Base class for all keys.
 * The base class for all Sword based keys.
 * @author The BibleTime team
 * @version $Id: cswordkey.h,v 1.28 2007/10/26 18:26:23 jerickson314 Exp $
 */

class CSwordKey {

protected:
	/** Constructor. May not be called because this class contains pure virtual methods.
	* @param module The module which belongs to this key, may be NULL
	*/
	CSwordKey(CSwordModuleInfo* const module = 0); //protected constructor, because CSwordKey shouldn't be used (it's an abstract base class).
	/** Copy constructor.
	*/
	CSwordKey(const CSwordKey&); //copy constructor

public:
	enum TextRenderType {
		Normal = 0,
		HTMLEscaped
	};
	/** Destructor.
	* Public, not protected like the constructor, because CSwordKey pointers may be deleted by all others.
	*/
	virtual ~CSwordKey() {}

	;

	//pure virtual functions
	/** Returns the current key.
	 * @return The current key which belongs to the current object.
	 */
	virtual const TQString key() const = 0;
	/** Sets the current key. Sets the key using a utf8 enabled TQString.
	 * @param key The key which should be used to set the current one
	 */
	virtual const bool key(const TQString& key) = 0;
	/** Set the key using a utf8-decoded c-string
	 * @param key The key which should be used to set the current one
	 */
	virtual const bool key(const char* key) = 0;
	/** Clone this object. Clone this current object and return it.
	 * @return A clone of the current object.
	 */
	virtual CSwordKey* copy() const = 0;

	//implemented functions
	/** Set/get the module. Set and get the module which belongs to this key.
	 * @return The module which belongs to this key.
	 */
	inline virtual CSwordModuleInfo* const module(CSwordModuleInfo* const newModule = 0);
	/** Returns the raw, unchanged text. Returns the text without any filter modifications,
	 * just in the way it comes out of the module.
	 */
	virtual const TQString rawText();
	/** Returns the rendered text. Returns the text of the current key after passign it through the
	 * modules filters.
	 */
	virtual const TQString renderedText( const CSwordKey::TextRenderType mode = CSwordKey::Normal );
	/** Stripped down text. Returns the text after removing all markup tags from it.
	 */
	virtual const TQString strippedText();
	/**
	 * This returns a new object of the right CSwordKey* implementation
	 * (e.g. CSwordVerseKey or CSwordLDKey)
	 * The type is determined by the type of the module.
	 * @see CSwordModuleInfo, CSwordBibleModuleInfo, CSwordCommentaryModuleInfo, CSwordLexiconModukleInfo
	 */
	static CSwordKey* createInstance(CSwordModuleInfo * const module);
	/**
	 * The assignment operator for more easy use of the key classes.
	 */
	inline virtual CSwordKey& operator = ( const TQString& );

protected:
	/** Returns the current key in raw format, for Sword.
	 * @return The current key which belongs to the current object.
	 */
	virtual const char* rawKey() const = 0;
	CSwordModuleInfo* m_module; //module pointer used by all keys
};


/** The assignment operator for more easy use of the key classes. */
inline CSwordKey& CSwordKey::operator = ( const TQString& newKey ) {
	key(newKey);
	return *this;
}

inline CSwordModuleInfo* const CSwordKey::module(CSwordModuleInfo* const newModule) {
	if (newModule) {
		m_module = newModule;
	}
	return m_module;
}

#endif
