/***************************************************************************
 *   Copyright (C) 2005-2007 Nicolas Hadacek <hadacek@kde.org>             *
 *   Copyright (C) 2003-2005 Alain Gibaud <alain.gibaud@free.fr>           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "hardware_config.h"

//-----------------------------------------------------------------------------
bool Hardware::Data::isEqual(const Data &data) const
{
  return ( data.portType==portType );
}

void Hardware::Data::readConfig(GenericConfig &config)
{
  portType = config.readEnumEntry<PortType>("port_type", PortType::Serial);
}

void Hardware::Data::writeConfig(GenericConfig &config) const
{
  config.writeEnumEntry<PortType>("port_type", portType);
}

//-----------------------------------------------------------------------------
void Hardware::Config::writeCurrentHardware(PortType type, const TQString &name)
{
  writeEntry(TQString("current_hardware_") + type.key(), name);
}

TQString Hardware::Config::currentHardware(PortType type)
{
  TQStringList names = hardwareNames(type);
  return readEntry(TQString("current_hardware_") + type.key(), names[0]);
}

TQString Hardware::Config::label(const TQString &name) const
{
  const DataInfo *info = standardHardwareDataInfo(name);
  if ( info==0 ) return TQString();
  return i18n(info->label);
}

TQString Hardware::Config::comment(const TQString &name) const
{
  const DataInfo *info = standardHardwareDataInfo(name);
  if ( info==0 || info->comment==0 ) return TQString();
  return i18n(info->comment);
}

void Hardware::Config::writeCustomHardware(const TQString& name, const Hardware::Data &hdata)
{
  Q_ASSERT( !isStandardHardware(name) );
  TQStringList customNames = readListEntry("custom_hardware_names", TQStringList());
  if ( !customNames.contains(name) ) {
    customNames += name;
    writeEntry("custom_hardware_names", customNames);
  }
  GenericConfig config(group() + "_custom_hardware_" + name);
  hdata.writeConfig(config);
}

void Hardware::Config::deleteCustomHardware(const TQString &name)
{
  Q_ASSERT( !isStandardHardware(name) );
  TQStringList customNames = readListEntry("custom_hardware_names", TQStringList());
  customNames.remove(name);
  writeEntry("custom_hardware_names", customNames);
  GenericConfig::deleteGroup(group() + "_custom_hardware_" + name);
}

Hardware::Data *Hardware::Config::hardwareData(const TQString &name) const
{
  if ( isStandardHardware(name) ) return standardHardwareData(name);
  Hardware::Data *hdata = createHardwareData();
  hdata->name = name;
  GenericConfig config(group() + "_custom_hardware_" + name);
  hdata->readConfig(config);
  return hdata;
}

TQStringList Hardware::Config::hardwareNames(PortType type)
{
  TQStringList names = standardHardwareNames(type);
  TQStringList customNames = readListEntry("custom_hardware_names", TQStringList());
  for (uint i=0; i<uint(customNames.count()); i++) {
    if ( type!=PortType::Nb_Types ) {
      Hardware::Data *hdata = hardwareData(customNames[i]);
      if ( hdata->portType==type ) names += customNames[i];
      delete hdata;
    } else names += customNames[i];
  }
  return names;
}
