/*******************************************************************************
  XDG desktop portal implementation for TDE
  Copyright © 2024 Mavridis Philippe <mavridisf@gmail.com>

  Partially based on code from kdbusnotification
  Copyright © 2021 Emanoil Kotsev <deloptes@gmail.com>

  This program or library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the License,
  or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
  details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

  Improvements and feedback are welcome!
*******************************************************************************/

// TQt
#include <tqtimer.h>

// TDE
#include <kdebug.h>

// Portal
#include "portal_daemon.h"
#include "portal_daemon.moc"
#include "util.h"

PortalDaemon::PortalDaemon()
: KUniqueApplication(),
  m_retryCount(0),
  d_root(nullptr),
  d_org(nullptr),
  d_freedesktop(nullptr),
  d_portal(nullptr),
  d_desktop(nullptr)
{
    connectDBus();
}

PortalDaemon::~PortalDaemon()
{
    disconnectDBus();
}

void PortalDaemon::connectDBus()
{
    if (d_desktop) disconnectDBus();

    m_connection = TQT_DBusConnection::addConnection(
        TQT_DBusConnection::SessionBus,
        DBUS_PORTAL_SERVICE
    );

    if (m_connection.isConnected())
    {
        m_connection.connect(this, TQ_SLOT(slotDBusSignal(const TQT_DBusMessage&)));
        if (m_connection.requestName(DBUS_PORTAL_SERVICE, TQT_DBusConnection::NoReplace))
        {
            m_connection.scheduleDispatch();
            kdDebug() << "TDE portal service successfully registered." << endl;
            return;
        }
    }

    ++m_retryCount;
    kdWarning() << "Failed to connect to DBus, retrying in "
                << DBUS_RETRY_TIMEOUT << " secs "
                << "(" << m_retryCount << "/" << DBUS_RETRY_COUNT << "tries)"
                << endl;

    if (m_retryCount < DBUS_RETRY_COUNT)
    {
        TQTimer::singleShot(DBUS_RETRY_TIMEOUT, this, TQ_SLOT(connectDBus()));
    }
}

void PortalDaemon::disconnectDBus()
{
    DEL(d_root)
    DEL(d_org)
    DEL(d_freedesktop)
    DEL(d_portal)
    DEL(d_desktop)

    if (m_connection.isConnected())
    {
        m_connection.disconnect(this, TQ_SLOT(slotDBusSignal(const TQT_DBusMessage&)));
        m_connection.closeConnection(DBUS_PORTAL_SERVICE);
    }

    m_retryCount = 0;
}

void PortalDaemon::slotDBusSignal(const TQT_DBusMessage &message)
{
    TQString serviceName = message[0].toString();
    if (message.interface() == TQString("org.freedesktop.DBus") &&
        message.member() == TQString("NameAcquired") &&
        serviceName == DBUS_PORTAL_SERVICE)
    {
        kdDebug() << "Portal daemon acquired unique DBus name: "
                  << serviceName << endl;

        d_root        = new RootNodeService(m_connection);
        d_org         = new OrgNodeService(m_connection);
        d_freedesktop = new FreeDesktopNodeService(m_connection);
        d_portal      = new PortalNodeService(m_connection);
        d_desktop     = new DesktopNodeService(m_connection);
    }
}

// kate: replace-tabs true; tab-width 4; indent-width 4;