/***************************************************************************
    smb4k_mv  -  This is the move utility of Smb4K
                             -------------------
    begin                : Sa Nov 19 2005
    copyright            : (C) 2005-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <locale.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <iostream>
using namespace std;

#define SMB4K_MV_VERSION "0.4"

void info()
{
	cout << "This is smb4k_mv (version " << SMB4K_MV_VERSION << "), the move utility of Smb4K" << endl;
	cout << "Copyright (C) 2005-2007, Alexander Reinholdt" << endl;
	cout << endl;
	cout << "Usage:" << endl;
	cout << "  smb4k_mv {user}:{group} {perms} {src} {dest}" << endl;
	cout << "  smb4k_mv --help" << endl;
	cout << "  smb4k_mv --version" << endl;
	cout << endl;
	cout << "Arguments:" << endl;
	cout << "  {user}\tThe (new) owner of the file" << endl;
	cout << endl;
	cout << "  {group}\tThe (new) group of the file" << endl;
	cout << endl;
	cout << "  {perms}\tThe new permissions of the file" << endl;
	cout << endl;
	cout << "  {src}\t\tThe (full) path of the source file" << endl;
	cout << endl;
	cout << "  {dest}\tThe destination where the file will be moved to" << endl;
	cout << endl;
	cout << "  --help\tDisplay this help screen and exit." << endl;
	cout << "  --version\tDisplay the version information and exit." << endl;
	cout << endl;
}


void version()
{
	cout << "Version: " << SMB4K_MV_VERSION << endl;
}


bool find_program( const char *name, char *path )
{
	const char *paths[] = { "/bin/", "/sbin/", "/usr/bin/", "/usr/sbin/", "/usr/local/bin/", "/usr/local/sbin/" };
	string file = "";

	for ( uint i = 0; i < sizeof( paths ) / sizeof( char * ); i++ )
	{
		string p( paths[i] );
		p.append( name );

		if ( access( p.c_str(), X_OK ) == 0 )
		{
			file.assign( p );
			break;
		}
	}

	if ( !strcmp( file.c_str(), "" ) )
	{
		cerr << "smb4k_mv: Could not find " << name << " binary" << endl;

		return false;
	}

	int len = strlen( file.c_str() ) + 1;
	(void) strncpy( path, file.c_str(), len );
	path[len-1] = '\0';

	return true;
}


int main( int argc, char *argv[], char *envp[] )
{
	(void) setlocale( LC_ALL, "" );

	if ( argc < 2 )
	{
		info();
		exit( EXIT_FAILURE );
	}

	char *user_group = NULL;
	char *permissions = NULL;
	char *source = NULL;
	char *destination = NULL;

	int c;

	while ( 1 )
	{
		int option_index = 0;

		static struct option long_options[] =
		{
			{ "help", 0, 0, 0 },
			{ "version", 0, 0, 0 },
			{ 0, 0, 0, 0 }
		};

		c = getopt_long( argc, argv, "", long_options, &option_index );

		if ( c == -1 )
		{
			break;
		}

		switch ( c )
		{
			case 0:
			{
				// Get the length of the option string, create a
				// new string and copy the option into it:
				int len = strlen( long_options[option_index].name ) + 1;
				char opt[len];
				opt[0] = '\0';
				(void) strncpy( opt, long_options[option_index].name, len );
				opt[len-1] = '\0';

				// Now check which option has been provided:
				if ( !strncmp( opt, "help", len ) )
				{
					info();

					// That's it. Exit here.
					exit( EXIT_SUCCESS );
				}
				else if ( !strncmp( opt, "version", len ) )
				{
					version();

					// That's it. Exit here.
					exit( EXIT_SUCCESS );
				}
				else
				{
					break;
				}

				break;
			}
			case '?':
			{
				// Abort the program if an unknown option
				// is encountered:
				exit( EXIT_FAILURE );
			}
			default:
			{
				break;
			}
		}
	}

	if ( optind < argc )
	{
		while ( optind < argc )
		{
			if ( optind == 1 )
			{
				if ( strchr( argv[optind], ':' ) )
				{
					int len = strlen( argv[optind] ) + 1;
					user_group = new char[len];
					user_group[0] = '\0';
					(void) strncpy( user_group, argv[optind], len );
					user_group[len-1] = '\0';
				}
				else
				{
					cerr << "smb4k_mv: First argument must contain a colon" << endl;
					exit( EXIT_FAILURE );
				}
			}
			else if ( optind == 2 )
			{
				int i = 0;

				while ( argv[optind][i] )
				{
					if ( !isdigit( argv[optind][i] ) )
					{
						cerr << "smb4k_mv: Second argument must only contain digits" << endl;
						exit( EXIT_FAILURE );
					}

					i++;
				}

				int len = strlen( argv[optind] ) + 1;
				permissions = new char[len];
				permissions[0] = '\0';
				(void) strncpy( permissions, argv[optind], len );
				permissions[len-1] = '\0';
			}
			else if ( optind == 3 )
			{
				// We only want regular files to be moved:
				struct stat file_stat;

				if ( lstat( argv[optind], &file_stat ) == -1 )
				{
					int err = errno;
					cerr << "smb4k_mv: " << strerror( err ) << endl;
					exit( EXIT_FAILURE );
				}

				if ( !S_ISREG( file_stat.st_mode ) ||
				     S_ISFIFO( file_stat.st_mode ) ||
				     S_ISLNK( file_stat.st_mode ) )
				{
					cerr << "smb4k_mv: '" << argv[optind] << "' is not a regular file" << endl;
					exit( EXIT_FAILURE );
				}

				int len = strlen( argv[optind] ) + 1;
				source = new char[len];
				source[0] = '\0';
				(void) strncpy( source, argv[optind], len );
				source[len-1] = '\0';
			}
			else if ( optind == 4 )
			{
				// Be sure that the destination either does not
				// exist or that it is a regular file:
				struct stat file_stat;
				bool file_exists = true;

				if ( lstat( argv[optind], &file_stat ) == -1 )
				{
					int err = errno;

					// If the file does not exist, that's very good
					// for our purposes. Do not error out in that case.
					if ( err != ENOENT )
					{
						cerr << "smb4k_mv: " << strerror( err ) << endl;
						exit( EXIT_FAILURE );
					}
					else
					{
						file_exists = false;
					}
				}

				if ( file_exists )
				{
					if ( !S_ISREG( file_stat.st_mode ) ||
					     S_ISFIFO( file_stat.st_mode ) ||
					     S_ISLNK( file_stat.st_mode ) )
					{
						cerr << "smb4k_mv: '" << argv[optind] << "' is not a regular file" << endl;
						exit( EXIT_FAILURE );
					}
				}

				int len = strlen( argv[optind] ) + 1;
				destination = new char[len];
				destination[0] = '\0';
				(void) strncpy( destination, argv[optind], len );
				destination[len-1] = '\0';
			}
			else
			{
				cerr << "smb4k_mv: There are too many arguments" << endl;
				exit( EXIT_FAILURE );
			}

			optind++;
		}
	}

	int i;

	// Change owner and group:
	char chown_prog[255];
	chown_prog[0] = '\0';

	if ( !find_program( "chown", chown_prog ) )
	{
		exit( EXIT_FAILURE );
	}

	char *chown_argv[] = { chown_prog, user_group, source, NULL };

	if ( fork() == 0 )
	{
		if ( (i = execve( chown_argv[0], chown_argv, envp )) == -1 )
		{
			int err = errno;
			cerr << "smb4k_mv: " << strerror( err ) << endl;
			exit( EXIT_FAILURE );
		}
	}
	else
	{
		wait( &i );
	}

	// Apply new permissions:
	char chmod_prog[255];
	chmod_prog[0] = '\0';

	if ( !find_program( "chmod", chmod_prog ) )
	{
		exit( EXIT_FAILURE );
	}

	char *chmod_argv[] = { chmod_prog, permissions, source, NULL };

	if ( fork() == 0 )
	{
		if ( (i = execve( chmod_argv[0], chmod_argv, envp )) == -1 )
		{
			int err = errno;
			cerr << "smb4k_mv: " << strerror( err ) << endl;
			exit( EXIT_FAILURE );
		}
	}
	else
	{
		wait( &i );
	}

	// Now, finally, move the file:
	char mv_prog[255];
	mv_prog[0] = '\0';

	if ( !find_program( "mv", mv_prog ) )
	{
		exit( EXIT_FAILURE );
	}

	char *mv_argv[] = { mv_prog, source, destination, NULL };

	if ( fork() == 0 )
	{
		if ( (i = execve( mv_argv[0], mv_argv, envp )) == -1 )
		{
			int err = errno;
			cerr << "smb4k_mv: " << strerror( err ) << endl;
			exit( EXIT_FAILURE );
		}
	}
	else
	{
		wait( &i );
	}

	return EXIT_SUCCESS;
}

