/***************************************************************************
 *   Copyright (C) 2006 by Robert Hogan                                    *
 *   robert@roberthogan.net                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.              *
 ***************************************************************************/

#include "appinfo.h"
#include "appimpl.h"

#include <stdlib.h>

#include <tqmap.h>

#include <kdebug.h>
#include <kstandarddirs.h>
#include <kdesktopfile.h>
#include <tdelocale.h>
#include <kservice.h>
#include <tdeio/job.h>

#include "config.h"

AppImpl::AppImpl(TDEIO_AppInfo *slave) : TQObject(), m_slave(slave)
{
}

void AppImpl::listRoot()
{
	kdDebug(7129) << "AppImpl::listRoot" << endl;

	// Top level entry
	TDEIO::UDSEntry entry;
	createTopLevelEntry(entry);
	m_slave->listEntry(entry, false);

	TQStringList dirList = TQStringList::split(":", getenv("PATH"));
	kdDebug(7129) << dirList << endl;

	TQMap<TQString, bool> knownApps;
	TQValueList<TDEIO::UDSEntry> list;
	for (const TQString &dirname : dirList)
	{
		TQDir dir(dirname);
		if (!dir.exists())
		{
			continue;
		}

		TQStringList filenames = dir.entryList(TQDir::Files | TQDir::Readable);
		TDEIO::UDSEntry entry;
		for (const TQString &filename : filenames)
		{
			TQString fullname = dirname + filename;
			if (!knownApps.contains(fullname))
			{
				knownApps[fullname] = true;
				createEntry(entry, filename);
				list.append(entry);
				if (list.count() >= 50)
				{
					m_slave->listEntries(list);
					list.clear();
				}
			}
		}
	}
	if (list.count() > 0)
	{
		m_slave->listEntries(list);
		list.clear();
	}
  m_slave->finished();
}

bool AppImpl::parseURL(const KURL &url, TQString &name, TQString &path) const
{
  TQString url_path = url.path();
  int i = url_path.find('/', 1);
  if (i > 0)
  {
    name = url_path.mid(1, i-1);
    path = url_path.mid(i+1);
  }
  else
  {
    name = url_path.mid(1);
    path = TQString::null;
  }

  return !name.isEmpty();
}

bool AppImpl::statByName(const TQString &filename, TDEIO::UDSEntry &entry)
{
  kdDebug(7129) << "AppImpl::statByName" << endl;

	TQStringList dirList = TQStringList::split(":", getenv("PATH"));

  TQStringList names_found;
  for (const TQString &dirname : dirList)
  {
    TQDir dir(dirname);
    if (!dir.exists())
    {
      continue;
    }

    TQStringList filenames = dir.entryList(TQDir::Files | TQDir::Readable);
    for (const TQString &fname : filenames)
    {
      if (fname == filename)
      {
        createEntry(entry, fname);
        return true;
      }
    }
  }

  return false;
}

static void addAtom(TDEIO::UDSEntry &entry, unsigned int ID, long l,
                    const TQString &s = TQString::null)
{
  TDEIO::UDSAtom atom;
  atom.m_uds = ID;
  atom.m_long = l;
  atom.m_str = s;
  entry.append(atom);
}

void AppImpl::createTopLevelEntry(TDEIO::UDSEntry &entry) const
{
  entry.clear();
  addAtom(entry, TDEIO::UDS_NAME, 0, ".");
  addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
  addAtom(entry, TDEIO::UDS_ACCESS, 0555);
  addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/system_directory");
  addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "system");
}

void AppImpl::createEntry(TDEIO::UDSEntry &entry, const TQString &file)
{
  entry.clear();

  addAtom(entry, TDEIO::UDS_NAME, 0, file);
  addAtom(entry, TDEIO::UDS_URL, 0, "appinfo:/" + file);
  addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
  addAtom(entry, TDEIO::UDS_ACCESS, 0555);
  addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");

  KService::Ptr service = KService::serviceByDesktopName(file);
  if (service && service->isValid())
  {
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, service->icon());
  }
  else
  {
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "binary");
  }
}

void AppImpl::createExeEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname,
		    const TQStringList &fullname)
{
  TDEIO::UDSEntry entry;
  for (const TQString &name : fullname)
  {
    if (name.isEmpty())
    {
      continue;
    }

    entry.clear();
    KService::Ptr service = KService::serviceByDesktopName(shortname);
    if (service && service->isValid())
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Application (%1)").arg(service->name()));
      addAtom(entry, TDEIO::UDS_URL, 0, locate("apps", service->desktopEntryPath()));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "application/x-desktop");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, service->icon());
      list.append(entry);
      return;
    }

    addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Application (%1)").arg(name));
    addAtom(entry, TDEIO::UDS_URL, 0, name);
    addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
    addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "application");
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "binary");
    list.append(entry);
  }
}

void AppImpl::createManPageEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TDEIO::UDSEntry entry;
  addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Manual for %1").arg(shortname));
  addAtom(entry, TDEIO::UDS_URL, 0, "man:/"+shortname);
  addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
  addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "application/x-desktop");
  addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "application-x-troff-man");
  addAtom(entry, TDEIO::UDS_SIZE, 0, "");
  list.append(entry);
}

void AppImpl::createHomeDirEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList homedir;
  homedir << TQString("%1/").arg(getenv("HOME"));
  TQString dirname = "." + shortname;
  TQStringList fullname = getFullLocation(homedir, dirname,
      TQDir::FilterSpec(TQDir::Hidden | TQDir::Dirs | TQDir::Readable), true, false);

  TDEIO::UDSEntry entry;
  for (const TQString &fname : fullname)
  {
    if (fname.isEmpty())
    {
      continue;
    }

    entry.clear();
    addAtom(entry, TDEIO::UDS_NAME, 0, i18n("User Data (%1)").arg(fname));
    addAtom(entry, TDEIO::UDS_URL, 0, fname);
    addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
    addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    list.append(entry);
  }
}

void AppImpl::createTDEDataDirEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList dirList = TDEGlobal::instance()->dirs()->resourceDirs("data");
  TQStringList TDEDataDir = getFullLocation(dirList, shortname,
      TQDir::FilterSpec(TQDir::Dirs | TQDir::Readable), false, false);
  if (TDEDataDir.isEmpty())
  {
    return;
  }

	TDEIO::UDSEntry entry;
  for (const TQString &dirname : TDEDataDir)
  {
    if (dirname.isEmpty())
    {
      continue;
    }

    entry.clear();
    addAtom(entry, TDEIO::UDS_NAME, 0, i18n("App Data (%1)").arg(dirname));
    addAtom(entry, TDEIO::UDS_URL, 0, dirname);
    addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
    addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    list.append(entry);
  }
}

void AppImpl::createStandardDataDirEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList dirList;
  dirList << "/usr/share/";
  dirList << "/usr/local/share/";
  TQStringList StandardDataDir = getFullLocation(dirList, shortname,
      TQDir::FilterSpec(TQDir::Files | TQDir::Dirs | TQDir::Readable), true, false);
  if (StandardDataDir.isEmpty())
  {
    return;
  }

  TDEIO::UDSEntry entry;
  for (const TQString &dirname : StandardDataDir)
  {
    if (dirname.isEmpty())
    {
      continue;
    }

    entry.clear();
    addAtom(entry, TDEIO::UDS_NAME, 0, i18n("App Data (%1)").arg(dirname));
    addAtom(entry, TDEIO::UDS_URL, 0, dirname);
    addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
    addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
    addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    list.append(entry);
  }
}

void AppImpl::createTDEConfigEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
	TDEIO::UDSEntry entry;

	// Global TDE config
	TQString tdeCfgDir(TDESYSCONFDIR);

	if (!tdeCfgDir.isEmpty())
	{
		TQStringList dirList;
		dirList << tdeCfgDir;
		TQStringList TDEConfigFiles = getFullLocation(dirList, shortname + "rc",
				TQDir::FilterSpec(TQDir::Files | TQDir::Dirs | TQDir::Readable), false, true);

		for (const TQString &filename : TDEConfigFiles)
		{
			if (!filename.isEmpty())
			{
				entry.clear();
				addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Config File (%1)").arg(filename));
				addAtom(entry, TDEIO::UDS_URL, 0, filename);
				addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
				addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "text/plain");
				addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "configure");
				list.append(entry);
			}
		}
	}

	// Resource file
  TQString TDEDataDir = locate("config", shortname + "rc");
  kdDebug(7129) << "TDEDataDir: " << TDEDataDir << endl;
  if (TDEDataDir.isEmpty())
  {
    return;
  }
  entry.clear();
  addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Config File (%1)").arg(TDEDataDir));
  addAtom(entry, TDEIO::UDS_URL, 0, TDEDataDir);
  addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
  addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "text/plain");
  addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "configure");
  list.append(entry);
}

void AppImpl::createStandardConfigEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList dirList;
  dirList << "/etc/";
  dirList << "/usr/etc/";
  dirList << "/usr/local/etc/";
  TQStringList StandardConfigFile = getFullLocation(dirList,shortname,
        TQDir::FilterSpec(TQDir::Files | TQDir::Dirs | TQDir::Readable), true, false);
  if (StandardConfigFile.isEmpty())
  {
    return;
  }

  TDEIO::UDSEntry entry;
  for (const TQString &fname : StandardConfigFile)
  {
    if (fname.isEmpty())
    {
      continue;
    }

    bool isFolder = false;
    TQFileInfo fi(fname);
    if (fi.isDir())
    {
      isFolder = true;
    }

    kdDebug(7129) << "createStandardConfigEntry: " << fname << endl;

    entry.clear();
    addAtom(entry, TDEIO::UDS_URL, 0, fname);
    TQString icon;
    if (isFolder)
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Config Data(%1)").arg(fname));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    }
    else
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Config File(%1)").arg(fname));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "text/plain");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "configure");
    }
    list.append(entry);
  }
}

void AppImpl::createTmpDirEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList dirList;
  dirList << "/tmp/";
  TQStringList TmpDir = getFullLocation(dirList, shortname,
        TQDir::FilterSpec(TQDir::Files | TQDir::Dirs | TQDir::Readable), true, true);
  if (TmpDir.isEmpty())
  {
    return;
  }

  for (const TQString &fname : TmpDir)
  {
    if (fname.isEmpty())
    {
      continue;
    }

    bool isFolder = false;
    TQFileInfo fi(fname);
    if (fi.isDir())
    {
      isFolder = true;
    }

    kdDebug(7129) << "createTmpDirEntry: " << fname << endl;

    TDEIO::UDSEntry entry;
    addAtom(entry, TDEIO::UDS_URL, 0, fname);
    if (isFolder)
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Tmp Data (%1)").arg(fname));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    }
    else
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("Tmp File (%1)").arg(fname));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "text/plain");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "eraser");
    }
    list.append(entry);
  }
}

void AppImpl::createXDGDirEntry(TQValueList<TDEIO::UDSEntry> &list, const TQString &shortname)
{
  TQStringList dirList;
  dirList << TQString("%1/.config/").arg(getenv("HOME"));
  TQStringList xdgDir = getFullLocation(dirList, shortname,
        TQDir::FilterSpec(TQDir::Files | TQDir::Dirs | TQDir::Readable), false, true);

  for (const TQString &filename : xdgDir)
  {
    if (filename.isEmpty())
    {
      continue;
    }

    bool isFolder = false;
    TQFileInfo fi(filename);
    if (fi.isDir())
    {
      isFolder = true;
    }

    kdDebug(7129) << "createXdgDirEntry: " << filename << endl;

    TDEIO::UDSEntry entry;
    addAtom(entry, TDEIO::UDS_URL, 0, filename);
    if (isFolder)
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("XDG Config Data (%1)").arg(filename));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "inode/directory");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "folder");
    }
    else
    {
      addAtom(entry, TDEIO::UDS_NAME, 0, i18n("XDG Cconfig File (%1)").arg(filename));
      addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
      addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, "text/plain");
      addAtom(entry, TDEIO::UDS_ICON_NAME, 0, "eraser");
    }
    list.append(entry);
  }
}

bool AppImpl::listAppContents(const TQString &name, TQValueList<TDEIO::UDSEntry> &list)
{
  kdDebug(7129) << "AppImpl::listAppContents" << endl;

  // Create entry for binary file
  createExeEntry(list, name, getAppAddress(name));

  // Create entry for standard config and data folders
  createStandardConfigEntry(list, name);
  createStandardDataDirEntry(list, name);

  // Create entry for TDE config and data folders
  createTDEConfigEntry(list, name);
  createTDEDataDirEntry(list, name);

  // Create entry for data folder in home dir
  createHomeDirEntry(list, name);

  //Create entry for app XDF config folder in home dir
  createXDGDirEntry(list, name);

  // Create entry for manual
  createManPageEntry(list, name);

  //Create entry for folders in tmp dir
  createTmpDirEntry(list, name);

  return true;
}

TQStringList AppImpl::getAppAddress(const TQString &name)
{
	TQStringList dirList = TQStringList::split(":", getenv("PATH"));
  return getFullLocation(dirList, name,
          TQDir::FilterSpec(TQDir::Files | TQDir::Readable), false, false);

}
TQStringList AppImpl::getFullLocation(const TQStringList &dirList, const TQString &name,
                                 const TQDir::FilterSpec &filter, bool beginswith, bool recursive)
{
	TQMap<TQString, bool> knownApps;
  TQStringList finds;
  for (const TQString &dirname : dirList)
  {
    TQDir dir = dirname;
    if (!dir.exists())
    {
      continue;
    }

    TQStringList filenames = dir.entryList(filter);
    for (const TQString &filename : filenames)
    {
      if (recursive && filename != ".." && filename != ".")
      {
        TQFileInfo fi(dirname + filename);
        if (fi.isDir())
        {
          TQStringList recurfinds;
          recurfinds = getFullLocation(TQStringList(dirname + filename + "/"),
                  name, filter, beginswith, recursive);
          if (!recurfinds.isEmpty())
          {
            for (const TQString &recurFullname : recurfinds)
            {
							if (!knownApps.contains(recurFullname))
							{
								knownApps[recurFullname] = true;
								finds << recurFullname;
							}
            }
          }
        }
      }

      if (name == filename || (beginswith && filename.startsWith(name)))
      {
				TQString fullname = dirname + filename;
				if (!knownApps.contains(fullname))
				{
					knownApps[fullname] = true;
					finds << fullname;
				}
      }
    }
  }
  return finds;
}

#include "appimpl.moc"
