/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2006-04-19
 * Description : A tab to display general image information
 *
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqlayout.h>
#include <tqstyle.h>
#include <tqfile.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <tqfileinfo.h>
#include <tqwhatsthis.h>
#include <tqframe.h>
#include <tqscrollview.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialogbase.h>
#include <tdefileitem.h>
#include <ksqueezedtextlabel.h>
#include <kseparator.h>

// LibKDcraw includes.

#include <libkdcraw/version.h>
#include <libkdcraw/kdcraw.h>

#if KDCRAW_VERSION < 0x000106
#include <libkdcraw/dcrawbinary.h>
#endif

// Local includes.

#include "ddebug.h"
#include "dmetadata.h"
#include "navigatebarwidget.h"
#include "imagepropertiestab.h"
#include "imagepropertiestab.moc"

namespace Digikam
{

class ImagePropertiesTabPriv
{
public:

    ImagePropertiesTabPriv()
    {
        settingsArea           = 0;
        title                  = 0;
        file                   = 0;
        folder                 = 0;
        modifiedDate           = 0;
        size                   = 0;
        owner                  = 0;
        permissions            = 0;
        title2                 = 0;
        mime                   = 0;
        dimensions             = 0;
        compression            = 0;
        bitDepth               = 0;
        colorMode              = 0;
        title3                 = 0;
        make                   = 0;
        model                  = 0;
        photoDate              = 0;
        aperture               = 0;
        focalLength            = 0;
        exposureTime           = 0;
        sensitivity            = 0;
        exposureMode           = 0;
        flash                  = 0;
        whiteBalance           = 0;
        labelFile              = 0;
        labelFolder            = 0;
        labelFileModifiedDate  = 0;
        labelFileSize          = 0;
        labelFileOwner         = 0;
        labelFilePermissions   = 0;
        labelImageMime         = 0;
        labelImageDimensions   = 0;
        labelImageCompression  = 0;
        labelImageBitDepth     = 0;
        labelImageColorMode    = 0;
        labelPhotoMake         = 0;
        labelPhotoModel        = 0;
        labelPhotoDateTime     = 0;
        labelPhotoAperture     = 0;
        labelPhotoFocalLength  = 0;
        labelPhotoExposureTime = 0;
        labelPhotoSensitivity  = 0;
        labelPhotoExposureMode = 0;
        labelPhotoFlash        = 0;
        labelPhotoWhiteBalance = 0;
    }

    TQLabel             *title;
    TQLabel             *file;
    TQLabel             *folder;
    TQLabel             *modifiedDate;
    TQLabel             *size;
    TQLabel             *owner;
    TQLabel             *permissions;

    TQLabel             *title2;
    TQLabel             *mime;
    TQLabel             *dimensions;
    TQLabel             *compression;
    TQLabel             *bitDepth;
    TQLabel             *colorMode;

    TQLabel             *title3;
    TQLabel             *make;
    TQLabel             *model;
    TQLabel             *photoDate;
    TQLabel             *aperture;
    TQLabel             *focalLength;
    TQLabel             *exposureTime;
    TQLabel             *sensitivity;
    TQLabel             *exposureMode;
    TQLabel             *flash;
    TQLabel             *whiteBalance;

    TQFrame             *settingsArea;

    KSqueezedTextLabel *labelFile;
    KSqueezedTextLabel *labelFolder;
    KSqueezedTextLabel *labelFileModifiedDate;
    KSqueezedTextLabel *labelFileSize;
    KSqueezedTextLabel *labelFileOwner;
    KSqueezedTextLabel *labelFilePermissions;

    KSqueezedTextLabel *labelImageMime;
    KSqueezedTextLabel *labelImageDimensions;
    KSqueezedTextLabel *labelImageCompression;
    KSqueezedTextLabel *labelImageBitDepth;
    KSqueezedTextLabel *labelImageColorMode;

    KSqueezedTextLabel *labelPhotoMake;
    KSqueezedTextLabel *labelPhotoModel;
    KSqueezedTextLabel *labelPhotoDateTime;
    KSqueezedTextLabel *labelPhotoAperture;
    KSqueezedTextLabel *labelPhotoFocalLength;
    KSqueezedTextLabel *labelPhotoExposureTime;
    KSqueezedTextLabel *labelPhotoSensitivity;
    KSqueezedTextLabel *labelPhotoExposureMode;
    KSqueezedTextLabel *labelPhotoFlash;
    KSqueezedTextLabel *labelPhotoWhiteBalance;
};

ImagePropertiesTab::ImagePropertiesTab(TQWidget* parent, bool navBar)
                  : NavigateBarTab(parent)
{
    d = new ImagePropertiesTabPriv;

    setupNavigateBar(navBar);

    TQScrollView *sv = new TQScrollView(this);
    sv->viewport()->setBackgroundMode(TQt::PaletteBackground);
    sv->setResizePolicy(TQScrollView::AutoOneFit);
    sv->setFrameStyle(TQFrame::NoFrame);

    d->settingsArea = new TQFrame(sv->viewport());
    d->settingsArea->setFrameStyle( TQFrame::StyledPanel | TQFrame::Sunken );
    d->settingsArea->setLineWidth( style().pixelMetric(TQStyle::PM_DefaultFrameWidth, this) );

    sv->addChild(d->settingsArea);
    m_navigateBarLayout->addWidget(sv);

    // --------------------------------------------------

    TQGridLayout *settingsLayout = new TQGridLayout(d->settingsArea, 33, 1, KDialog::spacingHint(), 0);

    // --------------------------------------------------

    d->title                    = new TQLabel(i18n("<big><b>File Properties</b></big>"), d->settingsArea);
    d->file                     = new TQLabel(i18n("<b>File</b>:"), d->settingsArea);
    d->folder                   = new TQLabel(i18n("<b>Folder</b>:"), d->settingsArea);
    d->modifiedDate             = new TQLabel(i18n("<b>Modified</b>:"), d->settingsArea);
    d->size                     = new TQLabel(i18n("<b>Size</b>:"), d->settingsArea);
    d->owner                    = new TQLabel(i18n("<b>Owner</b>:"), d->settingsArea);
    d->permissions              = new TQLabel(i18n("<b>Permissions</b>:"), d->settingsArea);

    KSeparator *line            = new KSeparator(TQt::Horizontal, d->settingsArea);
    d->title2                   = new TQLabel(i18n("<big><b>Image Properties</b></big>"), d->settingsArea);
    d->mime                     = new TQLabel(i18n("<b>Type</b>:"), d->settingsArea);
    d->dimensions               = new TQLabel(i18n("<b>Dimensions</b>:"), d->settingsArea);
    d->compression              = new TQLabel(i18n("<b>Compression</b>:"), d->settingsArea);
    d->bitDepth                 = new TQLabel(i18n("<nobr><b>Bit depth</b></nobr>:"), d->settingsArea);
    d->colorMode                = new TQLabel(i18n("<nobr><b>Color mode</b></nobr>:"), d->settingsArea);

    KSeparator *line2           = new KSeparator(TQt::Horizontal, d->settingsArea);
    d->title3                   = new TQLabel(i18n("<big><b>Photograph Properties</b></big>"), d->settingsArea);
    d->make                     = new TQLabel(i18n("<b>Make</b>:"), d->settingsArea);
    d->model                    = new TQLabel(i18n("<b>Model</b>:"), d->settingsArea);
    d->photoDate                = new TQLabel(i18n("<b>Created</b>:"), d->settingsArea);
    d->aperture                 = new TQLabel(i18n("<b>Aperture</b>:"), d->settingsArea);
    d->focalLength              = new TQLabel(i18n("<b>Focal</b>:"), d->settingsArea);
    d->exposureTime             = new TQLabel(i18n("<b>Exposure</b>:"), d->settingsArea);
    d->sensitivity              = new TQLabel(i18n("<b>Sensitivity</b>:"), d->settingsArea);
    d->exposureMode             = new TQLabel(i18n("<nobr><b>Mode/Program</b></nobr>:"), d->settingsArea);
    d->flash                    = new TQLabel(i18n("<b>Flash</b>:"), d->settingsArea);
    d->whiteBalance             = new TQLabel(i18n("<nobr><b>White balance</b></nobr>:"), d->settingsArea);

    d->labelFile                = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelFolder              = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelFileModifiedDate    = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelFileSize            = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelFileOwner           = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelFilePermissions     = new KSqueezedTextLabel(0, d->settingsArea);

    d->labelImageMime           = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelImageDimensions     = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelImageCompression    = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelImageBitDepth       = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelImageColorMode      = new KSqueezedTextLabel(0, d->settingsArea);

    d->labelPhotoMake           = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoModel          = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoDateTime       = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoAperture       = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoFocalLength    = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoExposureTime   = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoSensitivity    = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoExposureMode   = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoFlash          = new KSqueezedTextLabel(0, d->settingsArea);
    d->labelPhotoWhiteBalance   = new KSqueezedTextLabel(0, d->settingsArea);

    int hgt = fontMetrics().height()-2;
    d->title->setAlignment(TQt::AlignCenter);
    d->file->setMaximumHeight(hgt);
    d->folder->setMaximumHeight(hgt);
    d->modifiedDate->setMaximumHeight(hgt);
    d->size->setMaximumHeight(hgt);
    d->owner->setMaximumHeight(hgt);
    d->permissions->setMaximumHeight(hgt);
    d->labelFile->setMaximumHeight(hgt);
    d->labelFolder->setMaximumHeight(hgt);
    d->labelFileModifiedDate->setMaximumHeight(hgt);
    d->labelFileSize->setMaximumHeight(hgt);
    d->labelFileOwner->setMaximumHeight(hgt);
    d->labelFilePermissions->setMaximumHeight(hgt);

    d->title2->setAlignment(TQt::AlignCenter);
    d->mime->setMaximumHeight(hgt);
    d->dimensions->setMaximumHeight(hgt);
    d->compression->setMaximumHeight(hgt);
    d->bitDepth->setMaximumHeight(hgt);
    d->colorMode->setMaximumHeight(hgt);
    d->labelImageMime->setMaximumHeight(hgt);
    d->labelImageDimensions->setMaximumHeight(hgt);
    d->labelImageCompression->setMaximumHeight(hgt);
    d->labelImageBitDepth->setMaximumHeight(hgt);
    d->labelImageColorMode->setMaximumHeight(hgt);

    d->title3->setAlignment(TQt::AlignCenter);
    d->make->setMaximumHeight(hgt);
    d->model->setMaximumHeight(hgt);
    d->photoDate->setMaximumHeight(hgt);
    d->aperture->setMaximumHeight(hgt);
    d->focalLength->setMaximumHeight(hgt);
    d->exposureTime->setMaximumHeight(hgt);
    d->sensitivity->setMaximumHeight(hgt);
    d->exposureMode->setMaximumHeight(hgt);
    d->flash->setMaximumHeight(hgt);
    d->whiteBalance->setMaximumHeight(hgt);
    d->labelPhotoMake->setMaximumHeight(hgt);
    d->labelPhotoModel->setMaximumHeight(hgt);
    d->labelPhotoDateTime->setMaximumHeight(hgt);
    d->labelPhotoAperture->setMaximumHeight(hgt);
    d->labelPhotoFocalLength->setMaximumHeight(hgt);
    d->labelPhotoExposureTime->setMaximumHeight(hgt);
    d->labelPhotoSensitivity->setMaximumHeight(hgt);
    d->labelPhotoExposureMode->setMaximumHeight(hgt);
    d->labelPhotoFlash->setMaximumHeight(hgt);
    d->labelPhotoWhiteBalance->setMaximumHeight(hgt);

    // --------------------------------------------------

    settingsLayout->addMultiCellWidget(d->title, 0, 0, 0, 1);
    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(), 
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 1, 1, 0, 1);
    settingsLayout->addMultiCellWidget(d->file, 2, 2, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFile, 2, 2, 1, 1);
    settingsLayout->addMultiCellWidget(d->folder, 3, 3, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFolder, 3, 3, 1, 1);
    settingsLayout->addMultiCellWidget(d->modifiedDate, 4, 4, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFileModifiedDate, 4, 4, 1, 1);
    settingsLayout->addMultiCellWidget(d->size, 5, 5, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFileSize, 5, 5, 1, 1);
    settingsLayout->addMultiCellWidget(d->owner, 6, 6, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFileOwner, 6, 6, 1, 1);
    settingsLayout->addMultiCellWidget(d->permissions, 7, 7, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelFilePermissions, 7, 7, 1, 1);

    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(),
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 8, 8, 0, 1);
    settingsLayout->addMultiCellWidget(line, 9, 9, 0, 1);
    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(),
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 10, 10, 0, 1);

    settingsLayout->addMultiCellWidget(d->title2, 11, 11, 0, 1);
    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(),
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 12, 12, 0, 1);
    settingsLayout->addMultiCellWidget(d->mime, 13, 13, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelImageMime, 13, 13, 1, 1);
    settingsLayout->addMultiCellWidget(d->dimensions, 14, 14, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelImageDimensions, 14, 14, 1, 1);
    settingsLayout->addMultiCellWidget(d->compression, 15, 15, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelImageCompression, 15, 15, 1, 1);
    settingsLayout->addMultiCellWidget(d->bitDepth, 16, 16, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelImageBitDepth, 16, 16, 1, 1);
    settingsLayout->addMultiCellWidget(d->colorMode, 17, 17, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelImageColorMode, 17, 17, 1, 1);

    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(), 
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 18, 18, 0, 1);
    settingsLayout->addMultiCellWidget(line2, 19, 19, 0, 1);
    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(), 
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 20, 20, 0, 1);

    settingsLayout->addMultiCellWidget(d->title3, 21, 21, 0, 1);
    settingsLayout->addMultiCell(new TQSpacerItem(KDialog::spacingHint(), KDialog::spacingHint(), 
                                 TQSizePolicy::Minimum, TQSizePolicy::MinimumExpanding), 22, 22, 0, 1);
    settingsLayout->addMultiCellWidget(d->make, 23, 23, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoMake, 23, 23, 1, 1);
    settingsLayout->addMultiCellWidget(d->model, 24, 24, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoModel, 24, 24, 1, 1);
    settingsLayout->addMultiCellWidget(d->photoDate, 25, 25, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoDateTime, 25, 25, 1, 1);
    settingsLayout->addMultiCellWidget(d->aperture, 26, 26, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoAperture, 26, 26, 1, 1);
    settingsLayout->addMultiCellWidget(d->focalLength, 27, 27, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoFocalLength, 27, 27, 1, 1);
    settingsLayout->addMultiCellWidget(d->exposureTime, 28, 28, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoExposureTime, 28, 28, 1, 1);
    settingsLayout->addMultiCellWidget(d->sensitivity, 29, 29, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoSensitivity, 29, 29, 1, 1);
    settingsLayout->addMultiCellWidget(d->exposureMode, 30, 30, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoExposureMode, 30, 30, 1, 1);
    settingsLayout->addMultiCellWidget(d->flash, 31, 31, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoFlash, 31, 31, 1, 1);
    settingsLayout->addMultiCellWidget(d->whiteBalance, 32, 32, 0, 0);
    settingsLayout->addMultiCellWidget(d->labelPhotoWhiteBalance, 32, 32, 1, 1);

    settingsLayout->setRowStretch(33, 10);
    settingsLayout->setColStretch(1, 10);
}

ImagePropertiesTab::~ImagePropertiesTab()
{
    delete d;
}

void ImagePropertiesTab::setCurrentURL(const KURL& url)
{
    if (url.isEmpty())
    {
        setNavigateBarFileName();

        d->labelFile->setText(TQString());
        d->labelFolder->setText(TQString());
        d->labelFileModifiedDate->setText(TQString());
        d->labelFileSize->setText(TQString());
        d->labelFileOwner->setText(TQString());
        d->labelFilePermissions->setText(TQString());

        d->labelImageMime->setText(TQString());
        d->labelImageDimensions->setText(TQString());
        d->labelImageCompression->setText(TQString());
        d->labelImageBitDepth->setText(TQString());
        d->labelImageColorMode->setText(TQString());

        d->labelPhotoMake->setText(TQString());
        d->labelPhotoModel->setText(TQString());
        d->labelPhotoDateTime->setText(TQString());
        d->labelPhotoAperture->setText(TQString());
        d->labelPhotoFocalLength->setText(TQString());
        d->labelPhotoExposureTime->setText(TQString());
        d->labelPhotoSensitivity->setText(TQString());
        d->labelPhotoExposureMode->setText(TQString());
        d->labelPhotoFlash->setText(TQString());
        d->labelPhotoWhiteBalance->setText(TQString());

        setEnabled(false);
        return;
    }

    setEnabled(true);

    TQString str;
    TQString unavailable(i18n("<i>unavailable</i>"));

    KFileItem fi(KFileItem::Unknown, KFileItem::Unknown, url);
    TQFileInfo fileInfo(url.path());
    DMetadata metaData(url.path());

    // -- File system information ------------------------------------------

    d->labelFile->setText(url.fileName());
    d->labelFolder->setText(url.directory());

    TQDateTime modifiedDate = fileInfo.lastModified();
    str = TDEGlobal::locale()->formatDateTime(modifiedDate, true, true);
    d->labelFileModifiedDate->setText(str);

    str = TQString("%1 (%2)").arg(TDEIO::convertSize(fi.size()))
                            .arg(TDEGlobal::locale()->formatNumber(fi.size(), 0));
    d->labelFileSize->setText(str);

    d->labelFileOwner->setText( TQString("%1 - %2").arg(fi.user()).arg(fi.group()) );
    d->labelFilePermissions->setText( fi.permissionsString() );

    // -- Image Properties --------------------------------------------------

    TQSize   dims;
    TQString compression, bitDepth, colorMode;
#if KDCRAW_VERSION < 0x000106
    TQString rawFilesExt(KDcrawIface::DcrawBinary::instance()->rawFiles());
#else
    TQString rawFilesExt(KDcrawIface::KDcraw::rawFiles());
#endif
    TQString ext = fileInfo.extension(false).upper();

    if (!ext.isEmpty() && rawFilesExt.upper().contains(ext))
    {
        d->labelImageMime->setText(i18n("RAW Image"));
        compression = i18n("None");
        bitDepth    = "48";
        dims        = metaData.getImageDimensions();
	colorMode   = i18n("Uncalibrated");
    }
    else
    {
        d->labelImageMime->setText(fi.mimeComment());

        KFileMetaInfo meta = fi.metaInfo();
        if (meta.isValid())
        {
            if (meta.containsGroup("Jpeg EXIF Data"))     // JPEG image ?
            {
                dims        = meta.group("Jpeg EXIF Data").item("Dimensions").value().toSize();

                TQString quality = meta.group("Jpeg EXIF Data").item("JPEG quality").value().toString();
                quality.isEmpty() ? compression = unavailable :
                                    compression = i18n("JPEG quality %1").arg(quality);
                bitDepth    = meta.group("Jpeg EXIF Data").item("BitDepth").value().toString();
                colorMode   = meta.group("Jpeg EXIF Data").item("ColorMode").value().toString();
            }

            if (meta.containsGroup("General"))
            {
                if (dims.isEmpty() ) 
                    dims = meta.group("General").item("Dimensions").value().toSize();
                if (compression.isEmpty()) 
                    compression =  meta.group("General").item("Compression").value().toString();
                if (bitDepth.isEmpty()) 
                    bitDepth = meta.group("General").item("BitDepth").value().toString();
                if (colorMode.isEmpty()) 
                    colorMode = meta.group("General").item("ColorMode").value().toString();
            }

            if (meta.containsGroup("Technical"))
            {
                if (dims.isEmpty()) 
                    dims = meta.group("Technical").item("Dimensions").value().toSize();
                if (compression.isEmpty()) 
                    compression = meta.group("Technical").item("Compression").value().toString();
                if (bitDepth.isEmpty()) 
                    bitDepth = meta.group("Technical").item("BitDepth").value().toString();
                if (colorMode.isEmpty()) 
                    colorMode =  meta.group("Technical").item("ColorMode").value().toString();
            }
        }
    }

    TQString mpixels;
    mpixels.setNum(dims.width()*dims.height()/1000000.0, 'f', 2);
    str = (!dims.isValid()) ? i18n("Unknown") : i18n("%1x%2 (%3Mpx)")
          .arg(dims.width()).arg(dims.height()).arg(mpixels);
    d->labelImageDimensions->setText(str);
    d->labelImageCompression->setText(compression.isEmpty() ? unavailable : compression);
    d->labelImageBitDepth->setText(bitDepth.isEmpty() ? unavailable : i18n("%1 bpp").arg(bitDepth));
    d->labelImageColorMode->setText(colorMode.isEmpty() ? unavailable : colorMode);

    // -- Photograph information ------------------------------------------
    // NOTA: If something is changed here, please updated albumfiletip section too.

    PhotoInfoContainer photoInfo = metaData.getPhotographInformations();

    if (photoInfo.isEmpty())
    {
        d->title3->hide();
        d->make->hide();
        d->model->hide();
        d->photoDate->hide();
        d->aperture->hide();
        d->focalLength->hide();
        d->exposureTime->hide();
        d->sensitivity->hide();
        d->exposureMode->hide();
        d->flash->hide();
        d->whiteBalance->hide();
        d->labelPhotoMake->hide();
        d->labelPhotoModel->hide();
        d->labelPhotoDateTime->hide();
        d->labelPhotoAperture->hide();
        d->labelPhotoFocalLength->hide();
        d->labelPhotoExposureTime->hide();
        d->labelPhotoSensitivity->hide();
        d->labelPhotoExposureMode->hide();
        d->labelPhotoFlash->hide();
        d->labelPhotoWhiteBalance->hide();
    }
    else
    {
        d->title3->show();
        d->make->show();
        d->model->show();
        d->photoDate->show();
        d->aperture->show();
        d->focalLength->show();
        d->exposureTime->show();
        d->sensitivity->show();
        d->exposureMode->show();
        d->flash->show();
        d->whiteBalance->show();
        d->labelPhotoMake->show();
        d->labelPhotoModel->show();
        d->labelPhotoDateTime->show();
        d->labelPhotoAperture->show();
        d->labelPhotoFocalLength->show();
        d->labelPhotoExposureTime->show();
        d->labelPhotoSensitivity->show();
        d->labelPhotoExposureMode->show();
        d->labelPhotoFlash->show();
        d->labelPhotoWhiteBalance->show();
    }

    d->labelPhotoMake->setText(photoInfo.make.isEmpty() ? unavailable : photoInfo.make);
    d->labelPhotoModel->setText(photoInfo.model.isEmpty() ? unavailable : photoInfo.model);

    if (photoInfo.dateTime.isValid())
    {
        str = TDEGlobal::locale()->formatDateTime(photoInfo.dateTime, true, true);
        d->labelPhotoDateTime->setText(str);
    }
    else
        d->labelPhotoDateTime->setText(unavailable);

    d->labelPhotoAperture->setText(photoInfo.aperture.isEmpty() ? unavailable : photoInfo.aperture);

    if (photoInfo.focalLength35mm.isEmpty())
        d->labelPhotoFocalLength->setText(photoInfo.focalLength.isEmpty() ? unavailable : photoInfo.focalLength);
    else 
    {
        str = i18n("%1 (35mm: %2)").arg(photoInfo.focalLength).arg(photoInfo.focalLength35mm);
        d->labelPhotoFocalLength->setText(str);
    }

    d->labelPhotoExposureTime->setText(photoInfo.exposureTime.isEmpty() ? unavailable : photoInfo.exposureTime);
    d->labelPhotoSensitivity->setText(photoInfo.sensitivity.isEmpty() ? unavailable : i18n("%1 ISO").arg(photoInfo.sensitivity));

    if (photoInfo.exposureMode.isEmpty() && photoInfo.exposureProgram.isEmpty())
        d->labelPhotoExposureMode->setText(unavailable);
    else if (!photoInfo.exposureMode.isEmpty() && photoInfo.exposureProgram.isEmpty())
        d->labelPhotoExposureMode->setText(photoInfo.exposureMode);        
    else if (photoInfo.exposureMode.isEmpty() && !photoInfo.exposureProgram.isEmpty())
        d->labelPhotoExposureMode->setText(photoInfo.exposureProgram);        
    else 
    {
        str = TQString("%1 / %2").arg(photoInfo.exposureMode).arg(photoInfo.exposureProgram);
        d->labelPhotoExposureMode->setText(str);
    }

    d->labelPhotoFlash->setText(photoInfo.flash.isEmpty() ? unavailable : photoInfo.flash);
    d->labelPhotoWhiteBalance->setText(photoInfo.whiteBalance.isEmpty() ? unavailable : photoInfo.whiteBalance);
}

}  // NameSpace Digikam
