/*
 *
 *  Device Mime Converter for libtdebluez
 *
 *  Copyright (C) 2003 by Fred Schaettgen
 *  Copyright (C) 2018  Emanoil Kotsev <deloptes@gmail.com>
 *
 *
 *  This file is part of libtdebluez.
 *
 *  libtdebluez is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  libtdebluez is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with kbluetooth; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <kmimetype.h>

#include "devicemimeconverter.h"

namespace TDEBluetooth
{

DeviceMimeConverter::DeviceMimeConverter()
{
    getIconName("bluetooth/unknown-device-class");
    getIconName("bluetooth/misc-device-class");
    getIconName("bluetooth/computer-device-class");
    getIconName("bluetooth/phone-device-class");
    getIconName("bluetooth/lan-device-class");
    getIconName("bluetooth/av-device-class");
    getIconName("bluetooth/peripheral-device-class");
    getIconName("bluetooth/mouse-device-class");
    getIconName("bluetooth/keyboard-device-class");
    getIconName("bluetooth/imaging-device-class");
}

void DeviceMimeConverter::getIconName(TQString mime)
{
    TQString iconName = KMimeType::mimeType(mime)->icon(TQString::null, false);
    mimeTypeToIconMap[mime] = iconName;
}

DeviceMimeConverter* DeviceMimeConverter::getInstance()
{
    static DeviceMimeConverter instance;
    return &instance;
}

TQString DeviceMimeConverter::classToIconName(int n)
{
    return DeviceMimeConverter::mimeTypeToIcon(DeviceMimeConverter::classToMimeType(n));
}

/*
 * device classes
 *
    AUDIO_VIDEO (Value: 0x00000400)
    COMPUTER (Value: 0x00000100)
    HEALTH (Value: 0x00000900)
    IMAGING (Value: 0x00000600)
    MISC (Value: 0x00000000)
    NETWORKING (Value: 0x00000300)
    PERIPHERAL (Value: 0x00000500)
    PHONE (Value: 0x00000200)
    TOY (Value: 0x00000800)
    UNCATEGORIZED (Value: 0x00001f00)
    WEARABLE (Value: 0x00000700)
*/

TQString DeviceMimeConverter::classToMimeType(int n)
{
    TQString mimeType;
    int major = ((n & 0x001F00) >> 8);
    int minor = ((n >> 2) & 0x30);
    switch (major)
    {
    case 0x00:
        mimeType = "bluetooth/misc-device-class";
        break;
    case 0x01:
        mimeType = "bluetooth/computer-device-class";
        break;
    case 0x02:
        mimeType = "bluetooth/phone-device-class";
        break;
    case 0x03:
        mimeType = "bluetooth/lan-device-class";
        break;
    case 0x04:
        mimeType = "bluetooth/av-device-class";
        break;
    case 0x05:
        switch (minor)
        {
        case 0x10:
            mimeType = "bluetooth/keyboard-device-class";
            break;
        case 0x20:
            mimeType = "bluetooth/mouse-device-class";
            break;
        default:
            mimeType = "bluetooth/peripheral-device-class";
        }
        break;
    case 0x06:
        mimeType = "bluetooth/imaging-device-class";
        break;
    case 0x07:
        mimeType = "bluetooth/wearable-device-class";
        break;
    case 0x08:
        mimeType = "bluetooth/toy-device-class";
        break;
    case 0x09:
        mimeType = "bluetooth/health-device-class";
        break;
    default:
        mimeType = "bluetooth/unknown-device-class";
    }
    return mimeType;
}

TQString DeviceMimeConverter::mimeTypeToIcon(TQString mime)
{
    DeviceMimeConverter* c = DeviceMimeConverter::getInstance();
    if (c->mimeTypeToIconMap.find(mime) != c->mimeTypeToIconMap.end())
    {
        return c->mimeTypeToIconMap[mime];
    }
    else
    {
        return c->mimeTypeToIconMap["bluetooth/unknown-device-class"];
    }
}

} // TDEBluetooth
