// Author: Denis Kozadaev - (c) 2025


#include "mainwindow.h"

#include <tdeactionclasses.h>
#include <tdemenubar.h>
#include <tdelocale.h>

#include <tqdialog.h>
#include <tqpushbutton.h>
#include <tqradiobutton.h>
#include <tqbuttongroup.h>
#include <tqcheckbox.h>
#include <tqlayout.h>


MainWindow::MainWindow(TQWidget* parent, const char* name)
	:TDEMainWindow(parent, name)
{
	TDEAction	*actionQuit, *actionNew;

	mMenu = new TDEPopupMenu(this);

	actionQuit = KStdAction::quit(this, TQ_SLOT(close()), 0);
	actionNew = new TDEAction(i18n("&New Game"), "reload",
		TQt::CTRL + TQt::Key_N, this, TQ_SLOT(newGame()),
		this, i18n("New Game"));

	actionNew->plug(mMenu);
	actionQuit->plug(mMenu);
	menuBar()->insertItem(i18n("Knight's tour"), mMenu);

	mGame = new TDEPopupMenu(this);
	mGame->setCheckable(true);
	idTip = mGame->insertItem(i18n("&Highlight tips"),
		this, TQ_SLOT(toggleTips()));
	idKnight = mGame->insertItem(i18n("&Show the knight"),
		this, TQ_SLOT(toggleKnight()));
	idCell = mGame->insertItem(i18n("&Random starting cell"),
		this, TQ_SLOT(randomCell()));
	menuBar()->insertItem(i18n("Settings"), mGame);


	mBoard = new GameBoard(this);
	setCentralWidget(mBoard);

	settings = new TQSettings(TQSettings::Native);
	settings->setPath("DilOSTeam", "KnightsTour", TQSettings::User);
	loadSettings();

	mBoard->newGame(defSize);
}

MainWindow::~MainWindow()
{

	saveSettings();

	delete mBoard;
	delete mGame;
	delete mMenu;
	delete settings;
}


/*
 * Load program configuration
 */
void
MainWindow::loadSettings()
{
	bool	val;

	defSize = settings->readNumEntry("/KnightTour/Size", 8);
	switch (defSize) {
	case 8:
	case 10:
	case 15:
		/* valid size */
		break;
	default:
		/* set size by default */
		defSize = 8;
		break;
	}

	val = settings->readBoolEntry("/KnightTour/Tip", true);
	mGame->setItemChecked(idTip, !val);
	toggleTips();

	val = settings->readBoolEntry("/KnightTour/Visibility", true);
	mGame->setItemChecked(idKnight, !val);
	toggleKnight();

	val = settings->readBoolEntry("/KnightTour/RandomCell", true);
	mGame->setItemChecked(idCell, !val);
	randomCell();
}


/*
 * Save the current program configuration
 */
void
MainWindow::saveSettings()
{

	settings->writeEntry("/KnightTour/Size", defSize);
	settings->writeEntry("/KnightTour/Tip", mGame->isItemChecked(idTip));
	settings->writeEntry("/KnightTour/Visibility",
		mGame->isItemChecked(idKnight));
	settings->writeEntry("/KnightTour/RandomCell",
		mGame->isItemChecked(idCell));
}


/*
 * Request a new game
 */
void
MainWindow::newGame()
{
	TQDialog	*dlg;
	TQVBoxLayout	*vbox, *gbox;
	TQHBoxLayout	*hbox;
	TQButtonGroup	*grp;
	TQRadioButton	*b8, *b10, *b15;
	TQCheckBox	*rnd;
	TQButton	*ok, *cancel;

	dlg = new TQDialog(this, nullptr, true);
	dlg->setCaption(i18n("Board size"));

	vbox = new TQVBoxLayout(dlg);

	grp = new TQButtonGroup(i18n("Select the board size"), dlg);
	grp->setFocusPolicy(TQWidget::NoFocus);
	vbox->addWidget(grp);

	gbox = new TQVBoxLayout(grp);

	b8 = new TQRadioButton(i18n("8x8"), grp);
	b8->setFocusPolicy(TQWidget::NoFocus);
	b8->setChecked(defSize == 8);
	gbox->addWidget(b8);

	b10 = new TQRadioButton(i18n("10x10"), grp);
	b10->setChecked(defSize == 10);
	b10->setFocusPolicy(TQWidget::NoFocus);
	gbox->addWidget(b10);

	b15 = new TQRadioButton(i18n("15x15"), grp);
	b15->setChecked(defSize == 15);
	b15->setFocusPolicy(TQWidget::NoFocus);
	gbox->addWidget(b15);

	rnd = new TQCheckBox(i18n("Randomize starting cell"), dlg);
	rnd->setChecked(mGame->isItemChecked(idCell));
	vbox->addWidget(rnd);

	hbox = new TQHBoxLayout();
	vbox->addLayout(hbox);

	cancel = new TQPushButton(i18n("Cancel"), dlg);
	connect(cancel, TQ_SIGNAL(clicked()), dlg, TQ_SLOT(reject()));
	hbox->addWidget(cancel);

	ok = new TQPushButton(i18n("Apply"), dlg);
	connect(ok, TQ_SIGNAL(clicked()), dlg, TQ_SLOT(accept()));
	hbox->addWidget(ok);

	if (dlg->exec()) {
		if (b8->isChecked())
			defSize = 8;
		else if (b10->isChecked())
			defSize = 10;
		else if (b15->isChecked())
			defSize = 15;
		else {
			/* WTF? Set by default */
			defSize = 8;
		}
		mGame->setItemChecked(idCell, rnd->isChecked());
		mBoard->setRandomCell(rnd->isChecked());
		mBoard->newGame(defSize);
	}

	delete ok;
	delete cancel;
	delete hbox;
	delete b15;
	delete b10;
	delete b8;
	delete gbox;
	delete grp;
	delete vbox;
	delete dlg;
}


/*
 * Toggle the highlight tips to move the knight
 */
void
MainWindow::toggleTips()
{

	mGame->setItemChecked(idTip, !mGame->isItemChecked(idTip));
	mBoard->setTips(mGame->isItemChecked(idTip));
}


/*
 * Toggle the knight visibility
 */
void
MainWindow::toggleKnight()
{

	mGame->setItemChecked(idKnight, !mGame->isItemChecked(idKnight));
	mBoard->setKnight(mGame->isItemChecked(idKnight));
}


/*
 * Toggle random starting cell of the knight
 * or selected at the start
 */
void
MainWindow::randomCell()
{

	mGame->setItemChecked(idCell, !mGame->isItemChecked(idCell));
	mBoard->setRandomCell(mGame->isItemChecked(idCell));
}

#include "mainwindow.moc"
