/***************************************************************************
                          kpayeesview.h
                          -------------
    begin                : Thu Jan 24 2002
    copyright            : (C) 2000-2002 by Michael Edwardes
                                    2005 by Andrea Nicolai
                                    2006 by Thomas Baumgart
    email                : mte@users.sourceforge.net
                           Javier Campos Morales <javi_c@users.sourceforge.net>
                           Felix Rodriguez <frodriguez@users.sourceforge.net>
                           John C <thetacoturtle@users.sourceforge.net>
                           Thomas Baumgart <ipwizard@users.sourceforge.net>
                           Kevin Tambascio <ktambascio@users.sourceforge.net>
                           Andreas Nicolai <Andreas.Nicolai@gmx.net>
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPAYEESVIEW_H
#define KPAYEESVIEW_H

// ----------------------------------------------------------------------------
// TQt Includes

#include <tqwidget.h>
class TQSplitter;

// ----------------------------------------------------------------------------
// TDE Includes

#include <tdelistview.h>
#include <tdepopupmenu.h>

// ----------------------------------------------------------------------------
// Project Includes

#include "kpayeesviewdecl.h"
#include "kmymoneytransaction.h"
#include <kmymoney/mymoneypayee.h>

class TDEListViewSearchLineWidget;

/**
  * @author Michael Edwardes, Thomas Baumgart
  */

/**
  * This class is used to store a sorted vector of pointers to
  * the transactions that are visible in a ledger view. When the
  * vector is created, the sort method is set to SortPostDate.
  * The sort type can be changed using the method setSortType().
  */
class KTransactionPtrVector : public TQPtrVector<KMyMoneyTransaction> {
public:
  /**
    * This enumerator defines the possible sort methods.
    * Possible values are:
    *
    */
  enum TransactionSortE {
    SortEntryDate = 0,      /**< Sort the vector so that the transactions appear sorted
                              *  according to their entry date
                              */
    SortPostDate,           /**< Sort the vector so that the transactions appear sorted
                              *     according to their post date
                              */
    SortTypeNr,             /**< Sort the vector so that the transactions appear sorted
                              *     according to their action and nr
                              */
    SortReceiver,           /**< Sort the vector so that the transactions appear sorted
                              *     according to their receiver
                              */
    SortValue,              /**< Sort the vector so that the transactions appear sorted
                              *     according to their value
                              */
    SortNr,                 /**< Sort the vector so that the transactions appear sorted
                              *     according to nr field contents
                              */
    SortEntryOrder          /**< Sort the vector so that the transactions appear sorted
                              *     according to order of entry
                              */
  };

  KTransactionPtrVector() { m_sortType = SortPostDate; };
  ~KTransactionPtrVector() {}

  /**
    * This method is used to set a different sort type.
    * The vector is resorted. See KTransactionPtrVector::TransactionSortE
    * for possible values.
    */
  void setSortType(const TransactionSortE type);

  /**
    * This method returns the current sort type.
    *
    * @return transactionSortE value of sort order. See
    *         KTransactionPtrVector::TransactionSortE for possible values.
    */
  TransactionSortE sortType(void) const { return m_sortType; };

  /**
    * This method is used to set the account id to have a chance to
    * get information about the split referencing the current account
    * during the sort phase.
    */
  void setAccountId(const TQString& id);

  /**
    * This method is used to set the payee id to have a chance to
    * get information about the split referencing the current payee
    * during the sort phase.
    */
  void setPayeeId(const TQString& id);

protected:
  int compareItems(KTransactionPtrVector::Item d1, KTransactionPtrVector::Item d2);

private:
  int compareItems(const TQString& s1, const TQString& s2) const;

private:
  enum {
    AccountMode = 0,
    PayeeMode
  };
  short             m_idMode;
  TQString           m_id;
  TransactionSortE  m_sortType;
};



/**
  * This class represents an item in the payees list view.
  */
class KPayeeListItem : public TDEListViewItem
{
public:
  /**
    * Constructor to be used to construct a payee entry object.
    *
    * @param parent pointer to the TDEListView object this entry should be
    *               added to.
    * @param payee const reference to MyMoneyPayee for which
    *               the TDEListView entry is constructed
    */
  KPayeeListItem(TDEListView *parent, const MyMoneyPayee& payee);
  ~KPayeeListItem();

  /**
    * This method is re-implemented from TQListViewItem::paintCell().
    * Besides the standard implementation, the TQPainter is set
    * according to the applications settings.
    */
  void paintCell(TQPainter *p, const TQColorGroup & cg, int column, int width, int align);

  const MyMoneyPayee& payee(void) const { return m_payee; };

private:
  MyMoneyPayee  m_payee;
};

/**
  * This class represents an item in the transaction list view. It is used
  * by the KPayeesView to select between transactions.
  */
class KTransactionListItem : public TDEListViewItem
{
public:
  KTransactionListItem(TDEListView* view, KTransactionListItem* parent, const TQString& accountId, const TQString& transaction);
  ~KTransactionListItem();

  const TQString& transactionId(void) const { return m_transactionId; };

  const TQString& accountId(void) const { return m_accountId; };

  /**
    * use my own paint method
    */
  void paintCell(TQPainter *p, const TQColorGroup &cg, int column, int width, int alignment);

  /**
    * use my own backgroundColor method
    */
  const TQColor backgroundColor();

private:
  TQString m_transactionId;
  TQString m_accountId;
};

class KPayeesView : public KPayeesViewDecl
{
  TQ_OBJECT
  

public:
  KPayeesView(TQWidget *parent=0, const char *name=0);
  ~KPayeesView();
  void show(void);

public slots:
  void slotSelectPayeeAndTransaction(const TQString& payeeId, const TQString& accountId = TQString(), const TQString& transactionId = TQString());
  void slotLoadPayees(void);
  void slotStartRename(void);
  void slotHelp(void);

protected:
  void resizeEvent(TQResizeEvent*);
  void loadPayees(void);
  void selectedPayees(TQValueList<MyMoneyPayee>& payeesList) const;
  void ensurePayeeVisible(const TQString& id);
  void clearItemData(void);

protected slots:
  /**
    * This method loads the m_transactionList, clears
    * the m_TransactionPtrVector and rebuilds and sorts
    * it according to the current settings. Then it
    * loads the m_transactionView with the transaction data.
    */
  void showTransactions(void);

  /**
    * This slot is called whenever the selection in m_payeesList
    * has been changed.
    */
  void slotSelectPayee(void);

  /**
    * This slot marks the current selected payee as modified (dirty).
    */
  void slotPayeeDataChanged(void);
  void slotKeyListChanged(void);

  /**
    * This slot is called when the name of a payee is changed inside
    * the payee list view and only a single payee is selected.
    */
  void slotRenamePayee(TQListViewItem *p, int col, const TQString& txt);

  /**
    * Updates the payee data in m_payee from the information in the
    * payee information widget.
    */
  void slotUpdatePayee(void);

  void slotTransactionDoubleClicked(TQListViewItem *);

private slots:
  void rearrange(void);

  /**
    * This slot receives the signal from the listview control that an item was right-clicked,
    * If @p item points to a real payee item, emits openContextMenu().
    *
    * @param lv pointer to the listview sending the signal
    * @param item the item on which the cursor resides
    * @param p position of the pointer device
    */
  void slotOpenContextMenu(TDEListView* lv, TQListViewItem* item, const TQPoint& p);

  void slotQueueUpdate(void);

  void slotActivateUpdate(void);

  void slotChooseDefaultAccount(void);

private:
  void readConfig(void);

signals:
  void transactionSelected(const TQString& accountId, const TQString& transactionId);
  void openContextMenu(const MyMoneyObject& obj);
  void selectObjects(const TQValueList<MyMoneyPayee>& payees);

private:
  MyMoneyPayee m_payee;
  TQString      m_newName;

  TQSplitter*    m_splitter;

  /**
    * This member holds a list of all transactions
    */
  TQValueList<KMyMoneyTransaction> m_transactionList;

  /**
    * This member keeps a vector of pointers to all visible (filtered)
    * transaction in m_transactionList in sorted order. Sorting is done
    * in KTransactionPtrVector::compareItems
    */
  KTransactionPtrVector m_transactionPtrVector;

  /**
    * This member holds the state of the toggle switch used
    * to suppress updates due to MyMoney engine data changes
    */
  bool m_needReload;

  /**
    * Search widget for the list
    */
  TDEListViewSearchLineWidget*  m_searchWidget;
  bool m_needConnection;

  /**
    * Counting semaphore to collect updates
    */
  int m_updatesQueued;

  /**
   * Semaphore to suppress loading during selection
   */
  bool m_inSelection;
};

#endif
