/***************************************************************************
 *   Copyright (C) 2003 by Harald Fernengel                                *
 *   harry@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tqstring.h>
#include <tqvariant.h>
#include <tqheader.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlistview.h>
#include <tqgroupbox.h>
#include <tqhbox.h>
#include <tqregexp.h>

#include <tdeconfig.h>
#include <kdebug.h>
#include <tdeglobal.h>
#include <tdelocale.h>
#include <kservice.h>
#include <ktrader.h>
#include <tdeapplication.h>
#include <kdevplugin.h>
#include "domutil.h"

#include "languageselectwidget.h"
#include "plugincontroller.h"

class LangPluginItem : public TQCheckListItem
{
public:
    // name - "Name", label - "GenericName", info - "Comment"
    LangPluginItem( TQListView * parent, TQString const & name, TQString const & label,
                TQString const & info )
        : TQCheckListItem( parent, label, TQCheckListItem::CheckBox),
        _name( name ), _info( info )
    {}

    TQString info() { return _info; }
    TQString name() { return _name; }

private:
    TQString _name;
    TQString _info;
};


LanguageSelectWidget::LanguageSelectWidget(TQDomDocument &projectDom,
                                   TQWidget *parent, const char *name)
    : TQWidget(parent, name), m_projectDom(projectDom)
{
    init();
}

void LanguageSelectWidget::init()
{
    TQVBoxLayout *layout = new TQVBoxLayout(this);

    TQGroupBox * groupBox1 = new TQGroupBox( i18n("Additional Language Support"), this );
    groupBox1->setColumnLayout(0, Qt::Vertical );
    groupBox1->layout()->setSpacing( 6 );
    groupBox1->layout()->setMargin( 11 );
    TQVBoxLayout * groupBox1Layout = new TQVBoxLayout( groupBox1->layout() );
    groupBox1Layout->setAlignment( TQt::AlignTop );

    _currentLanguage = new TQLabel( "", groupBox1 );

    _pluginList = new TQListView( groupBox1 );
    _pluginList->setResizeMode( TQListView::LastColumn );
    _pluginList->addColumn("");
    _pluginList->header()->hide();

    groupBox1Layout->addWidget(_currentLanguage);
    groupBox1Layout->addWidget( _pluginList );
    layout->addWidget( groupBox1 );

    TQGroupBox * groupBox2 = new TQGroupBox( i18n("Description"), this );
    groupBox2->setColumnLayout(0, Qt::Vertical );
    groupBox2->layout()->setSpacing( 6 );
    groupBox2->layout()->setMargin( 11 );
    TQVBoxLayout * groupBox2Layout = new TQVBoxLayout( groupBox2->layout() );
    groupBox2Layout->setAlignment( TQt::AlignTop );

    _pluginDescription = new TQLabel( groupBox2 );
    _pluginDescription->setAlignment( int( TQLabel::WordBreak | TQLabel::AlignVCenter ) );

    groupBox2Layout->addWidget( _pluginDescription );

    layout->addWidget( groupBox2 );

    connect( _pluginList, TQT_SIGNAL( selectionChanged( TQListViewItem * ) ), this, TQT_SLOT( itemSelected( TQListViewItem * ) ) );

    readProjectConfig();
}


LanguageSelectWidget::~LanguageSelectWidget()
{}

void LanguageSelectWidget::readProjectConfig()
{
    TDETrader::OfferList languageSupportOffers =
        TDETrader::self()->query(TQString::fromLatin1("TDevelop/LanguageSupport"),
                               TQString::fromLatin1("[X-TDevelop-Version] == %1"
                               ).arg( TDEVELOP_PLUGIN_VERSION ));

    TQStringList languages = DomUtil::readListEntry(m_projectDom, "/general/secondaryLanguages", "language");
    TQString language = DomUtil::readEntry(m_projectDom, "/general/primarylanguage");
    _currentLanguage->setText(i18n("Primary language is '%1'. Please select additional languages the project might contain.").arg(language));

    for (TDETrader::OfferList::ConstIterator it = languageSupportOffers.begin(); it != languageSupportOffers.end(); ++it)
    {
        TQString la = (*it)->property("X-TDevelop-Language").toString();
        if (la == language)
            continue;
        LangPluginItem *item = new LangPluginItem( _pluginList, (*it)->property("X-TDevelop-Language").toString(), (*it)->genericName(), (*it)->comment() );
        item->setOn(languages.contains(la));
    }

    TQListViewItem * first = _pluginList->firstChild();
    if ( first ) {
        _pluginList->setSelected( first, true );
    }
}

void LanguageSelectWidget::itemSelected( TQListViewItem * item )
{
    if ( !item ) return;

    LangPluginItem * pitem = static_cast<LangPluginItem*>( item );
    _pluginDescription->setText( pitem->info() );
}

void LanguageSelectWidget::saveProjectConfig()
{
    TQStringList languages;

    TQListViewItemIterator it( _pluginList );
    while ( it.current() )
    {
        LangPluginItem * item = static_cast<LangPluginItem*>( it.current() );
        if (item->isOn())
        {
            languages.append( item->name() );
        }
        ++it;
    }

    DomUtil::writeListEntry(m_projectDom, "/general/secondaryLanguages", "language", languages);
}


void LanguageSelectWidget::accept()
{
    saveProjectConfig();
    emit accepted();
}

#include "languageselectwidget.moc"
