/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz (peter.penz@gmx.at) and              *
 *   and Patrice Tremblay                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "statusbarspaceinfo.h"

#include <tqpainter.h>
#include <tqtimer.h>
#include <tdeglobalsettings.h>
#include <kdiskfreesp.h>
#include <tdelocale.h>
#include <tdeio/job.h>

StatusBarSpaceInfo::StatusBarSpaceInfo(TQWidget* parent) :
    TQWidget(parent),
    m_gettingSize(false),
    m_kBSize(0),
    m_kBAvailable(0)
{
    setMinimumWidth(200);

    // Update the space information each 10 seconds. Polling is useful
    // here, as files can be deleted/added outside the scope of Dolphin.
    TQTimer* timer = new TQTimer(this);
    connect(timer, TQT_SIGNAL(timeout()), this, TQT_SLOT(refresh()));
    timer->start(10000);
}

StatusBarSpaceInfo::~StatusBarSpaceInfo()
{
}

void StatusBarSpaceInfo::setURL(const KURL& url)
{
    m_url = url;
    refresh();
    update();
}

void StatusBarSpaceInfo::paintEvent(TQPaintEvent* /* event */)
{
    TQPainter painter(this);
    const int barWidth = width();
    const int barTop = 2;
    const int barHeight = height() - 4;

    TQString text;

    const int widthDec = 3;  // visual decrement for the available width

    const TQColor c1 = colorGroup().background();
    const TQColor c2 = TDEGlobalSettings::buttonTextColor();
    const TQColor frameColor((c1.red()   + c2.red())   / 2,
                            (c1.green() + c2.green()) / 2,
                            (c1.blue()  + c2.blue())  / 2);
    painter.setPen(frameColor);

    const TQColor backgrColor = TDEGlobalSettings::baseColor();
    painter.setBrush(backgrColor);

    painter.drawRect(TQRect(0, barTop + 1 , barWidth - widthDec, barHeight));

    if ((m_kBSize > 0) && (m_kBAvailable > 0)) {
        // draw 'used size' bar
        painter.setPen(TQt::NoPen);
        painter.setBrush(progressColor(backgrColor));
        int usedWidth = barWidth - static_cast<int>((m_kBAvailable *
                                                    static_cast<float>(barWidth)) / m_kBSize);
        const int left = 1;
        int right = usedWidth - (widthDec + 1);
        if (right < left) {
            right = left;
        }
        painter.drawRect(TQRect(left, barTop + 2, right, barHeight - 2));

        text = i18n("%1 free")
               .arg(TDEIO::convertSizeFromKB(m_kBAvailable));
    }
    else {
        if (m_gettingSize) {
            text = i18n("Getting size...");
        }
        else {
            text = "";
            TQTimer::singleShot(0, this, TQT_SLOT(hide()));
        }
    }

    // draw text (usually 'Y GB free')
    painter.setPen(TDEGlobalSettings::textColor());
    painter.drawText(TQRect(1, 1, barWidth - 2, barHeight + 4),
                     TQt::AlignHCenter | TQt::AlignVCenter | TQt::WordBreak,
                     text);
}


void StatusBarSpaceInfo::slotFoundMountPoint(const unsigned long& kBSize,
                                             const unsigned long& /* kBUsed */,
                                             const unsigned long& kBAvailable,
                                             const TQString& /* mountPoint */)
{
    m_gettingSize = false;
    m_kBSize = kBSize;
    m_kBAvailable = kBAvailable;

    // Bypass a the issue (?) of KDiskFreeSp that for protocols like
    // FTP, SMB the size of root partition is returned.
    // TODO: check whether KDiskFreeSp is buggy or Dolphin uses it in a wrong way
    const TQString protocol(m_url.protocol());
    if (!protocol.isEmpty() && (protocol != "file")) {
        m_kBSize = 0;
        m_kBAvailable = 0;
    }

    update();
}

void StatusBarSpaceInfo::slotDone()
{
    m_gettingSize = false;
    if ((m_kBSize > 0) && (m_kBAvailable > 0)) {
       show();
    }

    update();
}

void StatusBarSpaceInfo::refresh()
{
    m_gettingSize = true;
    m_kBSize = 0;
    m_kBAvailable = 0;

    const TQString mountPoint(TDEIO::findPathMountPoint(m_url.path()));

    KDiskFreeSp* job = new KDiskFreeSp(TQT_TQOBJECT(this));
    connect(job, TQT_SIGNAL(foundMountPoint(const unsigned long&,
                                        const unsigned long&,
                                        const unsigned long&,
                                        const TQString& )),
            this, TQT_SLOT(slotFoundMountPoint(const unsigned long&,
                                           const unsigned long&,
                                           const unsigned long&,
                                           const TQString& )));
    connect(job, TQT_SIGNAL(done()),
            this, TQT_SLOT(slotDone()));

    job->readDF(mountPoint);
}

TQColor StatusBarSpaceInfo::progressColor(const TQColor& bgColor) const
{
    TQColor color = TDEGlobalSettings::buttonBackground();

    // assure that enough contrast is given between the background color
    // and the progressbar color
    int bgRed   = bgColor.red();
    int bgGreen = bgColor.green();
    int bgBlue  = bgColor.blue();

    const int backgrBrightness = tqGray(bgRed, bgGreen, bgBlue);
    const int progressBrightness = tqGray(color.red(), color.green(), color.blue());

    const int limit = 32;
    const int diff = backgrBrightness - progressBrightness;
    bool adjustColor = ((diff >= 0) && (diff <  limit)) ||
                       ((diff  < 0) && (diff > -limit));
    if (adjustColor) {
        const int inc = (backgrBrightness < 2 * limit) ? (2 * limit) : -limit;
        color = TQColor(bgRed + inc, bgGreen + inc, bgBlue + inc);
    }

    return color;
}

#include "statusbarspaceinfo.moc"
