/*
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


// QT includes
#include <tqkeycode.h>
#include <tqcursor.h>

// KDE includes
#include <tdeapplication.h>
#include <tdemenubar.h>
#include <kstatusbar.h>
#include <tdemessagebox.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdestdaccel.h>
#include <kstdaction.h>
#include <kstdgameaction.h>
#include <tdeaction.h>
#include <tdeshortcut.h>

// own includes
#include "top.h"
#include "kpoker.h"
#include "defines.h"
#include "version.h"


PokerWindow::PokerWindow()
{
  m_kpok = new kpok(this, 0);
  setCentralWidget( m_kpok );
  m_kpok->show();

  clickToHoldIsShown = false;

  LHLabel = new TQLabel(statusBar());
  LHLabel->adjustSize();

  connect(m_kpok, TQT_SIGNAL(changeLastHand(const TQString &, bool)), 
	  this,   TQT_SLOT(setHand(const TQString &, bool)));
  connect(m_kpok, TQT_SIGNAL(showClickToHold(bool)), 
	  this,   TQT_SLOT(showClickToHold(bool)));
  connect(m_kpok, TQT_SIGNAL(clearStatusBar()), 
	  this,   TQT_SLOT(clearStatusBar()));
  connect(m_kpok, TQT_SIGNAL(statusBarMessage(TQString)), 
	  this,   TQT_SLOT(statusBarMessage(TQString)));

  statusBar()->addWidget(LHLabel, 0, true);
  m_kpok->updateLHLabel();
  //FIXME: LHLabel is shown twize until the bar is repainted!

  initTDEAction();
  readOptions();
}


PokerWindow::~PokerWindow()
{
}


// ----------------------------------------------------------------


void PokerWindow::initTDEAction()
{
  //Game
  KStdGameAction::gameNew(TQT_TQOBJECT(m_kpok), TQT_SLOT(newGame()), actionCollection());
  KStdGameAction::save(TQT_TQOBJECT(m_kpok), TQT_SLOT(saveGame()), actionCollection());
  KStdGameAction::quit(TQT_TQOBJECT(this), TQT_SLOT(close()), actionCollection());

  //Settings
  showMenubarAction =
    KStdAction::showMenubar(TQT_TQOBJECT(this), TQT_SLOT(toggleMenubar()), actionCollection());
   
  soundAction = new TDEToggleAction(i18n("Soun&d"), 0, TQT_TQOBJECT(m_kpok),
				  TQT_SLOT(toggleSound()), actionCollection(), "options_sound");
  if (m_kpok->getSound())
    m_kpok->toggleSound();
  blinkingAction = new TDEToggleAction(i18n("&Blinking Cards"), 0, TQT_TQOBJECT(m_kpok),
				     TQT_SLOT(toggleBlinking()), actionCollection(), "options_blinking");
  if (m_kpok->getBlinking())
    m_kpok->toggleBlinking();
  adjustAction = new TDEToggleAction(i18n("&Adjust Bet is Default"), 0,
				   TQT_TQOBJECT(m_kpok), TQT_SLOT(toggleAdjust()), actionCollection(), "options_adjust");
  if (m_kpok->getAdjust())
    m_kpok->toggleAdjust();

  showStatusbarAction =
    KStdAction::showStatusbar(TQT_TQOBJECT(this), TQT_SLOT(toggleStatusbar()), actionCollection());

  KStdAction::saveOptions(TQT_TQOBJECT(this),     TQT_SLOT(saveOptions()), actionCollection());
  KStdGameAction::carddecks(TQT_TQOBJECT(m_kpok), TQT_SLOT(slotCardDeck()), actionCollection());
  KStdAction::preferences(TQT_TQOBJECT(m_kpok),   TQT_SLOT(slotPreferences()), actionCollection());

  // Keyboard shortcuts.
  (void)new TDEAction(i18n("Draw"), TDEShortcut(TQt::Key_Return), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(drawClick()), actionCollection(), "draw");
  (void)new TDEAction(i18n("Exchange Card 1"), TDEShortcut(TQt::Key_1), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(exchangeCard1()), actionCollection(), "exchange_card_1");
  (void)new TDEAction(i18n("Exchange Card 2"), TDEShortcut(TQt::Key_2), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(exchangeCard2()), actionCollection(), "exchange_card_2");
  (void)new TDEAction(i18n("Exchange Card 3"), TDEShortcut(TQt::Key_3), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(exchangeCard3()), actionCollection(), "exchange_card_3");
  (void)new TDEAction(i18n("Exchange Card 4"), TDEShortcut(TQt::Key_4), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(exchangeCard4()), actionCollection(), "exchange_card_4");
  (void)new TDEAction(i18n("Exchange Card 5"), TDEShortcut(TQt::Key_5), TQT_TQOBJECT(m_kpok),
		    TQT_SLOT(exchangeCard5()), actionCollection(), "exchange_card_5");

  setupGUI( TDEMainWindow::Save | StatusBar | Keys | Create);
}


void PokerWindow::readOptions()
{
  TDEConfig* conf = kapp->config();
  conf->setGroup("General");

  if (m_kpok->getSound() != conf->readBoolEntry("Sound", true))
    soundAction->activate();

  if (m_kpok->getBlinking() != conf->readBoolEntry("Blinking", true))
    blinkingAction->activate();

  if (m_kpok->getAdjust() != conf->readBoolEntry("Adjust", true))
    adjustAction->activate();

  if ( showMenubarAction->isChecked() !=
       conf->readBoolEntry("ShowMenubar", true))
    showMenubarAction->activate();

  if ( showStatusbarAction->isChecked() !=
       conf->readBoolEntry("ShowStatusbar", true))
    showStatusbarAction->activate();
}


void PokerWindow::toggleMenubar()
{
  if (!menuBar()->isHidden())
    menuBar()->hide();
  else
    menuBar()->show();
}


void PokerWindow::toggleStatusbar()
{
  if (!statusBar()->isHidden())
    statusBar()->hide();
  else
    statusBar()->show();
}


/* Ask the user if he/she wants to save the game.  This virtual method
 * is called from the Quit TDEAction (I think).
 */

bool PokerWindow::queryClose()
{
  if (!m_kpok->isDirty())
    return true;

  // Only ask if the game is changed in some way.
  switch(KMessageBox::warningYesNoCancel(this, i18n("Do you want to save this game?"), TQString(), KStdGuiItem::save(), KStdGuiItem::dontSave())) {
  case KMessageBox::Yes :
    m_kpok->saveGame();
    return true;
  case KMessageBox::No :
    return true;
  default :
    return false;
  }
}


/* Show the hand or winner in the status bar at the lower right.
 *
 * Which is shown depends on wether this is a one player game or a two
 * player game.
 */

void PokerWindow::setHand(const TQString &newHand, bool lastHand)
{
  if (lastHand)
    LHLabel->setText(i18n("Last hand: ") + newHand);
  else
    LHLabel->setText(i18n("Last winner: ") + newHand);
  LHLabel->adjustSize();
}


void PokerWindow::showClickToHold(bool show)
{
  if (show) {
    statusBar()->clear();
    statusBar()->message(i18n("Click a card to hold it"));
    clickToHoldIsShown = true;
  } else if (clickToHoldIsShown) {
    statusBar()->clear();
    clickToHoldIsShown = false;
  }
}


void PokerWindow::statusBarMessage(TQString s)
{
  clearStatusBar();
  statusBar()->message(s);
  clickToHoldIsShown = false;
}


void PokerWindow::clearStatusBar()
{
  if (!clickToHoldIsShown)
    statusBar()->clear();
}


void PokerWindow::saveOptions()
{
  TDEConfig* conf = kapp->config();
  conf->setGroup("General");

  conf->writeEntry("Sound",         soundAction->isChecked());
  conf->writeEntry("Blinking",      blinkingAction->isChecked());
  conf->writeEntry("Adjust",        adjustAction->isChecked());
  conf->writeEntry("ShowMenubar",   showMenubarAction->isChecked());
  conf->writeEntry("ShowStatusbar", showStatusbarAction->isChecked());
}


bool PokerWindow::eventFilter(TQObject*, TQEvent* e)
{
  if (e->type() == TQEvent::MouseButtonPress) {

    if (((TQMouseEvent*)e)->button() == Qt::RightButton) {
      TQPopupMenu*  popup = (TQPopupMenu*) factory()->container("popup", this);
      if (popup)
	popup->popup(TQCursor::pos());
      return true;
    } else
      return false;
  }

  return false;
}

#include "top.moc"

