/* 
    This file is part of KDE Schema Parser

    Copyright (c) 2005 Tobias Koenig <tokoe@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
 */

#include <unistd.h>

#include <tqapplication.h>
#include <tqeventloop.h>
#include <tqfile.h>

#include <tdeio/job.h>
#include <tdetempfile.h>

#include "fileprovider.h"

using namespace Schema;

FileProvider::FileProvider()
  : TQObject( 0 ), mBlocked( false )
{
}

bool FileProvider::get( const TQString &url, TQString &target )
{
  if ( !mFileName.isEmpty() )
    cleanUp();

  if ( target.isEmpty() ) {
    KTempFile tmpFile;
    target = tmpFile.name();
    mFileName = target;
  }

  mData.truncate( 0 );

  tqDebug( "Downloading external schema '%s'", url.latin1() );

  TDEIO::TransferJob* job = TDEIO::get( KURL( url ), false, false );
  connect( job, TQ_SIGNAL( data( TDEIO::Job*, const TQByteArray& ) ),
           this, TQ_SLOT( slotData( TDEIO::Job*, const TQByteArray& ) ) );
  connect( job, TQ_SIGNAL( result( TDEIO::Job* ) ),
           this, TQ_SLOT( slotResult( TDEIO::Job* ) ) );

  mBlocked = true;
  while ( mBlocked ) {
    tqApp->eventLoop()->processEvents( TQEventLoop::ExcludeUserInput );
    usleep( 500 );
  }

  return true;
}

void FileProvider::cleanUp()
{
  ::unlink( TQFile::encodeName( mFileName ) );
  mFileName = TQString();
}

void FileProvider::slotData( TDEIO::Job*, const TQByteArray &data )
{
  unsigned int oldSize = mData.size();
  mData.resize( oldSize + data.size() );
  memcpy( mData.data() + oldSize, data.data(), data.size() );
}

void FileProvider::slotResult( TDEIO::Job *job )
{
  if ( job->error() ) {
    tqDebug( "%s", job->errorText().latin1() );
    return;
  }

  TQFile file( mFileName );
  if ( !file.open( IO_WriteOnly ) ) {
    tqDebug( "Unable to create temporary file" );
    return;
  }

  tqDebug( "Download successful" );
  file.writeBlock( mData );
  file.close();

  mData.truncate( 0 );

  mBlocked = false;
}

#include "fileprovider.moc"
