/*
    This file is part of KAddressBook.
    Copyright (c) 2002 Mike Pilone <mpilone@slac.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#include <tqlayout.h>
#include <tqpopupmenu.h>

#include <tdeabc/addressbook.h>
#include <tdeabc/distributionlistdialog.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <kxmlguifactory.h>
#include <kxmlguiclient.h>

#include "core.h"
#include "searchmanager.h"

#include "kaddressbookview.h"

KAddressBookView::KAddressBookView( KAB::Core *core, TQWidget *parent,
                                    const char *name )
    : TQWidget( parent, name ), mCore( core ), mFieldList()
{
  initGUI();

  connect( mCore->searchManager(), TQ_SIGNAL( contactsUpdated() ),
           TQ_SLOT( updateView() ) );
}

KAddressBookView::~KAddressBookView()
{
  kdDebug(5720) << "KAddressBookView::~KAddressBookView: destroying - "
                << name() << endl;
}

void KAddressBookView::readConfig( TDEConfig *config )
{
  mFieldList = TDEABC::Field::restoreFields( config, "KABCFields" );

  if ( mFieldList.isEmpty() )
    mFieldList = TDEABC::Field::defaultFields();

  mDefaultFilterType = (DefaultFilterType)config->readNumEntry( "DefaultFilterType", 1 );
  mDefaultFilterName = config->readEntry( "DefaultFilterName" );
}

void KAddressBookView::writeConfig( TDEConfig* )
{
  // Most of writing the config is handled by the ConfigureViewDialog
}

TQString KAddressBookView::selectedEmails()
{
  bool first = true;
  TQString emailAddrs;
  const TQStringList uidList = selectedUids();
  TDEABC::Addressee addr;
  TQString email;

  TQStringList::ConstIterator it;
  for ( it = uidList.begin(); it != uidList.end(); ++it ) {
    addr = mCore->addressBook()->findByUid( *it );

    if ( !addr.isEmpty() ) {
      TQString m = TQString();

      if ( addr.emails().count() > 1 )
        m = TDEABC::EmailSelector::getEmail( addr.emails(), addr.preferredEmail(), this );

      email = addr.fullEmail( m );

      if ( !first )
        emailAddrs += ", ";
      else
        first = false;

      emailAddrs += email;
    }
  }

  return emailAddrs;
}

TDEABC::Addressee::List KAddressBookView::addressees()
{
  if ( mFilter.isEmpty() )
    return mCore->searchManager()->contacts();

  TDEABC::Addressee::List addresseeList;
  const TDEABC::Addressee::List contacts = mCore->searchManager()->contacts();

  TDEABC::Addressee::List::ConstIterator it;
  TDEABC::Addressee::List::ConstIterator contactsEnd( contacts.end() );
  for ( it = contacts.begin(); it != contactsEnd; ++it ) {
    if ( mFilter.filterAddressee( *it ) )
      addresseeList.append( *it );
  }

  return addresseeList;
}

void KAddressBookView::initGUI()
{
  // Create the layout
  TQVBoxLayout *layout = new TQVBoxLayout( this );

  // Add the view widget
  mViewWidget = new TQWidget( this );
  layout->addWidget( mViewWidget );
}

TDEABC::Field::List KAddressBookView::fields() const
{
  return mFieldList;
}

void KAddressBookView::setFilter( const Filter &filter )
{
  mFilter = filter;
}

KAddressBookView::DefaultFilterType KAddressBookView::defaultFilterType() const
{
  return mDefaultFilterType;
}

const TQString &KAddressBookView::defaultFilterName() const
{
  return mDefaultFilterName;
}

KAB::Core *KAddressBookView::core() const
{
  return mCore;
}

void KAddressBookView::popup( const TQPoint &point )
{
  if ( !mCore->guiClient() ) {
    kdWarning() << "No GUI client set!" << endl;
    return;
  }

  TQPopupMenu *menu = static_cast<TQPopupMenu*>( mCore->guiClient()->factory()->container( "RMBPopup",
                                               mCore->guiClient() ) );
  if ( menu )
    menu->popup( point );
}

TQWidget *KAddressBookView::viewWidget()
{
  return mViewWidget;
}

void KAddressBookView::updateView()
{
  const TQStringList uidList = selectedUids();

  refresh(); // This relists and deselects everything, in all views

  if ( !uidList.isEmpty() ) {
    // Keep previous selection
    TQStringList::ConstIterator it, uidListEnd( uidList.end() );
    for ( it = uidList.begin(); it != uidListEnd; ++it )
      setSelected( *it, true );

  } else {
    const TDEABC::Addressee::List contacts = mCore->searchManager()->contacts();
    if ( !contacts.isEmpty() )
      setFirstSelected( true );
    else
      emit selected( TQString() );
  }
}

ViewConfigureWidget *ViewFactory::configureWidget( TDEABC::AddressBook *ab,
                                                   TQWidget *parent,
                                                   const char *name )
{
  return new ViewConfigureWidget( ab, parent, name );
}

#include "kaddressbookview.moc"
