/*
 * noatun.cpp
 *
 * Copyright (C) 1999 Charles Samuels <charles@kde.org>
 */

#include "userinterface.h"
#include <noatun/playlist.h>
#include <noatun/stdaction.h>
#include <noatun/app.h>
#include <noatun/player.h>
#include <noatun/controls.h>
#include <noatun/effects.h>

#include <tqpushbutton.h>
#include <tqdragobject.h>
#include <tqlayout.h>
#include <tqtooltip.h>
#include <tqobjectlist.h>
#include <tqobjectdict.h>

#include <tdepopupmenu.h>
#include <kstatusbar.h>
#include <tdeglobal.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <kurldrag.h>
#include <tdefiledialog.h>
#include <tdeconfig.h>

MilkChocolate::MilkChocolate() : TQWidget(0,"NoatunUI"), UserInterface()
{
	setAcceptDrops(true);
	static const int buttonSize=32;

	{ // The buttons
		mBack=new TQPushButton(this);
		mBack->setFixedSize(buttonSize,buttonSize);
		mBack->setPixmap(BarIcon("noatunback"));
		connect(mBack, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(back()));
		TQToolTip::add(mBack,i18n("Back"));

		mStop=new TQPushButton(this);
		mStop->setFixedSize(buttonSize,buttonSize);
		mStop->setPixmap(BarIcon("noatunstop"));
		connect(mStop, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(stop()));
		TQToolTip::add(mStop, i18n("Stop"));

		mPlay=new TQPushButton(this);
		mPlay->setToggleButton(true);
		mPlay->setFixedSize(buttonSize,buttonSize);
		mPlay->setPixmap(BarIcon("noatunplay"));
		connect(mPlay, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(playpause()));
		TQToolTip::add(mPlay, i18n("Play"));

		mForward=new TQPushButton(this);
		mForward->setFixedSize(buttonSize,buttonSize);
		mForward->setPixmap(BarIcon("noatunforward"));
		connect(mForward, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(forward()));
		TQToolTip::add(mForward, i18n("Forward"));

		mPlaylist=new TQPushButton(this);
		mPlaylist->setToggleButton(true);
		mPlaylist->setFixedSize(buttonSize,buttonSize);
		mPlaylist->setPixmap(BarIcon("noatunplaylist"));
		connect(mPlaylist, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(toggleListView()));
		TQToolTip::add(mPlaylist, i18n("Playlist"));

		mLoop=new TQPushButton(this);
		mLoop->setFixedSize(buttonSize,buttonSize);
		mLoop->setPixmap(BarIcon("noatunloopnone"));
		connect(mLoop, TQ_SIGNAL(clicked()), napp->player(), TQ_SLOT(loop()));
		TQToolTip::add(mLoop, i18n("Change loop style"));

		mPopup=new TQPushButton(this);
		mPopup->setFixedSize(buttonSize,buttonSize);
		mPopup->setPixmap(BarIcon("noatun"));
		connect(mPopup, TQ_SIGNAL(clicked()), TQ_SLOT(popup()));
//		TQToolTip::add(mRemoveCurrent, i18n("Remove current file from playlist"));

	}

	mVolume=new L33tSlider(0,100,10,0,TQt::Horizontal, this);
	mVolume->setValue(napp->player()->volume());
	mSeeker=new L33tSlider(0,1000,10,0,TQt::Horizontal, this);

	mStatusBar=new KStatusBar(this);


	TQGridLayout *l=new TQGridLayout(this);
	l->addWidget(mBack,0,0);
	l->addWidget(mStop,0,1);
	l->addWidget(mPlay,0,2);
	l->addWidget(mForward,0,3);
	l->addWidget(mPlaylist,0,4, TQt::AlignLeft);
	l->addWidget(mLoop,0,5);
	l->addWidget(mPopup,0,6);
	l->addColSpacing(4, buttonSize+8);

	l->addMultiCellWidget(mVolume,1,1,0,6);
	l->addMultiCellWidget(mSeeker,2,2,0,6);
	l->addMultiCellWidget(mStatusBar,3,3,0,6);

	statusBar()->message(i18n("No File Loaded"));
	statusBar()->insertItem("--:--/--:--", 1, 0, true);

	connect(napp, TQ_SIGNAL(hideYourself()), this, TQ_SLOT(hide()) );
	connect(napp, TQ_SIGNAL(showYourself()), this, TQ_SLOT(show()) );

	connect(napp->player(), TQ_SIGNAL(playing()), this, TQ_SLOT(slotPlaying()));
	connect(napp->player(), TQ_SIGNAL(stopped()), this, TQ_SLOT(slotStopped()));
	connect(napp->player(), TQ_SIGNAL(paused()), this, TQ_SLOT(slotPaused()));
	napp->player()->handleButtons();

	connect(napp->player(), TQ_SIGNAL(timeout()), this, TQ_SLOT(slotTimeout()));
	connect(napp->player(), TQ_SIGNAL(loopTypeChange(int)), this, TQ_SLOT(changeLoopType(int)));

//	if(seeker())
	{
		/* This skipToWrapper is needed to pass milliseconds to Player() as everybody
		 * below the GUI is based on milliseconds instead of some unprecise thingy
		 * like seconds or mille */
		connect(seeker(), TQ_SIGNAL(userChanged(int)), this, TQ_SLOT(skipToWrapper(int)));
		connect(this, TQ_SIGNAL(skipTo(int)), napp->player(), TQ_SLOT(skipTo(int)));
		connect(seeker(), TQ_SIGNAL(sliderMoved(int)), TQ_SLOT(sliderMoved(int)));
	}
	connect(mVolume, TQ_SIGNAL(sliderMoved(int)), napp->player(), TQ_SLOT(setVolume(int)));
	connect(mVolume, TQ_SIGNAL(userChanged(int)), napp->player(), TQ_SLOT(setVolume(int)));


	connect(napp->player(), TQ_SIGNAL(playlistShown()), TQ_SLOT(playlistShown()));
	connect(napp->player(), TQ_SIGNAL(playlistHidden()), TQ_SLOT(playlistHidden()));

	// Event Filter for the RMB
	for (TQPtrListIterator<TQObject> i(childrenListObject()); i.current(); ++i)
		(*i)->installEventFilter(this);

	setCaption("Noatun");
	setIcon(BarIcon("noatun"));
	show();

	// What it is now, stay, stay.. roll over, good boy!
	setFixedSize(minimumSize());
}

MilkChocolate::~MilkChocolate()
{
	// If cfg dialog is still open, delete it so it saves it's position
//	if(prefDlgExist)
//		delete prefDlg;
}

void MilkChocolate::closeEvent(TQCloseEvent*)
{
	unload();
}

void MilkChocolate::showEvent(TQShowEvent*e)
{
	TQWidget::showEvent(e);
}

void MilkChocolate::dragEnterEvent(TQDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(KURLDrag::canDecode(event));
}

void MilkChocolate::dropEvent(TQDropEvent *event)
{
	KURL::List uri;
	if (KURLDrag::decode(event, uri))
	{
		for (KURL::List::Iterator i = uri.begin(); i != uri.end(); ++i)
			napp->player()->openFile(*i, false);
	}
}

void MilkChocolate::mouseReleaseEvent(TQMouseEvent *e)
{
	TQWidget::mouseReleaseEvent(e);
	if (e->button()!=TQt::RightButton) return;
	NoatunStdAction::ContextMenu::showContextMenu();
}

void MilkChocolate::changeStatusbar(const TQString& text, const TQString &text2)
{
	if (!text2.isNull())
		statusBar()->changeItem(text2, 1);

	statusBar()->message(!text.isNull() ? text : napp->player()->current().title());
}

void MilkChocolate::changeCaption(const TQString& text)
{
	setCaption(text);
}

void MilkChocolate::popup()
{
	NoatunStdAction::ContextMenu::showContextMenu(
			mapToGlobal(mPopup->geometry().bottomLeft())
		);
}

void MilkChocolate::slotPlaying()
{
//	connect(twinmodule, TQ_SIGNAL(windowAdded(WId)), view, TQ_SLOT(attemptReparent(WId)));
	changeStatusbar(napp->player()->current().title(), napp->player()->lengthString());
	mPlay->setOn(true);
	mStop->setEnabled(true);
	mPlay->setPixmap(BarIcon("noatunpause"));
}

void MilkChocolate::slotStopped()
{
	if (!napp->player()->current()) return;
	changeStatusbar(i18n("No File Loaded"), napp->player()->lengthString());
	mStop->setEnabled(false);
	mPlay->setOn(false);
	seeker()->setValue(0);
	mPlay->setPixmap(BarIcon("noatunplay"));
}

void MilkChocolate::slotPaused()
{
	mStop->setEnabled(true);
	mPlay->setOn(false);
	mPlay->setPixmap(BarIcon("noatunplay"));
}

void MilkChocolate::slotTimeout()
{
	mVolume->setValue(napp->player()->volume());

	if (!napp->player()->current()) return;
	if (static_cast<L33tSlider*>(seeker())->currentlyPressed()) return;
	if (seeker())
	{
		seeker()->setRange ( 0, (int)napp->player()->getLength()/1000 );
		seeker()->setValue ( (int)napp->player()->getTime()/1000 );
	}
	changeStatusbar(0, napp->player()->lengthString());
}

void MilkChocolate::sliderMoved(int seconds)
{
	if (napp->player()->current())
		changeStatusbar(0, napp->player()->lengthString(seconds*1000));
}

void MilkChocolate::skipToWrapper(int second)
{
	emit skipTo((long)(second*1000));
}

void MilkChocolate::changeLoopType(int t)
{
	static const int time=2000;
	switch (t)
	{
	case(Player::None):
		statusBar()->message(i18n("No looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopnone"));
		break;
	case(Player::Song):
		statusBar()->message(i18n("Song looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopsong"));
		break;
	case(Player::Playlist):
		statusBar()->message(i18n("Playlist looping"), time);
		mLoop->setPixmap(BarIcon("noatunloopplaylist"));
		break;
	case(Player::Random):
		statusBar()->message(i18n("Random play"), time);
		mLoop->setPixmap(BarIcon("noatunlooprandom"));
	}

}

bool MilkChocolate::eventFilter(TQObject *o, TQEvent *e)
{
	if ((e->type() == TQEvent::MouseButtonRelease)
		&& ((static_cast<TQMouseEvent*>(e))->button()==TQt::RightButton))
	{
		mouseReleaseEvent(static_cast<TQMouseEvent*>(e));
		return true;
	}

	if (e->type() == TQEvent::Wheel)
	{
		wheelEvent(static_cast<TQWheelEvent*>(e));
		return true;
	}
	return TQWidget::eventFilter(o, e);
}

void MilkChocolate::playlistShown()
{
	mPlaylist->setOn(true);
}

void MilkChocolate::playlistHidden()
{
	mPlaylist->setOn(false);
}

void MilkChocolate::wheelEvent(TQWheelEvent *e)
{
	int delta=e->delta();
	mVolume->setValue(mVolume->value()+(delta/120));
	napp->player()->setVolume(mVolume->value()+(delta/120));
}

#include "userinterface.moc"
