/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2001-2002 by Andreas Zehender
    email                : zehender@kde.org
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#ifndef PMTEXTUREMAP_H
#define PMTEXTUREMAP_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pmtexturebase.h"
#include "pmvector.h"

#include <tqvaluelist.h>

/**
 * Base class for color, pigment, texture and normal maps
 */

class PMTextureMapBase : public PMTextureBase
{
   typedef PMTextureBase Base;
public:
   /**
    * Creates a PMTextureMapBase
    */
   PMTextureMapBase( PMPart* part );
   /**
    * Copy constructor
    */
   PMTextureMapBase( const PMTextureMapBase& b );
   /**
    * deletes the PMTextureMapBase
    */
   virtual ~PMTextureMapBase( );

   /** */
   virtual bool dataChangeOnInsertRemove( ) const { return true; }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual void childAdded( PMObject* o );
   /** */
   virtual bool takeChild( PMObject* o );

   /** */
   virtual void serialize( TQDomElement& e, TQDomDocument& doc ) const;
   /** */
   virtual void readAttributes( const PMXMLHelper& h );

   /**
    * Returns the map object type
    */
   virtual TQString mapType( ) const = 0;

   /**
    * Returns a new @ref PMTextureMapEdit
    */
   virtual PMDialogEditBase* editWidget( TQWidget* parent ) const;

   /** */
   virtual void createMemento( );
   /** */
   virtual void restoreMemento( PMMemento* s );

   /**
    * Returns the map values
    */
   TQValueList<double> mapValues( ) const { return m_mapValues; }
   /**
    * Sets the map values
    */
   void setMapValues( const TQValueList<double>& v );
   /**
    * Returns the map value for the object o
    */
   double mapValue( const PMObject* o ) const;
   /**
    * Returns the number of map entries
    */
   int mapEntries( ) const { return m_mapValues.size( ); }

private:
   PMObject* nextMapEntry( PMObject* o );
   void stringToValues( const TQString& str );
   TQString valuesToString( ) const;
   TQValueList<double>::Iterator valueForChild( PMObject* o );

   /**
    * IDs for @ref PMMementoData
    */
   // enum PMTextureMapBaseMementoID { };

   /**
    * list of map values
    */
   TQValueList<double> m_mapValues;
   /**
    * removed map values
    */
   TQValueList<double> m_removedValues;

   static PMMetaObject* s_pMetaObject;
};


/**
 * Class for texture maps
 */

class PMTextureMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a texture map
    */
   PMTextureMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMTextureMap( const PMTextureMap& m );
   /**
    * Deletes the texture map
    */
   virtual ~PMTextureMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMTextureMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "Texture" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmtexturemap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};

/**
 * Class for pigment maps
 */

class PMPigmentMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a pigment map
    */
   PMPigmentMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMPigmentMap( const PMPigmentMap& m );
   /**
    * Deletes the pigment map
    */
   virtual ~PMPigmentMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMPigmentMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "Pigment" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmpigmentmap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};

/**
 * Class for color maps
 */

class PMColorMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a color map
    */
   PMColorMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMColorMap( const PMColorMap& m );
   /**
    * Deletes the color map
    */
   virtual ~PMColorMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMColorMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "SolidColor" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmcolormap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};


/**
 * Class for normal maps
 */

class PMNormalMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a normal map
    */
   PMNormalMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMNormalMap( const PMNormalMap& m );
   /**
    * Deletes the normal map
    */
   virtual ~PMNormalMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMNormalMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "Normal" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmnormalmap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};

/**
 * Class for slope maps
 */

class PMSlopeMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a slope map
    */
   PMSlopeMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMSlopeMap( const PMSlopeMap& m );
   /**
    * Deletes the slope map
    */
   virtual ~PMSlopeMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMSlopeMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "Slope" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmslopemap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};

/**
 * Class for density maps
 */

class PMDensityMap : public PMTextureMapBase
{
public:
   typedef PMTextureMapBase Base;
   /**
    * Creates a density map
    */
   PMDensityMap( PMPart* part );
   /**
    * Copy constructor
    */
   PMDensityMap( const PMDensityMap& m );
   /**
    * Deletes the density map
    */
   virtual ~PMDensityMap( );

   /** */
   virtual PMObject* copy( ) const { return new PMDensityMap( *this ); }
   /** */
   virtual TQString description( ) const;
   /** */
   virtual TQString mapType( ) const { return TQString( "Density" ); }

   /** */
   virtual PMMetaObject* metaObject( ) const;
   /** */
   virtual void cleanUp( ) const;

   /** */
   virtual TQString pixmap( ) const { return TQString( "pmdensitymap" ); }

private:
   static PMMetaObject* s_pMetaObject;
};

#endif
