#ifndef GAME_H
#define GAME_H

#include <kdebug.h>
#include <tdelocale.h>
#include <kpixmap.h>
#include <arts/kplayobject.h>
#include <arts/kartsserver.h>
#include <arts/kartsdispatcher.h>

#include <math.h>

#include <tqcanvas.h>
#include <tqpainter.h>
#include <tqcolor.h>
#include <tqframe.h>
#include <tqlayout.h>
#include <tqmap.h>
#include <tqpen.h>
#include <tqpoint.h>
#include <tqpointarray.h>
#include <tqrect.h>
#include <tqstringlist.h>
#include <tqvaluelist.h>

#include "object.h"
#include "config.h"
#include "canvasitem.h"
#include "ball.h"
#include "statedb.h"
#include "rtti.h"
#include <tdemacros.h>

class TQLabel;
class TQSlider;
class TQCheckBox;
class TQTimer;
class TQKeyEvent;
class TQMouseEvent;
class TQPainter;
class TDEConfig;
class KPrinter;
class KolfGame;

enum Direction { D_Left, D_Right, Forwards, Backwards };
enum Amount { Amount_Less, Amount_Normal, Amount_More };
enum HoleResult { Result_Holed, Result_Miss, Result_LipOut };

class Player;

class BallStateInfo
{
public:
	void saveState(TDEConfig *cfg);
	void loadState(TDEConfig *cfg);

	int id;
	TQPoint spot;
	BallState state;
	bool beginningOfHole;
	int score;
};
class BallStateList : public TQValueList<BallStateInfo>
{
public:
	int hole;
	int player;
	bool canUndo;
	Vector vector;
};

class Player
{
public:
	Player() : m_ball(new Ball(0)) {};
	Ball *ball() const { return m_ball; }
	void setBall(Ball *ball) { m_ball = ball; }
	BallStateInfo stateInfo(int hole) const { BallStateInfo ret; ret.spot = TQPoint(m_ball->x(), m_ball->y()); ret.state = m_ball->curState(); ret.score = score(hole); ret.beginningOfHole = m_ball->beginningOfHole(); ret.id = m_id; return ret; }

	TQValueList<int> scores() const { return m_scores; }
	void setScores(const TQValueList<int> &newScores) { m_scores = newScores; }
	int score(int hole) const { return (*m_scores.at(hole - 1)); }
	int lastScore() const { return m_scores.last(); }
	int firstScore() const { return m_scores.first(); }

	void addStrokeToHole(int hole) { (*m_scores.at(hole - 1))++; }
	void setScoreForHole(int score, int hole) { (*m_scores.at(hole - 1)) = score; }
	void subtractStrokeFromHole(int hole) { (*m_scores.at(hole - 1))--; }
	void resetScore(int hole) { (*m_scores.at(hole - 1)) = 0; }
	void addHole() { m_scores.append(0); }
	unsigned int numHoles() const { return m_scores.count(); }

	TQString name() const { return m_name; }
	void setName(const TQString &name) { m_name = name; m_ball->setName(name); }

	void setId(int id) { m_id = id; }
	int id() const { return m_id; }

private:
	Ball *m_ball;
	TQValueList<int> m_scores;
	TQString m_name;
	int m_id;
};
typedef TQValueList<Player> PlayerList;

class Arrow : public TQCanvasLine
{
public:
	Arrow(TQCanvas *canvas);
	void setAngle(double newAngle) { m_angle = newAngle; }
	double angle() const { return m_angle; }
	void setLength(double newLength) { m_length = newLength; }
	double length() const { return m_length; }
	void setReversed(bool yes) { m_reversed = yes; }
	bool reversed() const { return m_reversed; }
	virtual void setVisible(bool);
	virtual void setPen(TQPen p);
	void aboutToDie();
	virtual void moveBy(double, double);
	void updateSelf();
	virtual void setZ(double newz);

private:
	double m_angle;
	double m_length;
	bool m_reversed;
	TQCanvasLine *line1;
	TQCanvasLine *line2;
};

class RectPoint;
class RectItem
{
public:
	virtual void newSize(int /*width*/, int /*height*/) {};
};

class RectPoint : public TQCanvasEllipse, public CanvasItem
{
public:
	RectPoint(TQColor color, RectItem *, TQCanvas *canvas);
	void dontMove() { dontmove = true; }
	virtual void moveBy(double dx, double dy);
	virtual Config *config(TQWidget *parent);
	virtual bool deleteable() const { return false; }
	virtual bool cornerResize() const { return true; }
	virtual CanvasItem *itemToDelete() { return dynamic_cast<CanvasItem *>(rect); }
	void setSizeFactor(double newFactor) { m_sizeFactor = newFactor; }

protected:
	RectItem *rect;
	double m_sizeFactor;

private:
	bool dontmove;
};

class Ellipse : public TQCanvasEllipse, public CanvasItem, public RectItem
{
public:
	Ellipse(TQCanvas *canvas);
	virtual void advance(int phase);

	int changeEvery() const { return m_changeEvery; }
	void setChangeEvery(int news) { m_changeEvery = news; }
	bool changeEnabled() const { return m_changeEnabled; }
	void setChangeEnabled(bool news);

	virtual void aboutToDie();
	virtual void aboutToSave();
	virtual void savingDone();

	virtual TQPtrList<TQCanvasItem> moveableItems() const;

	virtual void newSize(int width, int height);
	virtual void moveBy(double dx, double dy);

	virtual void editModeChanged(bool changed);

	virtual void save(TDEConfig *cfg);
	virtual void load(TDEConfig *cfg);

	virtual Config *config(TQWidget *parent);

protected:
	RectPoint *point;
	int m_changeEvery;
	bool m_changeEnabled;

private:
	int count;
	bool dontHide;
};
class EllipseConfig : public Config
{
	TQ_OBJECT
  

public:
	EllipseConfig(Ellipse *ellipse, TQWidget *);

private slots:
	void value1Changed(int news);
	void value2Changed(int news);
	void check1Changed(bool on);
	void check2Changed(bool on);

protected:
	TQVBoxLayout *m_vlayout;

private:
	TQLabel *slow1;
	TQLabel *fast1;
	TQLabel *slow2;
	TQLabel *fast2;
	TQSlider *slider1;
	TQSlider *slider2;
	Ellipse *ellipse;
};

class Puddle : public Ellipse
{
public:
	Puddle(TQCanvas *canvas);
	virtual bool collision(Ball *ball, long int id);
	virtual int rtti() const { return Rtti_DontPlaceOn; }
};
class PuddleObj : public Object
{
public:
	PuddleObj() { m_name = i18n("Puddle"); m__name = "puddle"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Puddle(canvas); }
};

class Sand : public Ellipse
{
public:
	Sand(TQCanvas *canvas);
	virtual bool collision(Ball *ball, long int id);
};
class SandObj : public Object
{
public:
	SandObj() { m_name = i18n("Sand"); m__name = "sand"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Sand(canvas); }
};

class Inside : public TQCanvasEllipse, public CanvasItem
{
public:
	Inside(CanvasItem *item, TQCanvas *canvas) : TQCanvasEllipse(canvas) { this->item = item; }
	virtual bool collision(Ball *ball, long int id) { return item->collision(ball, id); }

protected:
	CanvasItem *item;
};

class Bumper : public TQCanvasEllipse, public CanvasItem
{
public:
	Bumper(TQCanvas *canvas);

	virtual void advance(int phase);
	virtual void aboutToDie();
	virtual void moveBy(double dx, double dy);
	virtual void editModeChanged(bool changed);

	virtual bool collision(Ball *ball, long int id);

protected:
	TQColor firstColor;
	TQColor secondColor;
	Inside *inside;

private:
	int count;
};
class BumperObj : public Object
{
public:
	BumperObj() { m_name = i18n("Bumper"); m__name = "bumper"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Bumper(canvas); }
};

class Hole : public TQCanvasEllipse, public CanvasItem
{
public:
	Hole(TQColor color, TQCanvas *canvas);
	virtual bool place(Ball * /*ball*/, bool /*wasCenter*/) { return true; };

	virtual bool collision(Ball *ball, long int id);

protected:
	virtual HoleResult result(const TQPoint, double, bool *wasCenter);
};

class Cup : public Hole
{
public:
	Cup(TQCanvas *canvas);
	virtual bool place(Ball *ball, bool wasCenter);
	virtual void save(TDEConfig *cfg);
	virtual bool canBeMovedByOthers() const { return true; }
	virtual void draw(TQPainter &painter);

protected:
	TQPixmap pixmap;
};
class CupObj : public Object
{
public:
	CupObj() { m_name = i18n("Cup"); m__name = "cup"; m_addOnNewHole = true; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Cup(canvas); }
};

class BlackHole;
class BlackHoleConfig : public Config
{
	TQ_OBJECT
  

public:
	BlackHoleConfig(BlackHole *blackHole, TQWidget *parent);

private slots:
	void degChanged(int);
	void minChanged(double);
	void maxChanged(double);

private:
	BlackHole *blackHole;
};
class BlackHoleExit : public TQCanvasLine, public CanvasItem
{
public:
	BlackHoleExit(BlackHole *blackHole, TQCanvas *canvas);
	virtual int rtti() const { return Rtti_NoCollision; }
	virtual void aboutToDie();
	virtual void moveBy(double dx, double dy);
	virtual bool deleteable() const { return false; }
	virtual bool canBeMovedByOthers() const { return true; }
	virtual void editModeChanged(bool editing);
	virtual void setPen(TQPen p);
	virtual void showInfo();
	virtual void hideInfo();
	void updateArrowAngle();
	void updateArrowLength();
	virtual Config *config(TQWidget *parent);
	BlackHole *blackHole;

protected:
	Arrow *arrow;
};
class BlackHoleTimer : public TQObject
{
TQ_OBJECT
  

public:
	BlackHoleTimer(Ball *ball, double speed, int msec);

signals:
	void eject(Ball *ball, double speed);
	void halfway();

protected slots:
	void mySlot();
	void myMidSlot();

protected:
	double m_speed;
	Ball *m_ball;
};
class BlackHole : public TQObject, public Hole
{
TQ_OBJECT
  

public:
	BlackHole(TQCanvas *canvas);
	virtual bool canBeMovedByOthers() const { return true; }
	virtual void aboutToDie();
	virtual void showInfo();
	virtual void hideInfo();
	virtual bool place(Ball *ball, bool wasCenter);
	virtual void save(TDEConfig *cfg);
	virtual void load(TDEConfig *cfg);
	virtual Config *config(TQWidget *parent) { return new BlackHoleConfig(this, parent); }
	virtual TQPtrList<TQCanvasItem> moveableItems() const;
	double minSpeed() const { return m_minSpeed; }
	double maxSpeed() const { return m_maxSpeed; }
	void setMinSpeed(double news) { m_minSpeed = news; exitItem->updateArrowLength(); }
	void setMaxSpeed(double news) { m_maxSpeed = news; exitItem->updateArrowLength(); }

	int curExitDeg() const { return exitDeg; }
	void setExitDeg(int newdeg);

	virtual void editModeChanged(bool editing) { exitItem->editModeChanged(editing); }
	void updateInfo();

	virtual void shotStarted() { runs = 0; };

	virtual void moveBy(double dx, double dy);

public slots:
	void eject(Ball *ball, double speed);
	void halfway();

protected:
	int exitDeg;
	BlackHoleExit *exitItem;
	double m_minSpeed;
	double m_maxSpeed;

private:
	int runs;
	TQCanvasLine *infoLine;
	TQCanvasEllipse *outside;
	void finishMe();
};
class BlackHoleObj : public Object
{
public:
	BlackHoleObj() { m_name = i18n("Black Hole"); m__name = "blackhole"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new BlackHole(canvas); }
};

class WallPoint;
class Wall : public TQCanvasLine, public CanvasItem
{
public:
	Wall(TQCanvas *canvas);
	virtual void aboutToDie();
	double dampening;

	void setAlwaysShow(bool yes);
	virtual void setZ(double newz);
	virtual void setPen(TQPen p);
	virtual bool collision(Ball *ball, long int id);
	virtual void save(TDEConfig *cfg);
	virtual void load(TDEConfig *cfg);
	virtual void selectedItem(TQCanvasItem *item);
	virtual void editModeChanged(bool changed);
	virtual void moveBy(double dx, double dy);
	virtual void setVelocity(double vx, double vy);
	virtual void clean();

	// must reimp because we gotta move the end items,
	// and we do that in moveBy()
	virtual void setPoints(int xa, int ya, int xb, int yb) { TQCanvasLine::setPoints(xa, ya, xb, yb); moveBy(0, 0); }

	virtual int rtti() const { return Rtti_Wall; }
	virtual TQPtrList<TQCanvasItem> moveableItems() const;
	virtual void setGame(KolfGame *game);
	virtual void setVisible(bool);

	virtual TQPointArray areaPoints() const;

protected:
	WallPoint *startItem;
	WallPoint *endItem;
	bool editing;

private:
	long int lastId;

	friend class WallPoint;
};
class WallPoint : public TQCanvasEllipse, public CanvasItem
{
public:
	WallPoint(bool start, Wall *wall, TQCanvas *canvas);
	void setAlwaysShow(bool yes) { alwaysShow = yes; updateVisible(); }
	virtual void editModeChanged(bool changed);
	virtual void moveBy(double dx, double dy);
	virtual int rtti() const { return Rtti_WallPoint; }
	virtual bool deleteable() const { return false; }
	virtual bool collision(Ball *ball, long int id);
	virtual CanvasItem *itemToDelete() { return wall; }
	virtual void clean();
	virtual Config *config(TQWidget *parent) { return wall->config(parent); }
	void dontMove() { dontmove = true; };
	void updateVisible();

	Wall *parentWall() { return wall; }

protected:
	Wall *wall;
	bool editing;
	bool visible;

private:
	bool alwaysShow;
	bool start;
	bool dontmove;
	long int lastId;

	friend class Wall;
};
class WallObj : public Object
{
public:
	WallObj() { m_name = i18n("Wall"); m__name = "wall"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Wall(canvas); }
};

class Putter : public TQCanvasLine, public CanvasItem
{
public:
	Putter(TQCanvas *canvas);
	void go(Direction, Amount amount = Amount_Normal);
	void setOrigin(int x, int y);
	int curLen() const { return len; }
	double curAngle() const { return angle; }
	int curDeg() const { return rad2deg(angle); }
	virtual void showInfo();
	virtual void hideInfo();
	void setAngle(double news) { angle = news; finishMe(); }
	void setDeg(int news) { angle = deg2rad(news); finishMe(); }
	double curMaxAngle() const { return maxAngle; }
	virtual int rtti() const { return Rtti_Putter; }
	virtual void setVisible(bool yes);
	void saveAngle(Ball *ball) { angleMap[ball] = angle; }
	void setAngle(Ball *ball);
	void resetAngles() { angleMap.clear(); setZ(999999); }
	virtual bool canBeMovedByOthers() const { return true; }
	virtual void moveBy(double dx, double dy);
	void setShowGuideLine(bool yes);

private:
	TQPoint midPoint;
	double maxAngle;
	double angle;
	double oneDegree;
	TQMap<Ball *, double> angleMap;
	int len;
	void finishMe();
	int putterWidth;
	TQCanvasLine *guideLine;
	bool m_showGuideLine;
};

class Bridge;
class BridgeConfig : public Config
{
	TQ_OBJECT
  

public:
	BridgeConfig(Bridge *bridge, TQWidget *);

protected slots:
	void topWallChanged(bool);
	void botWallChanged(bool);
	void leftWallChanged(bool);
	void rightWallChanged(bool);

protected:
	TQVBoxLayout *m_vlayout;
	TQCheckBox *top;
	TQCheckBox *bot;
	TQCheckBox *left;
	TQCheckBox *right;

private:
	Bridge *bridge;
};
class Bridge : public TQCanvasRectangle, public CanvasItem, public RectItem
{
public:
	Bridge(TQRect rect, TQCanvas *canvas);
	virtual bool collision(Ball *ball, long int id);
	virtual void aboutToDie();
	virtual void editModeChanged(bool changed);
	virtual void moveBy(double dx, double dy);
	virtual void load(TDEConfig *cfg);
	virtual void save(TDEConfig *cfg);
	virtual bool vStrut() const { return true; }
	void doLoad(TDEConfig *cfg);
	void doSave(TDEConfig *cfg);
	virtual void newSize(int width, int height);
	virtual void setGame(KolfGame *game);
	virtual Config *config(TQWidget *parent) { return new BridgeConfig(this, parent); }
	void setSize(int width, int height);
	virtual TQPtrList<TQCanvasItem> moveableItems() const;

	void setWallColor(TQColor color);
	TQPen wallPen() const { return topWall->pen(); }

	double wallZ() const { return topWall->z(); }
	void setWallZ(double);

	void setTopWallVisible(bool yes) { topWall->setVisible(yes); }
	void setBotWallVisible(bool yes) { botWall->setVisible(yes); }
	void setLeftWallVisible(bool yes) { leftWall->setVisible(yes); }
	void setRightWallVisible(bool yes) { rightWall->setVisible(yes); }
	bool topWallVisible() const { return topWall->isVisible(); }
	bool botWallVisible() const { return botWall->isVisible(); }
	bool leftWallVisible() const { return leftWall->isVisible(); }
	bool rightWallVisible() const { return rightWall->isVisible(); }

protected:
	Wall *topWall;
	Wall *botWall;
	Wall *leftWall;
	Wall *rightWall;
	RectPoint *point;
};
class BridgeObj : public Object
{
public:
	BridgeObj() { m_name = i18n("Bridge"); m__name = "bridge"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Bridge(TQRect(0, 0, 80, 40), canvas); }
};

class Sign;
class SignConfig : public BridgeConfig
{
	TQ_OBJECT
  

public:
	SignConfig(Sign *sign, TQWidget *parent);

private slots:
	void textChanged(const TQString &);

private:
	Sign *sign;
};
class Sign : public Bridge
{
public:
	Sign(TQCanvas *canvas);
	void setText(const TQString &text);
	TQString text() const { return m_text; }
	virtual void draw(TQPainter &painter);
	virtual bool vStrut() const { return false; }
	virtual Config *config(TQWidget *parent) { return new SignConfig(this, parent); }
	virtual void save(TDEConfig *cfg);
	virtual void load(TDEConfig *cfg);

protected:
	TQString m_text;
	TQString m_untranslatedText;
};
class SignObj : public Object
{
public:
	SignObj() { m_name = i18n("Sign"); m__name = "sign"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Sign(canvas); }
};

class Windmill;
class WindmillGuard : public Wall
{
public:
	WindmillGuard(TQCanvas *canvas) : Wall(canvas) {};
	void setBetween(int newmin, int newmax) { max = newmax; min = newmin; }
	virtual void advance(int phase);

protected:
	int max;
	int min;
};
class WindmillConfig : public BridgeConfig
{
	TQ_OBJECT
  

public:
	WindmillConfig(Windmill *windmill, TQWidget *parent);

private slots:
	void speedChanged(int news);
	void endChanged(bool yes);

private:
	Windmill *windmill;
};
class Windmill : public Bridge
{
public:
	Windmill(TQRect rect, TQCanvas *canvas);
	virtual void aboutToDie();
	virtual void newSize(int width, int height);
	virtual void save(TDEConfig *cfg);
	virtual void load(TDEConfig *cfg);
	virtual void setGame(KolfGame *game);
	virtual Config *config(TQWidget *parent) { return new WindmillConfig(this, parent); }
	void setSize(int width, int height);
	virtual void moveBy(double dx, double dy);
	void setSpeed(int news);
	int curSpeed() const { return speed; }
	void setBottom(bool yes);
	bool bottom() const { return m_bottom; }

protected:
	WindmillGuard *guard;
	Wall *left;
	Wall *right;
	int speedfactor;
	int speed;
	bool m_bottom;
};
class WindmillObj : public Object
{
public:
	WindmillObj() { m_name = i18n("Windmill"); m__name = "windmill"; }
	virtual TQCanvasItem *newObject(TQCanvas *canvas) { return new Windmill(TQRect(0, 0, 80, 40), canvas); }
};

class HoleInfo;
class HoleConfig : public Config
{
	TQ_OBJECT
  

public:
	HoleConfig(HoleInfo *holeInfo, TQWidget *);

private slots:
	void authorChanged(const TQString &);
	void parChanged(int);
	void maxStrokesChanged(int);
	void nameChanged(const TQString &);
	void borderWallsChanged(bool);

private:
	HoleInfo *holeInfo;
};
class HoleInfo : public CanvasItem
{
public:
	HoleInfo() { m_lowestMaxStrokes = 4; }
	virtual ~HoleInfo() {}
	void setPar(int newpar) { m_par = newpar; }
	int par() const { return m_par; }
	void setMaxStrokes(int newMaxStrokes) { m_maxStrokes = newMaxStrokes; }
	int lowestMaxStrokes() const { return m_lowestMaxStrokes; }
	int maxStrokes() const { return m_maxStrokes; }
	bool hasMaxStrokes() const { return m_maxStrokes != m_lowestMaxStrokes; }
	void setAuthor(TQString newauthor) { m_author = newauthor; }
	TQString author() const { return m_author; }

	void setName(TQString newname) { m_name = newname; }
	void setUntranslatedName(TQString newname) { m_untranslatedName = newname; }
	TQString name() const { return m_name; }
	TQString untranslatedName() const { return m_untranslatedName; }

	virtual Config *config(TQWidget *parent) { return new HoleConfig(this, parent); }
	void borderWallsChanged(bool yes);
	bool borderWalls() const { return m_borderWalls; }

private:
	TQString m_author;
	TQString m_name;
	TQString m_untranslatedName;
	bool m_borderWalls;
	int m_par;
	int m_maxStrokes;
	int m_lowestMaxStrokes;
};

class StrokeCircle : public TQCanvasItem
{
public:
	StrokeCircle(TQCanvas *canvas);

	void setValue(double v);
	double value();
	void setMaxValue(double m);
	void setSize(int w, int h);
	void setThickness(int t);
	int thickness() const;
	int width() const;
	int height() const;
	virtual void draw(TQPainter &p);
	virtual TQRect boundingRect() const;
	virtual bool collidesWith(const TQCanvasItem*) const;
	virtual bool collidesWith(const TQCanvasSprite*, const TQCanvasPolygonalItem*, const TQCanvasRectangle*, const TQCanvasEllipse*, const TQCanvasText*) const;

private:
	double dvalue, dmax;
	int ithickness, iwidth, iheight;
};

struct TDE_EXPORT CourseInfo
{
	CourseInfo(const TQString &_name, const TQString &_untranslatedName, const TQString &_author, unsigned int _holes, unsigned int _par) { name = _name; untranslatedName = _untranslatedName, author = _author; holes = _holes; par = _par; }
	CourseInfo();

	TQString name;
	TQString untranslatedName;
	TQString author;
	unsigned int holes;
	unsigned int par;
};

class TDE_EXPORT KolfGame : public TQCanvasView
{
	TQ_OBJECT
  

public:
	KolfGame(ObjectList *obj, PlayerList *players, TQString filename, TQWidget *parent=0, const char *name=0 );
	~KolfGame();
	void setObjects(ObjectList *obj) { this->obj = obj; }
	void setFilename(const TQString &filename);
	TQString curFilename() const { return filename; }
	void emitLargestHole() { emit largestHole(highestHole); }
	TQCanvas *canvas() const { return course; }
	void removeItem(TQCanvasItem *item) { items.setAutoDelete(false); items.removeRef(item); }
	// returns whether it was a cancel
	bool askSave(bool);
	bool isEditing() const { return editing; }
	int currentHole() { return curHole; }
	void setStrict(bool yes) { strict = yes; }
	// returns true when you shouldn't do anything
	bool isPaused() const { return paused; }
	Ball *curBall() const { return (*curPlayer).ball(); }
	void updateMouse();
	void ballMoved();
	void updateHighlighter();
	void updateCourse() { course->update(); }
	TQCanvasItem *curSelectedItem() const { return selectedItem; }
	void setBorderWalls(bool);
	void setInPlay(bool yes) { inPlay = yes; }
	bool isInPlay() { return inPlay; }
	bool isInfoShowing() { return m_showInfo; }
	void stoppedBall();
	TQString courseName() const { return holeInfo.name(); }
	void hidePutter() { putter->setVisible(false); }
	void ignoreEvents(bool ignore) { m_ignoreEvents = ignore; }

	static void scoresFromSaved(TDEConfig *, PlayerList &players);
	static void courseInfo(CourseInfo &info, const TQString &filename);

public slots:
	void pause();
	void unPause();
	void save();
	void toggleEditMode();
	void setModified(bool mod = true);
	void addNewObject(Object *newObj);
	void addNewHole();
	void switchHole(int);
	void switchHole(const TQString &);
	void nextHole();
	void prevHole();
	void firstHole();
	void lastHole();
	void randHole();
	void playSound(TQString file, double vol = 1);
	void showInfoDlg(bool = false);
	void resetHole();
	void clearHole();
	void print(KPrinter &);
	void setShowInfo(bool yes);
	void toggleShowInfo();
	void updateShowInfo();
	void setUseMouse(bool yes) { m_useMouse = yes; }
	void setUseAdvancedPutting(bool yes);
	void setShowGuideLine(bool yes);
	void setSound(bool yes);
	void undoShot();
	void timeout();
	void saveScores(TDEConfig *);
	void startFirstHole(int hole);
	void sayWhosGoing();

signals:
	void holesDone();
	void newHole(int);
	void parChanged(int, int);
	void titleChanged(const TQString &);
	void largestHole(int);
	void scoreChanged(int, int, int);
	void newPlayersTurn(Player *);
	void playerHoled(Player *);
	void newSelectedItem(CanvasItem *);
	void checkEditing();
	void editingStarted();
	void editingEnded();
	void inPlayStart();
	void inPlayEnd();
	void maxStrokesReached(const TQString &);
	void currentHole(int);
	void modifiedChanged(bool);
	void newStatusText(const TQString &);

private slots:
	void shotDone();
	void holeDone();
	void startNextHole();
	void fastTimeout();
	void putterTimeout();
	void autoSaveTimeout();
	void addItemsToMoveableList(TQPtrList<TQCanvasItem>);
	void addItemToFastAdvancersList(CanvasItem *);
	void hideInfo();

	void emitMax();

protected:
	void mouseMoveEvent(TQMouseEvent *e);
	void mousePressEvent(TQMouseEvent *e);
	void mouseReleaseEvent(TQMouseEvent *e);
	void mouseDoubleClickEvent(TQMouseEvent *e);

	void handleMousePressEvent(TQMouseEvent *e);
	void handleMouseDoubleClickEvent(TQMouseEvent *e);
	void handleMouseMoveEvent(TQMouseEvent *e);
	void handleMouseReleaseEvent(TQMouseEvent *e);
	void keyPressEvent(TQKeyEvent *e);
	void keyReleaseEvent(TQKeyEvent *e);

	TQPoint viewportToViewport(const TQPoint &p);

private:
	TQCanvas *course;
	Putter *putter;
	PlayerList *players;
	PlayerList::Iterator curPlayer;
	Ball *whiteBall;
	StrokeCircle *strokeCircle;

	TQTimer *timer;
	TQTimer *autoSaveTimer;
	TQTimer *fastTimer;
	TQTimer *putterTimer;
	bool regAdv;

	ObjectList *obj;
	TQPtrList<TQCanvasItem> items;
	TQPtrList<TQCanvasItem> extraMoveable;
	TQPtrList<Wall> borderWalls;

	int timerMsec;
	int autoSaveMsec;
	int fastTimerMsec;
	int putterTimerMsec;

	void puttPress();
	void puttRelease();
	bool inPlay;
	bool putting;
	bool stroking;
	bool finishStroking;
	double strength;
	double maxStrength;
	int puttCount;
	bool puttReverse;

	int curHole;
	int highestHole;
	int curPar;

	int wallWidth;
	int height;
	int width;
	int margin;
	TQColor grass;

	int advancePeriod;

	int lastDelId;

	bool paused;

	TQString filename;
	bool recalcHighestHole;
	void openFile();

	bool strict;

	bool editing;
	TQPoint storedMousePos;
	bool moving;
	bool dragging;
	TQCanvasItem *movingItem;
	TQCanvasItem *selectedItem;
	TQCanvasRectangle *highlighter;

	// sound
	KArtsDispatcher artsDispatcher;
	KArtsServer artsServer;
	TQPtrList<KPlayObject> oldPlayObjects;
	bool m_sound;
	bool soundedOnce;
	TQString soundDir;

	bool m_ignoreEvents;

	HoleInfo holeInfo;
	TQCanvasText *infoText;
	void showInfo();
	StateDB stateDB;

	BallStateList ballStateList;
	void loadStateList();
	void recreateStateList();
	void addHoleInfo(BallStateList &list);

	bool dontAddStroke;

	bool addingNewHole;
	int scoreboardHoles;
	inline void resetHoleScores();

	bool m_showInfo;

	bool infoShown;

	TDEConfig *cfg;

	inline void addBorderWall(TQPoint start, TQPoint end);
	void shotStart();
	void startBall(const Vector &vector);

	bool modified;

	inline bool allPlayersDone();

	bool m_useMouse;
	bool m_useAdvancedPutting;

	TQPtrList<CanvasItem> fastAdvancers;
	bool fastAdvancedExist;

	TQString playerWhoMaxed;
};

#endif
