/***************************************************************************
*   Copyright (C) 2004 by Christoph Thielecke                             *
*   crissi99@gmx.de                                                       *
*                                                                         *
*   @description This class imports a openvpn configuration file                             *
*                                                                         *
*   @author Christoph Thielecke <crissi99@gmx.de>                    *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
//BEGIN INCLUDES
#include <tdemessagebox.h>
#include <tdeio/netaccess.h>
#include <tdetempfile.h>
#include <tdelocale.h>
#include <tdeconfig.h>
#include <kdialogbase.h>
#include <kcombobox.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <kpassdlg.h>
#include <tdestandarddirs.h>
#include <tqfile.h>
#include <tqurl.h>
#include <kurl.h>
#include <tqtextstream.h>
#include <tqcheckbox.h>
#include <string>
#include <tqprocess.h>

#include <iostream>

#include "importopenvpnprofiledialog.h"
#include "utils.h"
#include "importcertificatedialog.h"
//END INCLUDES

ImportOpenvpnProfileDialog::ImportOpenvpnProfileDialog ( KVpncConfig *config, TQWidget *parent, const TQString& caption, TQString file )
		: KDialogBase ( parent, "Import_Cisco_OpenVPN_profile", true, caption,
		                KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, true )
{
	if ( !file.isEmpty() )
		filename = file;
	else
		filename = "";
	Pkcs12CertFile = "";
	// 	CertPath="/etc/openvpn";
	importOk = false;
	this->config = config;

	main = new ImportOpenvpnProfileDialogBase ( this );
	setMainWidget ( main );
	//main->setMinimumSize(main->sizeHint());

	main->FilenameUrlrequester->setFilter ( "*.ovpn *.conf" );

	main->FilenameUrlrequester->setURL ( filename );

	main->OpenProfileManagerCheckBox->hide();

}

ImportOpenvpnProfileDialog::~ImportOpenvpnProfileDialog()
{
	delete main;
}

void ImportOpenvpnProfileDialog::accept()
{

	filename = main->FilenameUrlrequester->url();
	if ( !filename.isEmpty() )
	{
		if ( config->KvpncDebugLevel > 0 )
			config->appendLogEntry ( i18n ( "OpenVPN import: file: %1" ).arg ( filename ), config->debug );
		canAccept();
	}
	else
	{
		config->appendLogEntry ( i18n ( "OpenVPN import: file name empty" ), config->error );
		KMessageBox::sorry ( 0, i18n ( "File name can not be empty!" ), i18n ( "Empty File Name" ) );
	}
}

void ImportOpenvpnProfileDialog::canAccept()
{

	TQFile f ( filename );
	if ( !f.exists() )
	{
		KMessageBox::sorry ( 0, i18n ( "File not found." ), i18n ( "No File" ) );

		//		emit progress( 100 );
		return ;
	}

	//BEGIN Sample
	// sample config

	/*
	#
	# Sample OpenVPN configuration file for
	# home using SSL/TLS mode and RSA certificates/keys.
	#
	# '#' or ';' may be used to delimit comments.

	#######################################################
	# PATS INFORMATION:
	# No adaptations necessary in this file, unless you
	# give different names to your certificates or you
	# place them in different folders
	######################################################


	# Use a dynamic tun device.
	# For Linux 2.2 or non-Linux OSes,
	# you may want to use an explicit
	# unit number such as "tun1".
	# OpenVPN also supports virtual
	# ethernet "tap" devices.
	dev tap

	# Our OpenVPN peer is the PATS gateway.
	remote 143.129.70.202

	float

	# In SSL/TLS key exchange, Office will
	# assume server role and Home
	# will assume client role.
	tls-client

	# pull necessary here. This option will
	# enable the server to push commands to
	# the client
	pull

	route 10.0.0.0 255.255.0.0 vpn_gateway

	# Certificate Authority file
	ca cacert.pem

	# Our certificate/public key
	cert cert.pem

	# Our private key
	key key.pem

	# OpenVPN uses UDP port 5000 by default.
	port 5000

	# Downgrade UID and GID to
	# "nobody" after initialization
	# for extra security.
	; user nobody
	; group nobody

	# If you built OpenVPN with
	# LZO compression, uncomment
	# out the following line.
	# PREFFERED
	comp-lzo

	# Uncomment this section for a more reliable detection when a system
	# loses its connection.  For example, dial-ups or laptops that
	# travel to other locations.
	# This commands will be pushed through by the server
	; ping 15
	; ping-restart 45
	; ping-timer-rem
	; persist-tun
	; persist-key

	# Verbosity level.
	# 0 -- quiet except for fatal errors.
	# 1 -- mostly quiet, but display non-fatal network errors. PREFERRED
	# 3 -- medium output, good for normal operation.
	# 9 -- verbose, good for troubleshooting
	verb 1
	*/

	/* Example 2
	client
	dev tun

	proto udp
	remote 10.7.21.1 1194

	resolv-retry infinite
	ns-cert-type server
	nobind
	user nobody
	group nogroup
	persist-key
	persist-tun

	ca certs/richard-root-ca.crt
	cert certs/richard-openvpn-notebook-richard.crt
	key certs/richard-openvpn-notebook-richard.key

	cipher AES-128-CBC
	ping 15
	ping-restart 45
	ping-timer-rem
	persist-tun
	persist-key

	comp-lzo
	verb 3
	*/
	//END Sample

	TQFile OpenvpnConfigFile ( filename );
	TQTextStream stream ( &OpenvpnConfigFile );

	TQString importprefix = TQFileInfo ( filename ).dirPath();

	if ( config->KvpncDebugLevel > 2 )
		config->appendLogEntry ( i18n ( "OpenVPN import: import prefix: %1" ).arg ( importprefix ), config->debug );

	TQString certprefix = locateLocal ( "data", "kvpnc" );

	if ( OpenvpnConfigFile.open ( IO_ReadOnly ) )
	{
		VpnAccountData::ConnectionType ConnType = VpnAccountData::openvpn;
		acc = new VpnAccountData ( ConnType, "" );
		TQString profilename = Utils ( config ).removeSpecialCharsForFilename ( filename.section ( '/', -1, -1 ) ).section ( '.', -2, -2 );
		acc->setName ( profilename ); // /home/user/openvpn/CompanyABC.conf -> CompanyABC
		acc->setDescription ( i18n ( "Imported from %1" ).arg ( Utils ( config ).removeSpecialCharsForFilename ( filename.section ( '/', -1, -1 ) ) ) );
		acc->setLocalPort ( 1194 );
		acc->setAuthWithUsernameAndPassword ( false );
		acc->setRemoteNetMask ( "24" );
		acc->setAuthenticationAlgorithm ( "MD5" );
		acc->setDisableLzoCompression ( true );
		acc->setRemotePort ( 1194 );
		acc->setUseUdp( false );
		acc->setUseTlsAuth ( false );
		// 		acc->setUseConnectionStatusCheck( false );
		// 		acc->setDoReconnectAfterConnectionLost( false );

		// inline cert settings
		TQString CaCert="";
		TQString Cert="";
		TQString PrivateKey="";
		TQString StaticKey="";
		bool CaCertStartFound=false;
		bool CertStartFound=false;
		bool PrivateKeyStartFound=false;
		bool StaticKeyStartFound=false;
		bool TlsAuthStartFound=false;

		TQString line = "";
		while ( !stream.atEnd() )
		{
			TQString line_raw = stream.readLine();
			line = line_raw.simplifyWhiteSpace(); // line of text excluding '\n' and replace all white chars with one blank
			
			/* look for inline parts */
			if (CaCertStartFound)
			{
				if ( line.startsWith ( "</ca>") )
				{
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inlince ca end found." ), config->debug );
					
					TQString CaFilePath = certprefix+"/"+"openvpn_ca_"+profilename+".pem";
					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: ca file: %1" ).arg ( CaFilePath ), config->debug );
					
					TQFile CaFileHandle (CaFilePath);
					if( CaFileHandle.open(IO_WriteOnly))
					{
						TQTextStream ExportStream ( &CaFileHandle );
						ExportStream << CaCert;
						CaFileHandle.close();
						acc->setCaCertificate ( CaFilePath );
					}
					else
					{
						KMessageBox::error ( 0, i18n ( "Creating of \"%1\" has been failed!" ).arg ( CaFilePath ) );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: CA certificate: %1" ).arg ( acc->getCaCertificate() ), config->debug );
					CaCertStartFound = false;
				}
				else
				{
					CaCert += line_raw +"\n";
					continue;
				}
			}
			if (CertStartFound)
			{
				if ( line.startsWith ( "</cert>") )
				{
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inlince cert end found." ), config->debug );
					
					TQString CertFilePath = certprefix+"/"+"openvpn_cert_"+profilename+".pem";
					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: cert file: %1" ).arg ( CertFilePath ), config->debug );
					
					acc->setAuthType ( VpnAccountData::cert );
					acc->setCertPath (certprefix);
					
					TQFile CertFileHandle (CertFilePath);
					if( CertFileHandle.open(IO_WriteOnly))
					{
						TQTextStream ExportStream ( &CertFileHandle );
						ExportStream << Cert;
						CertFileHandle.close();
						acc->setX509Certificate ( CertFilePath );
					}
					else
					{
						KMessageBox::error ( 0, i18n ( "Creating of \"%1\" has been failed!" ).arg ( CertFilePath ) );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: certificate: %1" ).arg ( acc->getX509Certificate() ), config->debug );
					CertStartFound = false;
				}
				else
				{
					Cert += line_raw +"\n";
					continue;
				}
			}
			if (PrivateKeyStartFound)
			{
				if ( line.startsWith ( "</key>") )
				{
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inlince private key end found." ), config->debug );
					
					TQString PrivateKeyFilePath = certprefix+"/"+"openvpn_privatekey_"+profilename+".pem";
					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: private key file: %1" ).arg ( PrivateKeyFilePath ), config->debug );
					
					TQFile PrivateKeyFileHandle (PrivateKeyFilePath);
					if( PrivateKeyFileHandle.open(IO_WriteOnly))
					{
						TQTextStream ExportStream ( &PrivateKeyFileHandle );
						ExportStream << PrivateKey << "\n";
						PrivateKeyFileHandle.close();
						acc->setPrivateKey ( PrivateKeyFilePath );
					}
					else
					{
						KMessageBox::error ( 0, i18n ( "Creating of \"%1\" has been failed!" ).arg ( PrivateKeyFilePath ) );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: private key file: %1" ).arg ( acc->getPrivateKey()), config->debug );
					PrivateKeyStartFound = false;
				}
				else
				{
					PrivateKey += line_raw+"\n";
					continue;
				}
			}	
			if (TlsAuthStartFound)
			{
				if ( line.startsWith ( "</tls-auth>") )
				{
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inlince tls-auth end found." ), config->debug );
					
					TQString StaticKeyFilePath = certprefix+"/"+"openvpn_tlsauth_"+profilename+".pem";
					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: TLS auth file: %1" ).arg ( StaticKeyFilePath ), config->debug );
					
					TQFile StaticKeyFileHandle (StaticKeyFilePath);
					if( StaticKeyFileHandle.open(IO_WriteOnly))
					{
						TQTextStream ExportStream ( &StaticKeyFileHandle );
						ExportStream << StaticKey << "\n";
						StaticKeyFileHandle.close();
						acc->setTlsAuthFile ( StaticKeyFilePath );
						acc->setUseTlsAuth (true);
					}
					else
					{
						KMessageBox::error ( 0, i18n ( "Creating of \"%1\" has been failed!" ).arg ( StaticKeyFilePath ) );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: TLS auth file: %1" ).arg ( acc->getTlsAuthFile() ), config->debug );
					StaticKeyStartFound = false;
				}
				else
				{
					StaticKey += line_raw +"\n";
					continue;
				}
			}
			
			/* special useful information */
			if ( line.startsWith ( "# OVPN_ACCESS_SERVER_USERNAME=" ) )
			{
				int equalidx = line.find ( '=' );
				acc->setUserName(line.right (line.length()-equalidx-1));
				if ( config->KvpncDebugLevel > 0 )
					config->appendLogEntry ( i18n ( "OpenVPN import: username found (via special line): %1" ).arg ( acc->getUserName() ), config->debug );
			}
			line.remove ( '\"' );
			if ( !line.startsWith ( "#" ) && !line.startsWith ( ";" ) )
			{
				int commentidx = line.find ( '#' );
				if ( commentidx > -1 )
					line = line.left ( commentidx - 1 );

				if ( line.startsWith ( "remote " ) )
				{
					acc->setGateway ( line.section ( ' ', 1, 1 ) ); // IP address
					if ( !line.section ( ' ', 2, 2 ).isEmpty() )
					{
						acc->setUseRemotePort ( true );
						acc->setRemotePort ( line.section ( ' ', 2, 2 ).toInt() );

						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use userdefined remote port: %1" ).arg ( acc->getRemotePort() ), config->debug );
					}
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: gateway: %1" ).arg ( acc->getGateway() ), config->debug );
				}
				else if ( line.startsWith ( "dev " ) )
				{
					TQString Type = line.section ( ' ', 1, 1 ); // tun or tap
					if ( Type == "tun" || Type == "tap" )
					{
						acc->setTunnelDeviceType ( Type );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: tunnel device type: %1" ).arg ( acc->getTunnelDeviceType() ), config->debug );
					}
				}
				else if ( line.startsWith ( "route " ) )
				{
					// FIXME no fully supported yet!!!

					// FIXME made it possible to remember what type have to been set!
					TQString gateway_type = "";
					bool useGateway;
					acc->setUseAdditionalNetworkRoutes ( true );
					TQStringList AdditionalNetworkRoutes;
					// example entry:
					// <network>/<netmask>#<gateway>

					TQString network = line.section ( ' ', 1, 1 ) + "/" + line.section ( ' ', 2, 2 ); // ip and netmask

					TQString Part3 = line.section ( ' ', 3, 3 ); // possible gateway
					if ( Part3 != "vpn_gateway" && Part3 != "net_gateway" && Part3 != "remote_host" )
					{
						network.append ( "#" );
						network.append ( Part3 ); // gateway
						gateway_type = line.section ( ' ', 3, 3 );
						useGateway = true;
					}
					else
					{
						gateway_type = Part3;
					}

					AdditionalNetworkRoutes.append ( network );
					acc->setAdditionalNetworkRoutes ( AdditionalNetworkRoutes );

					if ( config->KvpncDebugLevel > 0 )
						if ( useGateway )
							config->appendLogEntry ( i18n ( "OpenVPN import: special route found: %1, type: %2" ).arg ( network ).arg ( gateway_type ), config->debug );
						else
							config->appendLogEntry ( i18n ( "OpenVPN import: special route found: %1 over %3, type: %2" ).arg ( network ).arg ( gateway_type ).arg ( Part3 ), config->debug );
				}
				else if ( line.startsWith ( "port " ) )
				{
					acc->setUseLocalPort ( true );
					acc->setLocalPort ( line.section ( ' ', 1, 1 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: local port specified: %1" ).arg ( acc->getLocalPort() ), config->debug );
				}
				else if ( line.startsWith ( "comp-lzo" ) )
				{
					acc->setDisableLzoCompression ( false );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use LZO compression" ), config->debug );
				}
				else if ( line.startsWith ( "cert " ) )
				{
					acc->setAuthType ( VpnAccountData::cert );
					TQString cert = line.section ( ' ', 1 ).stripWhiteSpace();

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: cert file: %1" ).arg ( importprefix+"/"+cert ), config->debug );

					if ( TQFile ( TQString ( importprefix+"/"+cert ) ).exists() )
					{
						// copy to ~/.trinity/share/apps/kvpnc/ and rename it
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: copy %1 to %2" ).arg ( TQString ( importprefix+"/"+cert ) ).arg ( TQString ( certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert ) ), config->debug );
						TQProcess CopyProcess ( this );
						CopyProcess.addArgument ( "cp" );
						CopyProcess.addArgument ( TQString ( importprefix+"/"+cert ) );
						CopyProcess.addArgument ( TQString ( certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert ) );
						if ( CopyProcess.start() )
						{
							while (CopyProcess.isRunning())
							{
								sleep(1);
							}
							acc->setX509Certificate ( TQString ( certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert ) );
						}
						else
						{
							config->appendLogEntry ( i18n ( "Could not start %1!" ).arg ( "cp" ), config->error );
							acc->setX509Certificate ( cert );
						}
					}
					else
					{
						// we use the given path
						acc->setX509Certificate ( cert );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: certificate: %1" ).arg ( acc->getX509Certificate() ), config->debug );
				}
				else if ( line.startsWith ( "key " ) )
				{

					TQString privatekey = line.section ( ' ', 1 ).stripWhiteSpace();

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: private key file for certificate: %1" ).arg ( importprefix+"/"+privatekey ), config->debug );

					if ( TQFile ( TQString ( importprefix+"/"+privatekey ) ).exists() )
					{
						// copy to ~/.trinity/share/apps/kvpnc/ and rename it
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: copy %1 to %2" ).arg ( TQString ( importprefix+"/"+privatekey ) ).arg ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) ), config->debug );
						TQProcess CopyProcess ( this );
						CopyProcess.addArgument ( "cp" );
						CopyProcess.addArgument ( TQString ( importprefix+"/"+privatekey ) );
						CopyProcess.addArgument ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) );
						if ( CopyProcess.start() )
						{
							while (CopyProcess.isRunning())
							{
								sleep(1);
							}
							acc->setPrivateKey ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) );
						}
						else
						{
							config->appendLogEntry ( i18n ( "Could not start %1!" ).arg ( "cp" ), config->error );
							acc->setPrivateKey ( privatekey );
						}
					}
					else
					{
						// we use the given path
						acc->setPrivateKey ( privatekey );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: private key for certificate in file: %1" ).arg ( acc->getPrivateKey() ), config->debug );
				}
				else if ( line.startsWith ( "secret " ) )
				{
					acc->setAuthType ( VpnAccountData::psk );

					TQString privatekey = line.section ( ' ', 1 ).stripWhiteSpace();

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: private key file: %1" ).arg ( importprefix+"/"+privatekey ), config->debug );

					if ( TQFile ( TQString ( importprefix+"/"+privatekey ) ).exists() )
					{
						// copy to ~/.trinity/share/apps/kvpnc/ and rename it
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: copy %1 to %2" ).arg ( TQString ( importprefix+"/"+privatekey ) ).arg ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) ), config->debug );
						TQProcess CopyProcess ( this );
						CopyProcess.addArgument ( "cp" );
						CopyProcess.addArgument ( TQString ( importprefix+"/"+privatekey ) );
						CopyProcess.addArgument ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) );
						if ( CopyProcess.start() )
						{
							while (CopyProcess.isRunning())
							{
								sleep(1);
							}
							acc->setPreSharedKeyFile ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey ) );
							acc->setSavePsk ( false );
						}
						else
						{
							config->appendLogEntry ( i18n ( "Could not start %1!" ).arg ( "cp" ), config->error );
						}
					}
					else
					{
						// we use the given path
						acc->setPreSharedKeyFile ( privatekey );
					}
					acc->setPskIsInFile ( true );

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: PSK in file: %1" ).arg ( acc->getPreSharedKeyFile() ), config->debug );
				}
				else if ( line.startsWith ( "ca " ) )
				{
					acc->setAuthType ( VpnAccountData::cert );

					TQString ca = line.section ( ' ', 1 );

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry ( i18n ( "OpenVPN import: ca file: %1" ).arg ( importprefix+"/"+ca ), config->debug );

					if ( TQFile ( TQString ( importprefix+"/"+ca ) ).exists() )
					{
						// copy to ~/.trinity/share/apps/kvpnc/ and rename it
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: copy %1 to %2" ).arg ( TQString ( importprefix+"/"+ca ) ).arg ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca ) ), config->debug );
						TQProcess CopyProcess ( this );
						CopyProcess.addArgument ( "cp" );
						CopyProcess.addArgument ( TQString ( importprefix+"/"+ca ) );
						CopyProcess.addArgument ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca ) );
						if ( CopyProcess.start() )
						{
							while (CopyProcess.isRunning())
							{
								sleep(1);
							}
							acc->setCaCertificate ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca ) );
						}
						else
						{
							config->appendLogEntry ( i18n ( "Could not start %1!" ).arg ( "cp" ), config->error );
							acc->setCaCertificate ( ca );
						}
					}
					else
					{
						// we use the given path
						acc->setCaCertificate ( ca );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: CA certificate: %1" ).arg ( acc->getCaCertificate() ), config->debug );
				}
				else if ( line.startsWith ( "<ca>") )
				{
					/* inline ca cert */
					CaCertStartFound=true;
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inline ca start found" ), config->debug );
				}
				else if ( line.startsWith ( "<cert>") )
				{
					/* inline cert */
					CertStartFound=true;
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inline cert start found" ), config->debug );
				}
				else if ( line.startsWith ( "<key>") )
				{
					/* inline private key */
					PrivateKeyStartFound=true;
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inline private key start found" ), config->debug );
				}
				else if ( line.startsWith ( "<tls-auth>") )
				{
					/* inline tls-auth cert */
					TlsAuthStartFound=true;
					if ( config->KvpncDebugLevel > 4 )
						config->appendLogEntry ( i18n ( "OpenVPN import: inline tls-auth start found" ), config->debug );
				}
				else if ( line.startsWith ( "proto " ) )
				{
					if ( line.section ( ' ', 1, 1 ) == "udp" )
					{
						acc->setUseUdp ( true );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use UDP" ), config->debug );
					}
					else if ( line.section ( ' ', 1, 1 ) == "tcp-client" )
					{
						acc->setUseUdp ( false );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: dont use UDP" ), config->debug );
					}

				}
				else if ( line.startsWith ( "cipher " ) )
				{
					acc->setUseUserdefiniedCipher ( true );
					acc->setUserdefiniedCipher ( line.section ( ' ', 1, 1 ) );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use userdefined cipher" ), config->debug );
				}
				else if ( line.startsWith ( "tls-auth " ) )
				{
					acc->setUseTlsAuth ( true );

					TQString tlsauth = line.section ( ' ', 1,1 );
					TQString tlsauthdirection = line.section ( ' ', 2,2 );

					if ( config->KvpncDebugLevel > 2 )
					{
						config->appendLogEntry ( i18n ( "OpenVPN import: TLS auth file: %1" ).arg ( importprefix+"/"+tlsauth ), config->debug );
						config->appendLogEntry ( i18n ( "OpenVPN import: TLS auth direction: %1" ).arg ( tlsauthdirection ), config->debug );
					}

					if ( TQFile ( TQString ( importprefix+"/"+tlsauth ) ).exists() )
					{
						// copy to ~/.trinity/share/apps/kvpnc/ and rename it
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: copy %1 to %2" ).arg ( TQString ( importprefix+"/"+tlsauth ) ).arg ( TQString ( certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+tlsauth ) ), config->debug );
						TQProcess CopyProcess ( this );
						CopyProcess.addArgument ( "cp" );
						CopyProcess.addArgument ( TQString ( importprefix+"/"+tlsauth ) );
						CopyProcess.addArgument ( TQString ( certprefix+"/"+"openvpn_tlsauth_"+profilename+"_"+tlsauth ) );
						if ( CopyProcess.start() )
						{
							while (CopyProcess.isRunning())
							{
								sleep(1);
							}
							acc->setTlsAuthFile ( TQString ( certprefix+"/"+"openvpn_tlsauth_"+profilename+"_"+tlsauth ) );
						}
						else
						{
							config->appendLogEntry ( i18n ( "Could not start %1!" ).arg ( "cp" ), config->error );
							acc->setTlsAuthFile ( tlsauth );
						}
					}
					else
					{
						// we use the given path
						acc->setTlsAuthFile ( tlsauth );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use TLS auth" ), config->debug );

					if (!tlsauthdirection.isEmpty() && ( tlsauthdirection=="0") || ( tlsauthdirection=="1") || ( tlsauthdirection=="none") )
					{
						acc->setAuthenticationDirection(tlsauthdirection);
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use TLS auth direction: %1" ).arg(tlsauthdirection), config->debug );
					}
				}
				else if ( line.startsWith ( "redirect-gateway" ) )
				{
					acc->setUseRedirectGateway ( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use redirect gateway" ), config->debug );
				}
				else if ( line.startsWith ( "ns-cert-type " ) )
				{
					acc->setUseNsCertType ( true );

					if ( line.section ( ' ', 1, 1 ) == "client" )
					{
						acc->setNsCertType ( "client" );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use NS certificate type: %1" ).arg ( "client" ), config->debug );
					}
					if ( line.section ( ' ', 1, 1 ) == "server" )
					{
						acc->setNsCertType ( "server" );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use NS certificate type: %1" ).arg ( "server" ), config->debug );
					}
				}
				else if ( line.startsWith ( "auth-user-pass" ) )
				{
					acc->setAuthWithUsernameAndPassword ( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: authenticate with username and password" ), config->debug );
				}
				else if ( line.startsWith ( "http-proxy " ) )
				{
					acc->setUseHttpProxy ( true );
					acc->setHttpProxy ( line.section ( ' ', 1, 1 ) );
					acc->setHttpProxyPort ( line.section ( ' ', 2, 2 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use HTTP proxy: %1, Port: %2" ).arg ( acc->getHttpProxy() ).arg ( TQString().setNum ( acc->getHttpProxyPort() ) ), config->debug );
				}
				else if ( line.startsWith ( "http-proxy-timeout " ) )
				{
					acc->setUseHttpProxy ( true );
					acc->setHttpProxyTimeout ( line.section ( ' ', 1, 1 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use NS certificate type: %1" ).arg ( TQString().setNum ( acc->getHttpProxyTimeout() ) ), config->debug );
				}
				else if ( line.startsWith ( "pkcs12 " ) )
				{
					acc->setAuthType ( VpnAccountData::cert );
					Pkcs12CertFile = line.section ( ' ', 1 );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: pkcs12 file found: %1" ).arg ( Pkcs12CertFile ), config->debug );
				}
				else if ( line.startsWith ( "cd " ) )
				{
					CertPath = line.section ( ' ', 1 );

					//
					// 					if (!Pkcs12CertFile.isEmpty() && !acc->getCaCertificate().startsWith(CertPath))
					// 						Pkcs12CertFile=CertPath+"/"+Pkcs12CertFile;


					acc->setCertPath ( CertPath );

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: certificate prefix found: %1" ).arg ( CertPath ), config->debug );
				}
				else if ( line.startsWith ( "tls-remote" ) )
				{
					if ( !line.section ( ' ', 1, 1 ).isEmpty() )
					{
						acc->setUseTlsRemoteHost ( true );
						acc->setTlsRemoteHost ( line.section ( ' ', 1, 1 ) );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: use tls remote host: %1" ).arg ( acc->getTlsRemoteHost() ), config->debug );
					}
					else
					{
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry ( i18n ( "OpenVPN import: tls remote host option was specified but server is empty, disabling TLS remote host." ), config->debug );
					}
				}
				else if ( line.startsWith ( "float" ) )
				{
					acc->setAllowIpAddressChangeOfPeer ( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: allow IP address change of peer (for DHCP)" ), config->debug );
				}
				else if ( line.startsWith ( "ifconfig " ) )
				{
					acc->setUseVirtualIP ( true );
					TQString VirtualIpLocal = line.section ( ' ', 1, 1 );
					TQString VirtualIpRemote = line.section ( ' ', 2, 2 );
					acc->setLocalVirtualIP ( VirtualIpLocal );
					acc->setRemoteVirtualIP ( VirtualIpRemote );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use virtual IP address. Local: %1, remote: %2" ).arg ( VirtualIpLocal ).arg ( VirtualIpRemote ), config->debug );
				}
				else if ( line.startsWith ( "client" ) )
				{
					// client -> tls-client + pull
					acc->setDisablePushFromServer ( false );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: Disable push from server: %1" ).arg ( i18n("no") ), config->debug );
				}
				else if ( line.startsWith ( "tls-client" ) )
				{

// 					if ( config->KvpncDebugLevel > 0 )
// 						config->appendLogEntry ( i18n ( "OpenVPN import: use virtual IP address. Local: %1, remote: %2" ).arg ( VirtualIpLocal ).arg ( VirtualIpRemote ), config->debug );
				}
				else if ( line.startsWith ( "pull" ) )
				{
					acc->setDisablePushFromServer ( false );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: Disable push from server: %1" ).arg ( i18n("no") ), config->debug );
				}
				else if ( line.startsWith ( "fragment " ) )
				{
					acc->setUseFragment(  true );
					int Fragment = TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setFragment( Fragment );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use fragmention, size: %1" ).arg ( TQString().setNum(Fragment) ), config->debug );
				}
				else if ( line.startsWith ( "inactive " ) )
				{
// 					acc->setUseVirtualIP ( true );
// 					TQString VirtualIpLocal = line.section ( ' ', 1, 1 );
// 					TQString VirtualIpRemote = line.section ( ' ', 2, 2 );
// 					acc->setLocalVirtualIP ( VirtualIpLocal );
// 					acc->setRemoteVirtualIP ( VirtualIpRemote );
// 					if ( config->KvpncDebugLevel > 0 )
// 						config->appendLogEntry ( i18n ( "OpenVPN import: use virtual IP address. Local: %1, remote: %2" ).arg ( VirtualIpLocal ).arg ( VirtualIpRemote ), config->debug );
				}
				else if ( line.startsWith ( "mssfix" ) )
				{
					acc->setUseMssfix ( true );
					int Mssfix = TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setMssfix( Mssfix );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use Mssfix, size: %1" ).arg ( TQString().setNum(Mssfix )), config->debug );
				}
				else if ( line.startsWith ( "nobind" ) )
				{
					acc->setDisableBind ( true );

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: Disable bind: %1" ).arg ( i18n("yes") ), config->debug );
				}
				else if ( line.startsWith ( "ping " ) )
				{
					acc->setUseTunnelPing( true );
					int TunnelPing = TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setTunnelPing ( TunnelPing);
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use ping through tunnel every: %1" ).arg ( TQString().setNum(TunnelPing) ), config->debug );
				}
				else if ( line.startsWith ( "ping-restart " ) )
				{
					acc->setUseTunnelPingRestart( true );
					int TunnelPingRestart = TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setTunnelPingRestart ( TunnelPingRestart );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: restart ping through tunnel fails after: %1" ).arg ( TQString().setNum(TunnelPingRestart) ), config->debug );
				}
				else if ( line.startsWith ( "reneg-sec " ) )
				{
					acc->setUseRenegSec ( true );
					int RenegSec =TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setRenegSec ( RenegSec );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use reneg-sec: %1" ).arg ( TQString().setNum(RenegSec) ), config->debug );
				}
				else if ( line.startsWith ( "tun-mtu " ) )
				{
					acc->setUseMtu ( true );
					int Mtu = TQString(line.section ( ' ', 1, 1 )).toInt();
					acc->setMtu ( Mtu );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry ( i18n ( "OpenVPN import: use usedefinied MTU: %1" ).arg ( TQString().setNum(Mtu) ), config->debug );
				}
			}
		}

// 		if ( acc->getCertPath().isEmpty() ) {
// 			// 						acc->setCertPath("/etc/openvpn/");
// 			acc->setCertPath( locateLocal ( "data", "kvpnc" ) );
// 			CertPath = acc->getCertPath();
// 		}
// 		// 					/* we need to put the prefix to all cert paths */
// 		if ( !acc->getCaCertificate().isEmpty() && !acc->getCaCertificate().startsWith( CertPath ) ) {
// 			TQString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = TQString( Utils(this->GlobalConfig).removeSpecialCharsForFilename( filename).section( '/', 0, -2 ) + "/" );
//
// 			TQFile CaCertFileSrc ( prefix + acc->getCaCertificate() );
// 			TQFile CaCertFileDst ( TQString( CertPath + "/openvpnimport_" + acc->getCaCertificate() ) );
// 			acc->setCaCertificate( TQString( CertPath + "/openvpnimport_" + acc->getCaCertificate() ) );
//
// 			TQTextStream CaCertDstStream( &CaCertFileDst );
//
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy ca cert file %1 to %2." ).arg( CaCertFileSrc.name() ).arg( CaCertFileDst.name() ), config->debug );
//
//
// 			if ( CaCertFileSrc.open( IO_ReadOnly ) ) {
// 				if ( CaCertFileDst.open( IO_WriteOnly ) ) {
// 					TQString CaCertFileString = TQString( CaCertFileSrc.readAll() );
// 					CaCertDstStream << CaCertFileString;
// 					CaCertFileDst.close();
// 					if ( config->KvpncDebugLevel > 1 )
// 						config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 copied to %2." ).arg( CaCertFileSrc.name() ).arg( CaCertFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 could not open for writing." ).arg( CaCertFileDst.name() ), config->debug );
//
// 					//								return;
// 				}
// 				CaCertFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 could not open for reading." ).arg( CaCertFileSrc.name() ), config->debug );
//
// 				//							return;
// 			}
// 		}
// 		if ( !acc->getX509Certificate().isEmpty() && !acc->getX509Certificate().startsWith( CertPath ) ) {
// 			TQString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = TQString( Utils(this->GlobalConfig).removeSpecialCharsForFilename(filename).section( '/', 0, -2 ) + "/" );
//
// 			TQFile X509CertFileSrc ( prefix + acc->getX509Certificate() );
// 			TQFile X509CertFileDst ( TQString( CertPath + "/openvpnimport_" + acc->getX509Certificate() ) );
// 			acc->setX509Certificate( TQString( CertPath + "/openvpnimport_" + acc->getX509Certificate() ) );
//
// 			TQTextStream X509CertDstStream( &X509CertFileDst );
//
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy pre shared key file %1 to %2." ).arg( X509CertFileSrc.name() ).arg( X509CertFileDst.name() ), config->debug );
//
//
// 			if ( X509CertFileSrc.open( IO_ReadOnly ) ) {
// 				if ( X509CertFileDst.open( IO_WriteOnly ) ) {
// 					TQString X509CertFileString = TQString( X509CertFileSrc.readAll() );
// 					X509CertDstStream << X509CertFileString;
// 					X509CertFileDst.close();
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 copied to %2." ).arg( X509CertFileSrc.name() ).arg( X509CertFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 could not open for writing." ).arg( X509CertFileDst.name() ), config->debug );
//
// 					//								return;
// 				}
// 				X509CertFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 could not open for reading." ).arg( X509CertFileSrc.name() ), config->debug );
//
// 				//							return;
// 			}
// 		}
// 		if ( !acc->getPreSharedKey().isEmpty() && !acc->getPreSharedKey().startsWith( CertPath ) ) {
// 			TQString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = TQString( Utils(this->GlobalConfig).removeSpecialCharsForFilename( filename).section( '/', 0, -2 ) + "/" );
//
// 			TQFile PreSharedKeyFileSrc ( TQString( prefix + acc->getPreSharedKey() ) );
// 			TQFile PreSharedKeyFileDst ( TQString( CertPath + "/openvpnimport_" + acc->getPreSharedKey() ) );
// 			acc->setPreSharedKey( TQString( CertPath + "/openvpnimport_" + acc->getPreSharedKey() ) );
//
// 			TQTextStream PreSharedKeyDstStream( &PreSharedKeyFileDst );
//
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy pre shared key file %1 to %2." ).arg( PreSharedKeyFileSrc.name() ).arg( PreSharedKeyFileDst.name() ), config->debug );
//
// 			if ( PreSharedKeyFileSrc.open( IO_ReadOnly ) ) {
// 				if ( PreSharedKeyFileDst.open( IO_WriteOnly ) ) {
// 					TQString PreSharedKeyFileString = TQString( PreSharedKeyFileSrc.readAll() );
// 					PreSharedKeyDstStream << PreSharedKeyFileString;
// 					PreSharedKeyFileDst.close();
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 copied to %2." ).arg( PreSharedKeyFileSrc.name() ).arg( PreSharedKeyFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 could not open for writing." ).arg( PreSharedKeyFileDst.name() ), config->debug );
//
// 					//								return;
// 				}
// 				PreSharedKeyFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 could not open for reading." ).arg( PreSharedKeyFileSrc.name() ), config->debug );
//
// 				//							return;
// 			}
// 		}

		OpenvpnConfigFile.close();
		importOk = true;
	}
	else
		importOk = false;

	if ( !Pkcs12CertFile.isEmpty() )
	{

		TQString filepath = filename.section ( '/', 0, -2 ); // should give the base path

		ImportCertificateDialog dlg ( this, i18n ( "Import Certificate..." ).ascii(), config );

		TQString p12file="";

		if ( Pkcs12CertFile.find ( '/' ) < 0 )
			p12file = filepath + "/" + Pkcs12CertFile;
		else
			p12file = Pkcs12CertFile;

		dlg.main->ImporttypeComboBox->setCurrentItem ( ImportCertificateDialog::pkcs12_openvpn );
		dlg.typeToggeled ( ImportCertificateDialog::pkcs12_openvpn );
		dlg.main->ImporttypeComboBox->setEnabled ( false );
		// 		dlg.main->RacoonCertificatePathUrlrequester->setEnabled(true);
		dlg.main->RacoonCertificatePathUrlrequester->setURL ( "/etc/openvpn" );
		dlg.main->PrivateKeyPassGroupBox->setEnabled ( true );
// 		dlg.main->PrivateKeyPassGroupBox->setCurrentText(i18n("OpenVPN"));
		dlg.main->PrivateKeyPasswordEdit->setEnabled ( true );
		dlg.main->PrivateKeyAgainPasswordEdit->setEnabled ( true );

		dlg.main->FilenameUrlrequester->setURL ( p12file );
		dlg.main->ImportPasswordEdit->setEnabled ( true );
		dlg.main->P12GroupBox->setEnabled ( true );
		dlg.main->PrivateKeyPassGroupBox->setEnabled ( true );
		dlg.main->FilenameUrlrequester->setFilter ( "*.p12" );

		dlg.main->ImportPasswordEdit->setFocus();

		int result = dlg.exec();

		if ( result == TQDialog::Accepted )
		{
			if ( dlg.importSuccess )
			{
				config->appendLogEntry ( i18n ( "OpenVPN import: import of pkcs12 certificate file %1 was successful." ).arg ( Pkcs12CertFile ), config->info );
				importOk = true;

				acc->setX509Certificate ( dlg.main->RacoonCertificatePathUrlrequester->url() + "/mykeys_" + ( p12file.left ( p12file.length() - 4 ) ).section ( '/', -1, -1 ) + ".pem" );
				acc->setPrivateKey ( dlg.main->RacoonCertificatePathUrlrequester->url() + "/mykeys_" + ( p12file.left ( p12file.length() - 4 ) ).section ( '/', -1, -1 ) + ".pem" );
				acc->setCaCertificate ( dlg.main->RacoonCertificatePathUrlrequester->url() + "/ca_" + ( p12file.left ( p12file.length() - 4 ) ).section ( '/', -1, -1 ) + ".pem" );
				acc->setCertPath ( dlg.main->RacoonCertificatePathUrlrequester->url() );
				acc->setPrivateKeyPass ( dlg.main->ImportPasswordEdit->text() );


			}
			else
			{
				config->appendLogEntry ( i18n ( "OpenVPN import: import of pkcs12 certificate file %1 failed!" ).arg ( Pkcs12CertFile ), config->error );
				importOk = false;
			}
		}
		else
		{
			config->appendLogEntry ( i18n ( "OpenVPN import: import of pkcs12 certificate file was cancelled." ).arg ( Pkcs12CertFile ), config->info );
			importOk = false;
		}

	}

	//std::cout << "accept" << std::endl;
	TQDialog::accept();
}

#include "importopenvpnprofiledialog.moc"
