/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-10-15
 * Description : IPTC categories settings page.
 *
 * Copyright (C) 2006-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// QT includes.

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqvalidator.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialog.h>
#include <tdelistbox.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kactivelabel.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "iptccategories.h"
#include "iptccategories.moc"

namespace KIPIMetadataEditPlugin
{

class IPTCCategoriesPriv
{
public:

    IPTCCategoriesPriv()
    {
        addSubCategoryButton = 0;
        delSubCategoryButton = 0;
        subCategoriesBox     = 0;
        subCategoriesCheck   = 0;
        categoryCheck        = 0;
        categoryEdit         = 0;
        subCategoryEdit      = 0;
    }

    TQStringList  oldSubCategories;

    TQPushButton *addSubCategoryButton;
    TQPushButton *delSubCategoryButton;

    TQCheckBox   *subCategoriesCheck;
    TQCheckBox   *categoryCheck;

    KLineEdit   *categoryEdit;
    KLineEdit   *subCategoryEdit;

    TDEListBox    *subCategoriesBox;
};

IPTCCategories::IPTCCategories(TQWidget* parent)
              : TQWidget(parent)
{
    d = new IPTCCategoriesPriv;
    TQGridLayout *grid = new TQGridLayout(parent, 6, 1, 0, KDialog::spacingHint());
    grid->setAlignment( TQt::AlignTop );

    // IPTC only accept printable Ascii char.
    TQRegExp asciiRx("[\x20-\x7F]+$");
    TQValidator *asciiValidator = new TQRegExpValidator(asciiRx, this);

    // --------------------------------------------------------

    d->categoryCheck = new TQCheckBox(i18n("Identify subject of content (3 chars max):"), parent);    
    d->categoryEdit  = new KLineEdit(parent);
    d->categoryEdit->setValidator(asciiValidator);
    d->categoryEdit->setMaxLength(3);
    TQWhatsThis::add(d->categoryEdit, i18n("<p>Set here the category of content. This field is limited "
                                         "to 3 ASCII characters."));

    d->subCategoriesCheck = new TQCheckBox(i18n("Supplemental categories:"), parent);    

    d->subCategoryEdit = new KLineEdit(parent);
    d->subCategoryEdit->setValidator(asciiValidator);
    d->subCategoryEdit->setMaxLength(32);
    TQWhatsThis::add(d->subCategoryEdit, i18n("<p>Enter here a new supplemental category of content. "
                    "This field is limited to 32 ASCII characters."));

    d->subCategoriesBox = new TDEListBox(parent);
    d->subCategoriesBox->setVScrollBarMode(TQScrollView::AlwaysOn);
    
    d->addSubCategoryButton = new TQPushButton( i18n("&Add"), parent);
    d->delSubCategoryButton = new TQPushButton( i18n("&Delete"), parent);
    d->addSubCategoryButton->setIconSet(SmallIcon("add"));
    d->delSubCategoryButton->setIconSet(SmallIcon("remove"));
    d->delSubCategoryButton->setEnabled(false);

    grid->addMultiCellWidget(d->categoryCheck, 0, 0, 0, 1);
    grid->addMultiCellWidget(d->categoryEdit, 0, 0, 1, 1);
    grid->addMultiCellWidget(d->subCategoriesCheck, 1, 1, 0, 1);
    grid->addMultiCellWidget(d->subCategoryEdit, 2, 2, 0, 0);
    grid->addMultiCellWidget(d->subCategoriesBox, 3, 6, 0, 0);
    grid->addMultiCellWidget(d->addSubCategoryButton, 3, 3, 1, 1);
    grid->addMultiCellWidget(d->delSubCategoryButton, 4, 4, 1, 1);

    // --------------------------------------------------------

    KActiveLabel *note = new KActiveLabel(i18n("<b>Note: "
                 "<b><a href='http://en.wikipedia.org/wiki/IPTC'>IPTC</a></b> "
                 "text tags only support the printable "
                 "<b><a href='http://en.wikipedia.org/wiki/Ascii'>ASCII</a></b> "
                 "characters set and limit strings size. "
                 "Use contextual help for details.</b>"), parent);
    note->setMaximumWidth(150);

    grid->addMultiCellWidget(note, 5, 5, 1, 1);
    grid->setColStretch(0, 10);                     
    grid->setRowStretch(6, 10);      
                                         
    // --------------------------------------------------------

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->categoryEdit, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoriesBox, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoriesCheck, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoryEdit, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoriesBox, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->addSubCategoryButton, TQ_SLOT(setEnabled(bool)));

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            d->delSubCategoryButton, TQ_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->subCategoriesCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoryEdit, TQ_SLOT(setEnabled(bool)));

    connect(d->subCategoriesCheck, TQ_SIGNAL(toggled(bool)),
            d->subCategoriesBox, TQ_SLOT(setEnabled(bool)));

    connect(d->subCategoriesCheck, TQ_SIGNAL(toggled(bool)),
            d->addSubCategoryButton, TQ_SLOT(setEnabled(bool)));

    connect(d->subCategoriesCheck, TQ_SIGNAL(toggled(bool)),
            d->delSubCategoryButton, TQ_SLOT(setEnabled(bool)));

    // --------------------------------------------------------

    connect(d->subCategoriesBox, TQ_SIGNAL(selectionChanged()),
            this, TQ_SLOT(slotCategorySelectionChanged()));
    
    connect(d->addSubCategoryButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotAddCategory()));
    
    connect(d->delSubCategoryButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotDelCategory()));

    // --------------------------------------------------------

    connect(d->categoryCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));

    connect(d->subCategoriesCheck, TQ_SIGNAL(toggled(bool)),
            this, TQ_SIGNAL(signalModified()));

    connect(d->addSubCategoryButton, TQ_SIGNAL(clicked()),
            this, TQ_SIGNAL(signalModified()));
    
    connect(d->delSubCategoryButton, TQ_SIGNAL(clicked()),
            this, TQ_SIGNAL(signalModified()));

    connect(d->categoryEdit, TQ_SIGNAL(textChanged(const TQString &)),
            this, TQ_SIGNAL(signalModified()));
}

IPTCCategories::~IPTCCategories()
{
    delete d;
}

void IPTCCategories::slotDelCategory()
{
    int index = d->subCategoriesBox->currentItem();
    if (index == -1)
        return;

    TQListBoxItem* item = d->subCategoriesBox->item(index);
    if (!item) return;
    delete item;
}

void IPTCCategories::slotCategorySelectionChanged()
{
    if (d->subCategoriesBox->currentItem() != -1)
        d->delSubCategoryButton->setEnabled(true);
    else
        d->delSubCategoryButton->setEnabled(false);
}

void IPTCCategories::slotAddCategory()
{
    TQString newCategory = d->subCategoryEdit->text();
    if (newCategory.isEmpty()) return;

    bool found = false;
    for (TQListBoxItem *item = d->subCategoriesBox->firstItem();
         item; item = item->next()) 
    {
        if (newCategory == item->text()) 
        {
            found = true;
            break;
        }
    }

    if (!found)
        d->subCategoriesBox->insertItem(newCategory);
}

void IPTCCategories::readMetadata(TQByteArray& iptcData)
{
    blockSignals(true);
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);
    TQString data;

    d->categoryEdit->clear();
    d->categoryCheck->setChecked(false);
    data = exiv2Iface.getIptcTagString("Iptc.Application2.Category", false);    
    if (!data.isNull())
    {
        d->categoryEdit->setText(data);
        d->categoryCheck->setChecked(true);
    }
    d->categoryEdit->setEnabled(d->categoryCheck->isChecked());
    d->subCategoriesCheck->setEnabled(d->categoryCheck->isChecked());

    d->subCategoriesBox->clear();
    d->subCategoriesCheck->setChecked(false);
    d->oldSubCategories = exiv2Iface.getImageSubCategories();
    if (!d->oldSubCategories.isEmpty())
    {
        d->subCategoriesBox->insertStringList(d->oldSubCategories);
        d->subCategoriesCheck->setChecked(true);
    }
    d->subCategoryEdit->setEnabled(d->categoryCheck->isChecked() && d->subCategoriesCheck->isChecked());
    d->subCategoriesBox->setEnabled(d->categoryCheck->isChecked() && d->subCategoriesCheck->isChecked());
    d->addSubCategoryButton->setEnabled(d->categoryCheck->isChecked() && d->subCategoriesCheck->isChecked());
    d->delSubCategoryButton->setEnabled(d->categoryCheck->isChecked() && d->subCategoriesCheck->isChecked());

    blockSignals(false);
}

void IPTCCategories::applyMetadata(TQByteArray& iptcData)
{
    TQStringList newCategories;    
    KExiv2Iface::KExiv2 exiv2Iface;
    exiv2Iface.setIptc(iptcData);

    if (d->categoryCheck->isChecked())
        exiv2Iface.setIptcTagString("Iptc.Application2.Category", d->categoryEdit->text());
    else
        exiv2Iface.removeIptcTag("Iptc.Application2.Category");

    for (TQListBoxItem *item = d->subCategoriesBox->firstItem();
         item; item = item->next()) 
        newCategories.append(item->text());

    if (d->categoryCheck->isChecked() && d->subCategoriesCheck->isChecked())
        exiv2Iface.setImageSubCategories(d->oldSubCategories, newCategories);
    else
        exiv2Iface.setImageSubCategories(d->oldSubCategories, TQStringList());

    iptcData = exiv2Iface.getIptc();
}

}  // namespace KIPIMetadataEditPlugin

