/* ============================================================
 * File  : gpevents.h
 * Author: Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Date  : 2003-01-22
 * Description : 
 * 
 * Copyright 2003 by Renchi Raju <renchi@pooh.tam.uiuc.edu>

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#ifndef GPEVENTS_H
#define GPEVENTS_H

#include <tqevent.h>
#include <tqstring.h>
#include <tqimage.h>
#include <tqvaluelist.h>

#include "mtlist.h"
#include "gpfileiteminfo.h"

namespace KIPIKameraKlientPlugin
{

class GPEvent : public TQCustomEvent {

public:
    enum Type {
        Init = TQCustomEvent::User,
        GetSubFolders,
        MakeFolder,
        DeleteFolder,
        GetItemsInfo,
        GetAllItemsInfo,
        GetThumbnail,
        DownloadItem,
        DeleteItem,
        DeleteAllItems,
        OpenItem,
        OpenItemWithService,
        UploadItem,
        ExifInfo,
        Information,
        StatusMsg,
        Progress,
        Error,
        Busy
    };    
    GPEvent(Type type) : TQCustomEvent(type) {}
};

class GPEventError : public GPEvent {
public:
    GPEventError(const TQString errorMsg) : GPEvent(Error), errorMsg_(errorMsg) {}
    TQString errorMsg() const { return errorMsg_; }

private:
    TQString errorMsg_;

};

class GPEventGetSubFolders : public GPEvent {
public:
    GPEventGetSubFolders(const TQString& folder, const TQValueList<TQString>& subFolderList)
        : GPEvent(GetSubFolders),
          folder_(folder),
          subFolderList_(subFolderList)
        {}
    TQString folder() const { return folder_; }
    const MTList<TQString>& subFolderList() const { return subFolderList_; }

private:
    TQString folder_;
    MTList<TQString> subFolderList_;
};

class GPEventGetItemsInfo : public GPEvent {
public:
    GPEventGetItemsInfo(const TQString& folder, const GPFileItemInfoList& infoList)
        : GPEvent(GetItemsInfo),
          folder_(folder),
          infoList_(infoList)
        {}
    TQString folder() const { return folder_; }
    const MTList<GPFileItemInfo>& infoList() const { return infoList_; }

private:
    TQString folder_;
    MTList<GPFileItemInfo> infoList_;
};

class GPEventGetAllItemsInfo : public GPEvent {
public:
    GPEventGetAllItemsInfo(const GPFileItemInfoList& infoList)
        : GPEvent(GetAllItemsInfo),
          infoList_(infoList)
        {}
    const MTList<GPFileItemInfo>& infoList() const { return infoList_; }

private:
    MTList<GPFileItemInfo> infoList_;
};

class GPEventGetThumbnail : public GPEvent {
public:

    GPEventGetThumbnail(const TQString& folder, const TQString& imageName, const TQImage& thumbnail)
        : GPEvent(GetThumbnail),
          folder_(folder),
          imageName_(imageName),
          thumbnail_(thumbnail)
        {}
    TQString folder() const { return folder_; }
    TQString imageName() const { return imageName_; }
    TQImage  thumbnail() const { return thumbnail_; }

private:
    TQString folder_;
    TQString imageName_;
    TQImage  thumbnail_;
};

class GPEventDownloadItem : public GPEvent {
public:
    GPEventDownloadItem(const TQString& folder, const TQString& itemName)
        : GPEvent(DownloadItem),
          folder_(folder),
          itemName_(itemName)
        {}
    TQString folder() const { return folder_; }
    TQString itemName() const { return itemName_; }

private:
    TQString folder_;
    TQString itemName_;
};

class GPEventDeleteItem : public GPEvent {
public:
    GPEventDeleteItem(const TQString& folder, const TQString& itemName)
        : GPEvent(DeleteItem),
          folder_(folder),
          itemName_(itemName)
        {}
    TQString folder() const { return folder_; }
    TQString itemName() const { return itemName_; }

private:
    TQString folder_;
    TQString itemName_;
};

class GPEventOpenItem : public GPEvent {
public:
    GPEventOpenItem(const TQString& openFile) : GPEvent(OpenItem), openFile_(openFile)
        {}
    TQString openFile() const { return openFile_; }

private:
    TQString openFile_;
};

class GPEventOpenItemWithService : public GPEvent {
public:
    GPEventOpenItemWithService(const TQString& openFile, const TQString& serviceName)
        : GPEvent(OpenItemWithService),
          openFile_(openFile),
          serviceName_(serviceName)
        { }
    TQString openFile() const { return openFile_; }
    TQString serviceName() const { return serviceName_; }

private:
    TQString openFile_;
    TQString serviceName_;
};

class GPEventExifInfo : public GPEvent {
public:
    GPEventExifInfo(const TQString& folder, const TQString& itemName, char *data, int size)
        : GPEvent(ExifInfo),
          folder_(folder),
          itemName_(itemName),
          data_(data),
          size_(size)
        {}

    TQString folder() const { return folder_; }
    TQString itemName() const { return itemName_; }
    char*   data() { return data_; }
    int     size() const { return size_; }
    
private:
    TQString folder_;
    TQString itemName_;
    char *data_;
    int   size_;
};

class GPEventStatusMsg : public GPEvent {
public:
    GPEventStatusMsg(const TQString& msg) : GPEvent(StatusMsg), msg_(msg)
        {}
    TQString msg() const { return msg_; }

private:
    TQString msg_;
};

class GPEventProgress : public GPEvent {
public:
    GPEventProgress(int val) : GPEvent(Progress), val_(val)
        {}
    int val() { return val_; }

private:
    int val_;
};

class GPEventBusy : public GPEvent {
public:
    GPEventBusy(bool busy) : GPEvent(Busy), busy_(busy)
        {}
    bool busy() { return busy_; }

private:
    bool busy_;
};

}  // NameSpace KIPIKameraKlientPlugin

#endif /* GPEVENTS_H */
