/*
 * Copyright Johannes Sixt
 * This file is licensed under the GNU General Public License Version 2.
 * See the file COPYING in the toplevel directory of the source directory.
 */

#ifndef WINSTACK_H
#define WINSTACK_H

#include <tqdialog.h>
#include <tqlineedit.h>
#include <tqlayout.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqtooltip.h>
#include <ktabwidget.h>
#include <list>

// forward declarations
class KDebugger;
class WinStack;
class SourceWindow;
class DisassembledCode;
struct DbgAddr;

class FindDialog : public TQDialog
{
    TQ_OBJECT
public:
    FindDialog();
    ~FindDialog();

    bool caseSensitive() const { return m_caseCheck.isChecked(); }
    TQString searchText() const { return m_searchText.text(); }
    virtual void done(int result);

    TQLineEdit m_searchText;
    TQCheckBox m_caseCheck;
    TQPushButton m_buttonForward;
    TQPushButton m_buttonBackward;
    TQPushButton m_buttonClose;

signals:
    void closed();

protected:
    virtual void closeEvent(TQCloseEvent* ev);
    TQVBoxLayout m_layout;
    TQHBoxLayout m_buttons;
};


class ValueTip : public TQToolTip
{
public:
    ValueTip(WinStack* parent);
    virtual ~ValueTip() {}	// TQt3's TQToolTip lacks virtual dtor!
    virtual void maybeTip(const TQPoint& p);
    void tip(const TQRect& r, const TQString& s) { TQToolTip::tip(r, s); }
};


class WinStack : public KTabWidget
{
    TQ_OBJECT
public:
    WinStack(TQWidget* parent, const char* name);
    virtual ~WinStack();

    /**
     * Slot activate also looks in this directory when the specified file is
     * a relative path.
     */
    void setExtraDirectory(const TQString& dir) { m_lastOpenDir = dir; }
    void activateFile(const TQString& fileName);
    bool activeLine(TQString& filename, int& lineNo);
    bool activeLine(TQString& filename, int& lineNo, DbgAddr& address);
    void maybeTip(const TQPoint& p);
    bool hasWindows() const { return count() > 0; }
    TQString activeFileName() const;
    SourceWindow* activeWindow() const;
    SourceWindow* windowAt(int i) const;

    virtual TQSize sizeHint() const;

signals:
    void toggleBreak(const TQString&, int, const DbgAddr&, bool);
    void enadisBreak(const TQString&, int, const DbgAddr&);
    void newFileLoaded();
    void initiateValuePopup(const TQString&);
    void disassemble(const TQString&, int);
    void setTabWidth(int numChars);
    void moveProgramCounter(const TQString&, int, const DbgAddr&);

public slots:
    virtual void slotFindForward();
    virtual void slotFindBackward();
    virtual void activate(const TQString& filename, int lineNo, const DbgAddr& address);
    void updatePC(const TQString& filename, int lineNo, const DbgAddr& address, int frameNo);
    void reloadAllFiles();
    void updateLineItems(const KDebugger* deb);
    void slotSetTabWidth(int numChars);

    void slotFileReload();
    void slotViewFind();
    void slotBrkptSet();
    void slotBrkptSetTemp();
    void slotBrkptEnable();
    void slotMoveProgramCounter();
    void slotClose();

    // Displays the value tip at m_tipLocation
    void slotShowValueTip(const TQString& tipText);

    // Shows the disassembled code at the location given by file and lineNo
    void slotDisassembled(const TQString& fileName, int lineNo,
			  const std::list<DisassembledCode>& disass);

    // Updates line items after expanding/collapsing disassembled code
    void slotExpandCollapse(int lineNo);

protected:
    bool activatePath(TQString pathname, int lineNo, const DbgAddr& address);
    virtual bool activateWindow(SourceWindow* fw, int lineNo, const DbgAddr& address);	/* -1 doesnt change line */
    virtual void contextMenuEvent(TQContextMenuEvent* e);
    void setPC(bool set, const TQString& fileName, int lineNo,
	       const DbgAddr& address, int frameNo);
    SourceWindow* findByFileName(const TQString& fileName) const;
    TQString m_lastOpenDir;		/* where user opened last file */
    
    // program counter
    TQString m_pcFile;
    int m_pcLine;			/* -1 if no PC */
    TQString m_pcAddress;		/* exact address of PC */
    int m_pcFrame;

    ValueTip m_valueTip;
    TQRect m_tipLocation;		/* where tip should appear */

    int m_tabWidth;			/* number of chars */

public:
    // find dialog
    FindDialog m_findDlg;
};

#endif // WINSTACK_H
