/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-08-24
 * Description : a plugin to reduce CCD noise.
 *
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqtooltip.h>
#include <tqcheckbox.h>
#include <tqstring.h>
#include <tqtabwidget.h>
#include <tqimage.h>
#include <tqlayout.h>
#include <tqfile.h>
#include <tqtextstream.h>

// KDE includes.

#include <tdelocale.h>
#include <tdeaboutdata.h>
#include <kiconloader.h>
#include <tdeapplication.h>
#include <tdestandarddirs.h>
#include <knuminput.h>
#include <tdefiledialog.h>
#include <tdeglobalsettings.h>
#include <tdemessagebox.h>
#include <tdeconfig.h>

// Local includes.

#include "version.h"
#include "ddebug.h"
#include "dimg.h"
#include "imageiface.h"
#include "imagewidget.h"
#include "noisereduction.h"
#include "imageeffect_noisereduction.h"
#include "imageeffect_noisereduction.moc"

namespace DigikamNoiseReductionImagesPlugin
{

ImageEffect_NoiseReduction::ImageEffect_NoiseReduction(TQWidget* parent)
        : Digikam::CtrlPanelDlg(parent, i18n("Noise Reduction"),
                                "noisereduction", true, true, true,
                                Digikam::ImagePannelWidget::SeparateViewAll)
{
    TQString whatsThis;

    TDEAboutData* about = new TDEAboutData("digikam",
                                       I18N_NOOP("Noise Reduction"),
                                       digikam_version,
                                       I18N_NOOP("A noise reduction image filter plugin for digiKam."),
                                       TDEAboutData::License_GPL,
                                       "(c) 2004-2008, Gilles Caulier",
                                       0,
                                       "http://www.digikam.org");

    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    about->addAuthor("Peter Heckert", I18N_NOOP("Noise Reduction algorithm. Developer"),
                     "peter dot heckert at arcor dot de");

    setAboutData(about);

    // -------------------------------------------------------------

    TQTabWidget *mainTab = new TQTabWidget(m_imagePreviewWidget);

    TQWidget* firstPage = new TQWidget( mainTab );
    TQGridLayout* gridSettings = new TQGridLayout( firstPage, 6, 1, spacingHint());
    mainTab->addTab( firstPage, i18n("Details") );

    TQLabel *label1 = new TQLabel(i18n("Radius:"), firstPage);

    m_radiusInput = new KDoubleNumInput(firstPage);
    m_radiusInput->setPrecision(1);
    m_radiusInput->setRange(0.0, 10.0, 0.1, true);
    TQWhatsThis::add( m_radiusInput, i18n("<p><b>Radius</b>: this control selects the "
                                         "gliding window size used for the filter. Larger values do not increase "
                                         "the amount of time needed to filter each pixel in the image but "
                                         "can cause blurring. This window moves across the image, and the "
                                         "color in it is smoothed to remove imperfections. "
                                         "In any case it must be about the same size as the noise granularity "
                                         "or somewhat more. If it is set higher than necessary, then it "
                                         "can cause unwanted blur."));

    gridSettings->addMultiCellWidget(label1, 0, 0, 0, 0);
    gridSettings->addMultiCellWidget(m_radiusInput, 0, 0, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label3 = new TQLabel(i18n("Threshold:"), firstPage);

    m_thresholdInput = new KDoubleNumInput(firstPage);
    m_thresholdInput->setPrecision(2);
    m_thresholdInput->setRange(0.0, 1.0, 0.01, true);
    TQWhatsThis::add( m_thresholdInput, i18n("<p><b>Threshold</b>: use the slider for coarse adjustment, "
                                            "and the spin control for fine adjustment to control edge detection sensitivity. "
                                            "This value should be set so that edges and details are clearly visible "
                                            "and noise is smoothed out. "
                                            "Adjustment must be made carefully, because the gap between \"noisy\", "
                                            "\"smooth\", and \"blur\" is very small. Adjust it as carefully as you would adjust "
                                            "the focus of a camera."));

    gridSettings->addMultiCellWidget(label3, 1, 1, 0, 0);
    gridSettings->addMultiCellWidget(m_thresholdInput, 1, 1, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label4 = new TQLabel(i18n("Texture:"), firstPage);

    m_textureInput = new KDoubleNumInput(firstPage);
    m_textureInput->setPrecision(2);
    m_textureInput->setRange(-0.99, 0.99, 0.01, true);
    TQWhatsThis::add( m_textureInput, i18n("<p><b>Texture</b>: this control sets the texture accuracy. "
                                          "This value can be used, to get more or less texture accuracy. When decreased, "
                                          "then noise and texture are blurred out, when increased then texture is "
                                          "amplified, but also noise will increase. It has almost no effect on image edges."));

    gridSettings->addMultiCellWidget(label4, 2, 2, 0, 0);
    gridSettings->addMultiCellWidget(m_textureInput, 2, 2, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label7 = new TQLabel(i18n("Sharpness:"), firstPage);  // Filter setting "Lookahead".

    m_sharpnessInput = new KDoubleNumInput(firstPage);
    m_sharpnessInput->setPrecision(2);
    m_sharpnessInput->setRange(0.0, 1.0, 0.1, true);
    TQWhatsThis::add( m_sharpnessInput, i18n("<p><b>Sharpness</b>: "
                                            "This value improves the frequency response for the filter. "
                                            "When it is too strong then not all noise can be removed, or spike noise may appear. "
                                            "Set it near to maximum, if you want to remove very weak noise or JPEG-artifacts, "
                                            "without losing detail."));

    gridSettings->addMultiCellWidget(label7, 3, 3, 0, 0);
    gridSettings->addMultiCellWidget(m_sharpnessInput, 3, 3, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label5 = new TQLabel(i18n("Edge Lookahead:"), firstPage);     // Filter setting "Sharp".

    m_lookaheadInput = new KDoubleNumInput(firstPage);
    m_lookaheadInput->setPrecision(2);
    m_lookaheadInput->setRange(0.01, 20.0, 0.01, true);
    TQWhatsThis::add( m_lookaheadInput, i18n("<p><b>Edge</b>: "
                                            "This value defines the pixel distance to which the filter looks ahead for edges. "
                                            "When this value is increased, then spike noise is erased. "
                                            "You can eventually re-adjust the <b>Edge</b> filter, when you have changed this setting. "
                                            "When this value is too high, the adaptive filter can no longer accurately track "
                                            "image details, and noise or blurring can occur."));

    gridSettings->addMultiCellWidget(label5, 4, 4, 0, 0);
    gridSettings->addMultiCellWidget(m_lookaheadInput, 4, 4, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label10 = new TQLabel(i18n("Erosion:"), firstPage);

    m_phaseInput = new KDoubleNumInput(firstPage);
    m_phaseInput->setPrecision(1);
    m_phaseInput->setRange(0.5, 20.0, 0.5, true);
    TQWhatsThis::add( m_phaseInput, i18n("<p><b>Erosion</b>: "
                                        "Use this to increase edge noise erosion and spike noise erosion "
                                        "(noise is removed by erosion)."));

    gridSettings->addMultiCellWidget(label10, 5, 5, 0, 0);
    gridSettings->addMultiCellWidget(m_phaseInput, 5, 5, 1, 1);
    gridSettings->setColStretch(1, 10);
    gridSettings->setRowStretch(6, 10);

    // -------------------------------------------------------------

    TQWidget* secondPage = new TQWidget( mainTab );
    TQGridLayout* gridSettings2 = new TQGridLayout( secondPage, 4, 1, spacingHint());
    mainTab->addTab( secondPage, i18n("Advanced") );

    TQLabel *label2 = new TQLabel(i18n("Luminance:"), secondPage);

    m_lumToleranceInput = new KDoubleNumInput(secondPage);
    m_lumToleranceInput->setPrecision(1);
    m_lumToleranceInput->setRange(0.0, 1.0, 0.1, true);
    TQWhatsThis::add( m_lumToleranceInput, i18n("<p><b>Luminance</b>: this control sets the luminance tolerance of the image."
                     "We recommend using either the <b>Color</b> or the <b>Luminance</b> tolerance settings "
                     "to make an image correction, not both at the same time. These settings "
                     "do not influence the main smoothing process controlled by the <b>Details</b> "
                     "settings."));

    gridSettings2->addMultiCellWidget(label2, 0, 0, 0, 0);
    gridSettings2->addMultiCellWidget(m_lumToleranceInput, 0, 0, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label6 = new TQLabel(i18n("Color:"), secondPage);

    m_csmoothInput = new KDoubleNumInput(secondPage);
    m_csmoothInput->setPrecision(1);
    m_csmoothInput->setRange(0.0, 1.0, 0.1, true);
    TQWhatsThis::add( m_csmoothInput, i18n("<p><b>Color</b>: this control sets the color tolerance of the image. It is "
                                          "recommended using either the <b>Color</b> or the <b>Luminance</b> tolerance "
                                          "to make image correction, not both at the same time. These settings "
                                          "do not influence the main smoothing process controlled by the <b>Details</b> "
                                          "settings."));

    gridSettings2->addMultiCellWidget(label6, 1, 1, 0, 0);
    gridSettings2->addMultiCellWidget(m_csmoothInput, 1, 1, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label8 = new TQLabel(i18n("Gamma:"), secondPage);

    m_gammaInput = new KDoubleNumInput(secondPage);
    m_gammaInput->setPrecision(1);
    m_gammaInput->setRange(0.3, 3.0, 0.1, true);
    TQWhatsThis::add( m_gammaInput, i18n("<p><b>Gamma</b>: this control sets the gamma tolerance of the image. This value "
                                        "can be used to increase the tolerance values for darker areas (which commonly "
                                        "are noisier). This results in more blur for shadow areas."));

    gridSettings2->addMultiCellWidget(label8, 2, 2, 0, 0);
    gridSettings2->addMultiCellWidget(m_gammaInput, 2, 2, 1, 1);

    // -------------------------------------------------------------

    TQLabel *label9 = new TQLabel(i18n("Damping:"), secondPage);

    m_dampingInput = new KDoubleNumInput(secondPage);
    m_dampingInput->setPrecision(1);
    m_dampingInput->setRange(0.5, 20.0, 0.5, true);
    TQWhatsThis::add( m_dampingInput, i18n("<p><b>Damping</b>: this control sets the phase-jitter damping adjustment. "
                                          "This value defines how fast the adaptive filter-radius reacts to luminance "
                                          "variations. If increased, then edges appear smoother; if too high, then blur "
                                          "may occur. If at minimum, then noise and phase jitter at the edges can occur. It "
                                          "can suppress spike noise when increased, and this is the preferred method to "
                                          "remove it."));

    gridSettings2->addMultiCellWidget(label9, 3, 3, 0, 0);
    gridSettings2->addMultiCellWidget(m_dampingInput, 3, 3, 1, 1);
    gridSettings2->setColStretch(1, 10);
    gridSettings2->setRowStretch(4, 10);

    m_imagePreviewWidget->setUserAreaWidget(mainTab);

    // -------------------------------------------------------------

//     connect(m_radiusInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_lumToleranceInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_thresholdInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_textureInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_sharpnessInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_csmoothInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_lookaheadInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_gammaInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_dampingInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
// 
//     connect(m_phaseInput, TQ_SIGNAL(valueChanged(double)),
//             this, TQ_SLOT(slotTimer()));
}

ImageEffect_NoiseReduction::~ImageEffect_NoiseReduction()
{
}

void ImageEffect_NoiseReduction::renderingFinished()
{
    m_radiusInput->setEnabled(true);
    m_lumToleranceInput->setEnabled(true);
    m_thresholdInput->setEnabled(true);
    m_textureInput->setEnabled(true);
    m_sharpnessInput->setEnabled(true);
    m_csmoothInput->setEnabled(true);
    m_lookaheadInput->setEnabled(true);
    m_gammaInput->setEnabled(true);
    m_dampingInput->setEnabled(true);
    m_phaseInput->setEnabled(true);
}

void ImageEffect_NoiseReduction::readUserSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("noisereduction Tool Dialog");
    m_radiusInput->setEnabled(true);
    m_lumToleranceInput->setEnabled(true);
    m_thresholdInput->setEnabled(true);
    m_textureInput->setEnabled(true);
    m_sharpnessInput->setEnabled(true);
    m_csmoothInput->setEnabled(true);
    m_lookaheadInput->setEnabled(true);
    m_gammaInput->setEnabled(true);
    m_dampingInput->setEnabled(true);
    m_phaseInput->setEnabled(true);

    m_radiusInput->setValue(config->readDoubleNumEntry("RadiusAjustment", 1.0));
    m_lumToleranceInput->setValue(config->readDoubleNumEntry("LumToleranceAjustment", 1.0));
    m_thresholdInput->setValue(config->readDoubleNumEntry("ThresholdAjustment", 0.08));
    m_textureInput->setValue(config->readDoubleNumEntry("TextureAjustment", 0.0));
    m_sharpnessInput->setValue(config->readDoubleNumEntry("SharpnessAjustment", 0.25));
    m_csmoothInput->setValue(config->readDoubleNumEntry("CsmoothAjustment", 1.0));
    m_lookaheadInput->setValue(config->readDoubleNumEntry("LookAheadAjustment", 2.0));
    m_gammaInput->setValue(config->readDoubleNumEntry("GammaAjustment", 1.4));
    m_dampingInput->setValue(config->readDoubleNumEntry("DampingAjustment", 5.0));
    m_phaseInput->setValue(config->readDoubleNumEntry("PhaseAjustment", 1.0));

    m_radiusInput->setEnabled(false);
    m_lumToleranceInput->setEnabled(false);
    m_thresholdInput->setEnabled(false);
    m_textureInput->setEnabled(false);
    m_sharpnessInput->setEnabled(false);
    m_csmoothInput->setEnabled(false);
    m_lookaheadInput->setEnabled(false);
    m_gammaInput->setEnabled(false);
    m_dampingInput->setEnabled(false);
    m_phaseInput->setEnabled(false);
}

void ImageEffect_NoiseReduction::writeUserSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("noisereduction Tool Dialog");
    config->writeEntry("RadiusAjustment", m_radiusInput->value());
    config->writeEntry("LumToleranceAjustment", m_lumToleranceInput->value());
    config->writeEntry("ThresholdAjustment", m_thresholdInput->value());
    config->writeEntry("TextureAjustment", m_textureInput->value());
    config->writeEntry("SharpnessAjustment", m_sharpnessInput->value());
    config->writeEntry("CsmoothAjustment", m_csmoothInput->value());
    config->writeEntry("LookAheadAjustment", m_lookaheadInput->value());
    config->writeEntry("GammaAjustment", m_gammaInput->value());
    config->writeEntry("DampingAjustment", m_dampingInput->value());
    config->writeEntry("PhaseAjustment", m_phaseInput->value());
    config->sync();
}

void ImageEffect_NoiseReduction::resetValues()
{
    m_radiusInput->setEnabled(true);
    m_lumToleranceInput->setEnabled(true);
    m_thresholdInput->setEnabled(true);
    m_textureInput->setEnabled(true);
    m_sharpnessInput->setEnabled(true);
    m_csmoothInput->setEnabled(true);
    m_lookaheadInput->setEnabled(true);
    m_gammaInput->setEnabled(true);
    m_dampingInput->setEnabled(true);
    m_phaseInput->setEnabled(true);

    m_radiusInput->setValue(1.0);
    m_lumToleranceInput->setValue(1.0);
    m_thresholdInput->setValue(0.08);
    m_textureInput->setValue(0.0);
    m_sharpnessInput->setValue(0.25);
    m_csmoothInput->setValue(1.0);
    m_lookaheadInput->setValue(2.0);
    m_gammaInput->setValue(1.4);
    m_dampingInput->setValue(5.0);
    m_phaseInput->setValue(1.0);

    m_radiusInput->setEnabled(false);
    m_lumToleranceInput->setEnabled(false);
    m_thresholdInput->setEnabled(false);
    m_textureInput->setEnabled(false);
    m_sharpnessInput->setEnabled(false);
    m_csmoothInput->setEnabled(false);
    m_lookaheadInput->setEnabled(false);
    m_gammaInput->setEnabled(false);
    m_dampingInput->setEnabled(false);
    m_phaseInput->setEnabled(false);
}

void ImageEffect_NoiseReduction::prepareEffect()
{
    m_radiusInput->setEnabled(false);
    m_lumToleranceInput->setEnabled(false);
    m_thresholdInput->setEnabled(false);
    m_textureInput->setEnabled(false);
    m_sharpnessInput->setEnabled(false);
    m_csmoothInput->setEnabled(false);
    m_lookaheadInput->setEnabled(false);
    m_gammaInput->setEnabled(false);
    m_dampingInput->setEnabled(false);
    m_phaseInput->setEnabled(false);

    double r  = m_radiusInput->value();
    double l  = m_lumToleranceInput->value();
    double th = m_thresholdInput->value();
    double tx = m_textureInput->value();
    double s  = m_sharpnessInput->value();
    double c  = m_csmoothInput->value();
    double a  = m_lookaheadInput->value();
    double g  = m_gammaInput->value();
    double d  = m_dampingInput->value();
    double p  = m_phaseInput->value();

    Digikam::DImg image = m_imagePreviewWidget->getOriginalRegionImage();

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(new NoiseReduction(&image,
                       this, r, l, th, tx, s, c, a, g, d, p));
}

void ImageEffect_NoiseReduction::prepareFinal()
{
    m_radiusInput->setEnabled(false);
    m_lumToleranceInput->setEnabled(false);
    m_thresholdInput->setEnabled(false);
    m_textureInput->setEnabled(false);
    m_sharpnessInput->setEnabled(false);
    m_csmoothInput->setEnabled(false);
    m_lookaheadInput->setEnabled(false);
    m_gammaInput->setEnabled(false);
    m_dampingInput->setEnabled(false);
    m_phaseInput->setEnabled(false);

    double r  = m_radiusInput->value();
    double l  = m_lumToleranceInput->value();
    double th = m_thresholdInput->value();
    double tx = m_textureInput->value();
    double s  = m_sharpnessInput->value();
    double c  = m_csmoothInput->value();
    double a  = m_lookaheadInput->value();
    double g  = m_gammaInput->value();
    double d  = m_dampingInput->value();
    double p  = m_phaseInput->value();

    Digikam::ImageIface iface(0, 0);
    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(new NoiseReduction(iface.getOriginalImg(),
                       this, r, l, th, tx, s, c, a, g, d, p));
}

void ImageEffect_NoiseReduction::putPreviewData(void)
{
    m_imagePreviewWidget->setPreviewImage(m_threadedFilter->getTargetImage());
}

void ImageEffect_NoiseReduction::putFinalData(void)
{
    Digikam::ImageIface iface(0, 0);
    iface.putOriginalImage(i18n("Noise Reduction"), m_threadedFilter->getTargetImage().bits());
}

void ImageEffect_NoiseReduction::slotUser3()
{
    KURL loadRestorationFile = KFileDialog::getOpenURL(TDEGlobalSettings::documentPath(),
                               TQString( "*" ), this,
                               TQString( i18n("Photograph Noise Reduction Settings File to Load")) );
    if ( loadRestorationFile.isEmpty() )
        return;

    TQFile file(loadRestorationFile.path());

    if ( file.open(IO_ReadOnly) )
    {
        TQTextStream stream( &file );
        if ( stream.readLine() != "# Photograph Noise Reduction Configuration File" )
        {
            KMessageBox::error(this,
                               i18n("\"%1\" is not a Photograph Noise Reduction settings text file.")
                               .arg(loadRestorationFile.fileName()));
            file.close();
            return;
        }

        blockSignals(true);
        m_radiusInput->setValue( stream.readLine().toDouble() );
        m_lumToleranceInput->setValue( stream.readLine().toDouble() );
        m_thresholdInput->setValue( stream.readLine().toDouble() );
        m_textureInput->setValue( stream.readLine().toDouble() );
        m_sharpnessInput->setValue( stream.readLine().toDouble() );
        m_csmoothInput->setValue( stream.readLine().toDouble() );
        m_lookaheadInput->setValue( stream.readLine().toDouble() );
        m_gammaInput->setValue( stream.readLine().toDouble() );
        m_dampingInput->setValue( stream.readLine().toDouble() );
        m_phaseInput->setValue( stream.readLine().toDouble() );
        blockSignals(false);
//         slotEffect();
    }
    else
        KMessageBox::error(this, i18n("Cannot load settings from the Photograph Noise Reduction text file."));

    file.close();
}

void ImageEffect_NoiseReduction::slotUser2()
{
    KURL saveRestorationFile = KFileDialog::getSaveURL(TDEGlobalSettings::documentPath(),
                               TQString( "*" ), this,
                               TQString( i18n("Photograph Noise Reduction Settings File to Save")) );
    if ( saveRestorationFile.isEmpty() )
        return;

    TQFile file(saveRestorationFile.path());

    if ( file.open(IO_WriteOnly) )
    {
        TQTextStream stream( &file );
        stream << "# Photograph Noise Reduction Configuration File\n";
        stream << m_radiusInput->value() << "\n";
        stream << m_lumToleranceInput->value() << "\n";
        stream << m_thresholdInput->value() << "\n";
        stream << m_textureInput->value() << "\n";
        stream << m_sharpnessInput->value() << "\n";
        stream << m_csmoothInput->value() << "\n";
        stream << m_lookaheadInput->value() << "\n";
        stream << m_gammaInput->value() << "\n";
        stream << m_dampingInput->value() << "\n";
        stream << m_phaseInput->value() << "\n";

    }
    else
        KMessageBox::error(this, i18n("Cannot save settings to the Photograph Noise Reduction text file."));

    file.close();
}

}  // NameSpace DigikamNoiseReductionImagesPlugin

