/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-08-26
 * Description : a digiKam image editor plugin to emboss 
 *               an image.
 *
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006-2008 by Marcel Wiesweg <marcel dot wiesweg at gmx dot de>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */ 

// TQt includes.

#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqlayout.h>

// KDE includes.

#include <tdelocale.h>
#include <tdeaboutdata.h>
#include <kiconloader.h>
#include <tdeapplication.h>
#include <tdestandarddirs.h>
#include <knuminput.h>
#include <tdeconfig.h>

// Local includes.

#include "version.h"
#include "imageiface.h"
#include "imagewidget.h"
#include "emboss.h"
#include "imageeffect_emboss.h"
#include "imageeffect_emboss.moc"

namespace DigikamEmbossImagesPlugin
{

ImageEffect_Emboss::ImageEffect_Emboss(TQWidget* parent)
                  : Digikam::CtrlPanelDlg(parent, i18n("Emboss Image"), "emboss", false, false, true,
                                          Digikam::ImagePannelWidget::SeparateViewAll)
{
    TQString whatsThis;

    TDEAboutData* about = new TDEAboutData("digikam",
                                       I18N_NOOP("Emboss Image"), 
                                       digikam_version,
                                       I18N_NOOP("Emboss image effect plugin for digiKam."),
                                       TDEAboutData::License_GPL,
                                       "(c) 2004-2006, Gilles Caulier\n"
                                       "(c) 2006-2008, Gilles Caulier and Marcel Wiesweg",
                                       0,
                                       "http://www.digikam.org");

    about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                     "caulier dot gilles at gmail dot com");

    about->addAuthor("Pieter Z. Voloshyn", I18N_NOOP("Emboss algorithm"), 
                     "pieter dot voloshyn at gmail dot com");         

    about->addAuthor("Marcel Wiesweg", I18N_NOOP("Developer"),
                     "marcel dot wiesweg at gmx dot de");

    setAboutData(about);
    
    // -------------------------------------------------------------
    
    TQWidget *gboxSettings = new TQWidget(m_imagePreviewWidget);
    TQGridLayout* gridSettings = new TQGridLayout( gboxSettings, 1, 1, 0, spacingHint());
    TQLabel *label1 = new TQLabel(i18n("Depth:"), gboxSettings);
    
    m_depthInput = new KIntNumInput(gboxSettings);
    m_depthInput->setRange(10, 300, 1, true);
    TQWhatsThis::add( m_depthInput, i18n("<p>Set here the depth of the embossing image effect.") );
                                            
    gridSettings->addMultiCellWidget(label1, 0, 0, 0, 1);
    gridSettings->addMultiCellWidget(m_depthInput, 1, 1, 0, 1);
    
    m_imagePreviewWidget->setUserAreaWidget(gboxSettings);
    
    // -------------------------------------------------------------
    
    connect(m_depthInput, TQ_SIGNAL(valueChanged (int)),
            this, TQ_SLOT(slotTimer())); 
}

ImageEffect_Emboss::~ImageEffect_Emboss()
{
}

void ImageEffect_Emboss::renderingFinished()
{
    m_depthInput->setEnabled(true);
}

void ImageEffect_Emboss::readUserSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("emboss Tool Dialog");
    m_depthInput->blockSignals(true);
    m_depthInput->setValue(config->readNumEntry("DepthAjustment", 30));
    m_depthInput->blockSignals(false);
}

void ImageEffect_Emboss::writeUserSettings()
{
    TDEConfig* config = tdeApp->config();
    config->setGroup("emboss Tool Dialog");
    config->writeEntry("DepthAjustment", m_depthInput->value());
    config->sync();
}

void ImageEffect_Emboss::resetValues()
{
    m_depthInput->blockSignals(true);
    m_depthInput->setValue(30);
    m_depthInput->blockSignals(false);
} 

void ImageEffect_Emboss::prepareEffect()
{
    m_depthInput->setEnabled(false);

    Digikam::DImg image = m_imagePreviewWidget->getOriginalRegionImage();

    int depth = m_depthInput->value();

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(new Emboss(&image, this, depth));
}

void ImageEffect_Emboss::prepareFinal()
{
    m_depthInput->setEnabled(false);

    int depth = m_depthInput->value();

    Digikam::ImageIface iface(0, 0);
    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>(new Emboss(iface.getOriginalImg(), this, depth));
}

void ImageEffect_Emboss::putPreviewData(void)
{
    m_imagePreviewWidget->setPreviewImage(m_threadedFilter->getTargetImage());
}

void ImageEffect_Emboss::putFinalData(void)
{
    Digikam::ImageIface iface(0, 0);

    iface.putOriginalImage(i18n("Emboss"),
                           m_threadedFilter->getTargetImage().bits());
}

}  // NameSpace DigikamEmbossImagesPlugin

