/* This file is part of TDevelop
    Copyright (C) 2003 Roberto Raggi <roberto@kdevelop.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef TAG_H
#define TAG_H

#include <tqmap.h>
#include <tqvariant.h>
#include <tqshared.h>

class TQDataStream;

union TagFlags
{
    unsigned long flags;
    struct
    {
		unsigned long access:
                        3;
		unsigned long isVirtual:
                        1;
    }
    data;
} ;

class Tag
{
public:
    enum Kind
    {
	Kind_Unknown,

	Kind_Typedef = 1000,
	Kind_Namespace,
	Kind_UsingDirective,
	Kind_Base_class,
	Kind_Enum,
	Kind_Enumerator,
	Kind_Class,
	Kind_Struct,
	Kind_Union,
	Kind_VariableDeclaration,
	Kind_Variable,
	Kind_FunctionDeclaration,
	Kind_Function,
        Kind_NamespaceAlias,
        Kind_TranslationUnit,
    
	// ...

	Kind_Custom = 2000
    };

public:
    Tag();
    Tag( const Tag& source );
    ~Tag();
    
    operator bool() const {
        return kind() != Kind_Unknown && kind() != 0;
    }

    Tag& operator = ( const Tag& source );
    
    TQCString id() const
    {
        return data->id;
    }

    void setId( const TQCString& id )
    {
	detach();
        data->id = id;
    }

    int kind() const
    {
        return data->kind;
    }

    void setKind( int kind )
    {
	detach();
        data->kind = kind;
    }

    unsigned long flags() const
    {
        return data->flags;
    }

    void setFlags( unsigned long flags )
    {
	detach();
        data->flags = flags;
    }

    TQString fileName() const
    {
        return data->fileName;
    }

    void setFileName( const TQString& fileName )
    {
	detach();
        data->fileName = fileName;
    }

    TQString path( const TQString& sep = TQString::fromLatin1("::") ) const
    {
        TQString s = scope().join( sep );
        if( s.isNull() )
            return name();
	return s + sep + name();
    }

    TQString name() const
    {
        return data->name;
    }
    
    TQString comment() const {
        if( hasAttribute( "cmt" ) ) {
            return attribute( "cmt" ).asString();
        } else {
            return "";
        }
    }

    void setComment( const TQString& comment ) {
        setAttribute( "cmt", comment );
    }
    
    void setName( const TQString& name )
    {
	detach();
        data->name = name;
    }
    
    TQStringList scope() const
    {
        return data->scope;
    }

    void setScope( const TQStringList& scope )
    {
	detach();
        data->scope = scope;
    }

    void getStartPosition( int* line, int* column ) const
    {
	if( line ) *line = data->startLine;
	if( column ) *column = data->startColumn;
    }

    void setStartPosition( int line, int column )
    {
	detach();
	data->startLine = line;
	data->startColumn = column;
    }

    void getEndPosition( int* line, int* column ) const
    {
	if( line ) *line = data->endLine;
	if( column ) *column = data->endColumn;
    }

    void setEndPosition( int line, int column )
    {
	detach();
	data->endLine = line;
	data->endColumn = column;
    }

		TQString getSpecializationDeclaration() const {
			if( hasAttribute( "spc" ) )
				return data->attributes["spc"].asString();
			else
				return TQString();
		}

		bool hasSpecializationDeclaration() const {
			return data->attributes.contains( "spc" );
		}

		void setSpecializationDeclaration( const TQString& str ) {
			data->attributes["spc"] = str;
		}

	bool hasAttribute( const TQCString& name ) const
    {
	if( name == "kind" ||
	    name == "name" ||
	    name == "scope" ||
	    name == "fileName" ||
	    name == "startLine" ||
	    name == "startColumn" ||
	    name == "endLine" ||
	    name == "endColumn" )
	    return true;
        return data->attributes.contains( name );
    }

    TQVariant attribute( const TQCString& name ) const
    {
	if( name == "id" )
	    return data->id;
	else if( name == "kind" )
	    return data->kind;
	else if( name == "name" )
	    return data->name;
	else if( name == "scope" )
	    return data->scope;
	else if( name == "fileName" )
	    return data->fileName;
	else if( name == "startLine" )
	    return data->startLine;
	else if( name == "startColumn" )
	    return data->startColumn;
	else if( name == "endLine" )
	    return data->endLine;
	else if( name == "endColumn" )
	    return data->endColumn;
	else if( name == "prefix" )
	    return data->name.left( 2 );
        return data->attributes[ name ];
    }

    void setAttribute( const TQCString& name, const TQVariant& value )
    {
	detach();
	if( name == "id" )
	    data->id = value.toCString();
	else if( name == "kind" )
	    data->kind = value.toInt();
	else if( name == "name" )
	    data->name = value.toString();
	else if( name == "scope" )
	    data->scope = value.toStringList();
	else if( name == "fileName" )
	    data->fileName = value.toString();
	else if( name == "startLine" )
	    data->startLine = value.toInt();
	else if( name == "startColumn" )
	    data->startColumn = value.toInt();
	else if( name == "endLine" )
	    data->endLine = value.toInt();
	else if( name == "endColumn" )
	    data->endColumn = value.toInt();
	else
	    data->attributes[ name ] = value;
    }
    
    void addTemplateParam( const TQString& param , const TQString& def = "" ) {
        TQMap<TQCString, TQVariant>::iterator it = data->attributes.find( "tpl" );
        if( it != data->attributes.end() && (*it).type() == TQVariant::StringList ) {
        }else{
            it = data->attributes.insert( "tpl", TQVariant( TQStringList() ) );
        }
        
        TQStringList& l( (*it).asStringList() );
        l << param;
        l << def;
    }
    
    void load( TQDataStream& stream );
    void store( TQDataStream& stream ) const;

private:
    Tag copy();
    void detach();
    
private:
    struct TagData: public TQShared
    {
	TQCString id;
	int kind;
	unsigned long flags;
	TQString name;
	TQStringList scope;
	TQString fileName;
	int startLine, startColumn;
	int endLine, endColumn;
	TQMap<TQCString, TQVariant> attributes;
    } *data;
};

TQDataStream& operator << ( TQDataStream&, const Tag& );
TQDataStream& operator >> ( TQDataStream&, Tag& );

#endif
