/*
** Copyright (C) 1999,2000 Toivo Pedaste <toivo@ucs.uwa.edu.au>
**
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "kpackage.h"
#include "options.h"
#include "cache.h"
#include <tdelocale.h>
#include <kdebug.h>

extern Opts *opts;
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
cacheObj::cacheObj(const TQString &Pbase, const TQString &Plocation, const TQString &PcacheFile,
		   const TQString &Poption, bool Psubdirs)
{
  base = Pbase;
  location = Plocation;
  cacheFile = PcacheFile;
  option = Poption;
  subdirs = Psubdirs;
}

cacheObj::~cacheObj()
{}

TQString cacheObj::PDir()
{
  struct stat buf;
  stat(TQFile::encodeName(TQDir::homeDirPath()), &buf);

  TQString tmpd = opts->CacheDir ;

  TQDir d(tmpd);
  if (!d.exists()) {
    if (!d.mkdir(tmpd)) {
      KpMsgE(i18n("Cannot create folder %1").arg(tmpd),true);
      tmpd = "";
    } else {
      chown(TQFile::encodeName(tmpd),buf.st_uid,buf.st_gid);
    }
  }
  return tmpd;
}

TQString cacheObj::CDir()
{
  TQString tmpd = PDir();
  if (!tmpd.isEmpty()) {
    struct stat buf;
    stat(TQFile::encodeName(tmpd),&buf);

    tmpd += "dir/";

    TQDir d(tmpd);
    if (!d.exists()) {
      if (!d.mkdir(tmpd)) {
	KpMsgE(i18n("Cannot create folder %1").arg(tmpd),true);
	tmpd = "";
      } else {
	chown(TQFile::encodeName(tmpd),buf.st_uid,buf.st_gid);
      }
    }
  }
  return tmpd;
}

int cacheObj::newDCache(const TQString &url, const TQString &fn, TQString  &fname) {

  KURL u(url);
  if ( !u.isValid() ) {
    KpMsgE(i18n("Malformed URL: %1").arg(url),true);
    return -1;
  }

  TQString tmpd = cacheObj::CDir();
  if (tmpd.isEmpty()) {
    return -1;
  } else {
    if (u.protocol() == "file") {
      fname =  u.path();
      return 0;
    }

    fname = tmpd + fn;

    if (opts->DCache == Opts::NEVER) {
      return 1;
    }

    TQFileInfo f(fname);

    if (f.exists() && f.size() > 0) {
      return 0;;
    } else {
      if (f.size() == 0)
	rmDCache(fname);
      return 1;
    }
  }
}

void  cacheObj::rmDCache(const TQString &fn) {
  TQString tmpd = cacheObj::CDir();
  tmpd += fn;

  if (!tmpd.isEmpty()) {
    unlink(TQFile::encodeName(tmpd));
  }
}

void  cacheObj::clearDCache() {
  TQString tmpd = cacheObj::CDir();

  if (!tmpd.isEmpty()) {
    TQDir d(tmpd);
    CacheList cl(d) ;
    for (CacheList::iterator it = cl.begin() ; it != cl.end() ; ++it) {
      TQString s = tmpd;
      s += *it;
      unlink(TQFile::encodeName(s));
    }
    unlink(TQFile::encodeName(cl.getCLFileName())) ; // also delete the kpackage_cachelist file
  }
}

void  cacheObj::clearPCache() {
  TQString tmpd = cacheObj::PDir();

  if (!tmpd.isEmpty()) {
    TQDir d(tmpd);
    CacheList cl(d);
    for (CacheList::iterator it = cl.begin() ; it != cl.end() ; ++it) {
      TQString s = tmpd ;
      s += *it;
      unlink(TQFile::encodeName(s));
    }
    unlink(TQFile::encodeName(cl.getCLFileName())) ; // also delete the kpackage_cachelist file
  }
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
LcacheObj::LcacheObj()
{
  setAutoDelete(true);
}

LcacheObj::~LcacheObj()
{
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
CacheList::CacheList (const TQDir& dir)
{
  CLFile.setName (dir.path() + "/kpackage_cachelist") ;
  read() ;
}

void CacheList::read ()
{
  kdDebug() << "reading cachelist: " << CLFile.name() << "\n" ;
  if (CLFile.open (IO_ReadOnly)) {
    TQTextStream stream (&CLFile) ;
    TQString line ;
    while (!stream.eof()) {
      line = stream.readLine() ;
      if (line[0] != '#') { // not a comment
        append (line) ; // to this TQStringList
      }
    }
    CLFile.close() ;
  }
  else {
    //      kdDebug() << "could not open cachelist " << CLFile.name() << "!\n" ;
  }
}

void CacheList::write ()
{
  kdDebug() << "writing cachelist: " << CLFile.name() << "\n" ;
  if (CLFile.open (IO_WriteOnly)) {
    TQTextStream stream (&CLFile) ;
    stream << "# This file contains a list of files that have been cached in this folder.\n" ;
    stream << "# Please only delete this if you want kpackage to forget what it has cached.\n" ;
    for (TQStringList::iterator it = begin() ; it != end() ; ++it) {
      stream << *it << "\n" ;
    }
	CLFile.close() ;
  }
  else {
      kdDebug() << "could not open cachelist " << CLFile.name() << "!\n" ;
  }
}

TQString CacheList::getCLFileName () const
{
    return CLFile.name() ;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
