/* This file is part of the KDE project
   Copyright (C) 2004 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KEXISHAREDACTIONHOST_H
#define KEXISHAREDACTIONHOST_H

#include <tqguardedptr.h>
#include <tqasciidict.h>
#include <tqobject.h>
#include <tqpair.h>

#include <kstdaction.h>
#include <tdeaction.h>

class TDEShortcut;
class KGuiItem;
class TDEMainWindow;
class KexiActionProxy;
class KexiSharedActionHostPrivate;

namespace KexiPart {
	class Part;
}

//! Acts as application-wide host that offers shared actions.
/*!
 You can inherit this class together with TDEMainWindow (or any TDEMainWindow).
 Call setAsDefaultHost() to make the host default for all shared actions that have 
 not explicitly specified host.

 For example how all this is done, see KexiMainWindow implementation.

 \sa KexiActionProxy, KexiMainWindow
*/

class KEXICORE_EXPORT KexiSharedActionHost
{
	public:

		/*! Constructs host for main window \a mainWin. */
		KexiSharedActionHost(TDEMainWindow* mainWin);

		virtual ~KexiSharedActionHost();

		/*! \return true if \a w can accept shared actions. 
		 This method is used by focusWindow() to look up widgets hierarchy 
		 for widget that can accept shared actions. 
		 Default implementation always returns false. 
		 You can reimplement it e.g. like in KexiMainWindowImpl::acceptsSharedActions():
		 \code
			return o->inherits("KexiDialogBase") || o->inherits("KexiViewBase");
		 \endcode
		 */
		virtual bool acceptsSharedActions(TQObject *o);

		/*! \return window widget that is currently focused (using TQWidget::focusWidget())
		 and matches acceptsSharedActions(). If focused widget does not match,
		 it's parent, grandparent, and so on is checked. If all this fails, 
		 or no widget has focus, NULL is returned. 
		 Also works if currently focused window is detached (as in KMDI).
		 */
		TQWidget* focusWindow();

		/*! Sets this host as default shared actions host. */
		void setAsDefaultHost();

		/*! \return default shared actions host, used when no host 
		 is explicitly specified for shared actions.
		 There can be exactly one deault shared actions host. */
		static KexiSharedActionHost& defaultHost();

		/*! \return shared actions list. */
		TDEActionPtrList sharedActions() const;

		/*! PROTOTYPE, DO NOT USE YET */
		void setActionVolatile( TDEAction *a, bool set );


	protected:

		/*! Invalidates all shared actions declared using createSharedAction().
		 Any shared action will be enabled if \a o (typically: a child window or a dock window)
		 has this action plugged _and_ it is available (i.e. enabled). 
		 Otherwise the action is disabled.

		 If \a o is not KexiDialogBase or its child,
		 actions are only invalidated if these come from mainwindow's TDEActionCollection
		 (thus part-actions are untouched when the focus is e.g. in the Property Editor.

		 Call this method when it is known that some actions need invalidation 
		 (e.g. when new window is activated). See how it is used in KexiMainWindowImpl.
		*/
		virtual void invalidateSharedActions(TQObject *o);

		void setActionAvailable(const char *action_name, bool avail);

		/*! Plugs shared actions proxy \a proxy for this host. */
		void plugActionProxy(KexiActionProxy *proxy);

		/*! Updates availability of action \a action_name for object \a obj.
		 Object is mainly the window. */
		void updateActionAvailable(const char *action_name, bool avail, TQObject *obj);

		/*! \return main window for which this host is defined. */
		TDEMainWindow* mainWindow() const;

		/*! Creates shared action using \a text, \a pix_name pixmap, shortcut \a cut,
		 optional \a name. You can pass your own action collection as \a col. 
		 If \a col action collection is null, main window's action will be used.
		 Pass desired TDEAction subclass with \a subclassName (e.g. "TDEToggleAction") to have
		 that subclass allocated instead just TDEAction (what is the default).
		 Created action's data is owned by the main window. */
		TDEAction* createSharedAction(const TQString &text, const TQString &pix_name, 
			const TDEShortcut &cut, const char *name, TDEActionCollection* col = 0,
			const char *subclassName = 0);

		/*! Like above - creates shared action, but from standard action identified by \a id. 
		 Action's data is owned by the main window. */
		TDEAction* createSharedAction( KStdAction::StdAction id, const char *name, 
			TDEActionCollection* col = 0);

		/*! Creates shared action with name \a name and shortcut \a cut 
		 by copying all properties of \a guiItem. 
		 If \a col action collection is null, main window's action will be used. */
		TDEAction* createSharedAction(const KGuiItem& guiItem, const TDEShortcut &cut, const char *name,
			TDEActionCollection* col = 0);

		/*! \return action proxy for object \a o, or NULL if this object has 
		 no plugged shared actions. */
		KexiActionProxy* actionProxyFor(TQObject *o) const;

		/*! Like actionProxyFor(), but takes the proxy from the host completely.
		 This is called by KExiActionProxy on its destruction. */
		KexiActionProxy* takeActionProxyFor(TQObject *o);

	private:
		/*! Helper function for createSharedAction(). */
		TDEAction* createSharedActionInternal( TDEAction *action );

		KexiSharedActionHostPrivate *d;

	friend class KexiActionProxy;
	friend class KexiPart::Part;
	friend class KexiViewBase;
	friend class KexiDialogBase;
};

#endif

