/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "UseOrnamentDialog.h"
#include <tqlayout.h>
#include <tdeapplication.h>

#include "base/BaseProperties.h"
#include <tdelocale.h>
#include "misc/Strings.h"
#include "document/ConfigGroups.h"
#include "base/Composition.h"
#include "base/NotationTypes.h"
#include "base/TriggerSegment.h"
#include "gui/editors/notation/NotePixmapFactory.h"
#include <kcombobox.h>
#include <tdeconfig.h>
#include <kdialogbase.h>
#include <tqcheckbox.h>
#include <tqframe.h>
#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqlineedit.h>
#include <tqstring.h>
#include <tqvbox.h>
#include <tqwidget.h>


namespace Rosegarden
{

UseOrnamentDialog::UseOrnamentDialog(TQWidget *parent,
                                     Composition *composition) :
        KDialogBase(parent, "useornamentdialog", true, i18n("Use Ornament"),
                    Ok | Cancel, Ok),
        m_composition(composition)
{
    TQVBox *vbox = makeVBoxMainWidget();
    TQLabel *label;

    TQGroupBox *notationBox = new TQGroupBox(1, Qt::Horizontal, i18n("Notation"), vbox);

    TQFrame *frame = new TQFrame(notationBox);
    TQGridLayout *layout = new TQGridLayout(frame, 4, 1, 5, 5);

    label = new TQLabel(i18n("Display as:  "), frame);
    layout->addWidget(label, 0, 0);

    m_mark = new KComboBox(frame);
    layout->addWidget(m_mark, 0, 1);

    m_marks.push_back(Marks::Trill);
    m_marks.push_back(Marks::LongTrill);
    m_marks.push_back(Marks::TrillLine);
    m_marks.push_back(Marks::Turn);
    m_marks.push_back(Marks::Mordent);
    m_marks.push_back(Marks::MordentInverted);
    m_marks.push_back(Marks::MordentLong);
    m_marks.push_back(Marks::MordentLongInverted);

    const TQString markLabels[] = {
                                     i18n("Trill"), i18n("Trill with line"), i18n("Trill line only"),
                                     i18n("Turn"), i18n("Mordent"), i18n("Inverted mordent"),
                                     i18n("Long mordent"), i18n("Long inverted mordent"),
                                 };

    for (size_t i = 0; i < m_marks.size(); ++i) {
        m_mark->insertItem(NotePixmapFactory::toTQPixmap
                           (NotePixmapFactory::makeMarkMenuPixmap(m_marks[i])),
                           markLabels[i]);
    }
    m_mark->insertItem(i18n("Text mark"));

    connect(m_mark, TQT_SIGNAL(activated(int)), this, TQT_SLOT(slotMarkChanged(int)));

    m_textLabel = new TQLabel(i18n("   Text:  "), frame);
    layout->addWidget(m_textLabel, 0, 2);

    m_text = new TQLineEdit(frame);
    layout->addWidget(m_text, 0, 3);

    TQGroupBox *performBox = new TQGroupBox(1, Qt::Horizontal, i18n("Performance"), vbox);

    frame = new TQFrame(performBox);
    layout = new TQGridLayout(frame, 3, 2, 5, 5);

    label = new TQLabel(i18n("Perform using triggered segment: "), frame);
    layout->addWidget(label, 0, 0);

    m_ornament = new KComboBox(frame);
    layout->addWidget(m_ornament, 0, 1);

    int n = 1;
    for (Composition::triggersegmentcontaineriterator i =
                m_composition->getTriggerSegments().begin();
            i != m_composition->getTriggerSegments().end(); ++i) {
        m_ornament->insertItem
        (TQString("%1. %2").arg(n++).arg(strtoqstr((*i)->getSegment()->getLabel())));
    }

    label = new TQLabel(i18n("Perform with timing: "), frame);
    layout->addWidget(label, 1, 0);

    m_adjustTime = new KComboBox(frame);
    layout->addWidget(m_adjustTime, 1, 1);

    m_adjustTime->insertItem(i18n("As stored"));
    m_adjustTime->insertItem(i18n("Truncate if longer than note"));
    m_adjustTime->insertItem(i18n("End at same time as note"));
    m_adjustTime->insertItem(i18n("Stretch or squash segment to note duration"));

    m_retune = new TQCheckBox(i18n("Adjust pitch to note"), frame);
    m_retune->setChecked(true);

    layout->addWidget(m_retune, 2, 1);

    setupFromConfig();
}

void
UseOrnamentDialog::setupFromConfig()
{
    TDEConfig *config = kapp->config();
    config->setGroup(NotationViewConfigGroup);

    Mark mark = qstrtostr(config->readEntry("useornamentmark", "trill"));
    int seg = config->readNumEntry("useornamentlastornament", 0);
    std::string timing = qstrtostr
                         (config->readEntry
                          ("useornamenttiming",
                           strtoqstr(BaseProperties::TRIGGER_SEGMENT_ADJUST_SQUISH)));
    bool retune = config->readBoolEntry("useornamentretune", true);

    size_t i = 0;
    for (i = 0; i < m_marks.size(); ++i) {
        if (mark == m_marks[i]) {
            m_mark->setCurrentItem(i);
            m_text->setEnabled(false);
            break;
        }
    }
    if (i >= m_marks.size()) {
        m_mark->setCurrentItem(m_marks.size());
        m_text->setEnabled(true);
        m_text->setText(strtoqstr(Marks::getTextFromMark(mark)));
    }

    if (seg >= 0 && seg < m_ornament->count())
        m_ornament->setCurrentItem(seg);

    if (timing == BaseProperties::TRIGGER_SEGMENT_ADJUST_NONE) {
        m_adjustTime->setCurrentItem(0);
    } else if (timing == BaseProperties::TRIGGER_SEGMENT_ADJUST_SQUISH) {
        m_adjustTime->setCurrentItem(3);
    } else if (timing == BaseProperties::TRIGGER_SEGMENT_ADJUST_SYNC_START) {
        m_adjustTime->setCurrentItem(1);
    } else if (timing == BaseProperties::TRIGGER_SEGMENT_ADJUST_SYNC_END) {
        m_adjustTime->setCurrentItem(2);
    }

    m_retune->setChecked(retune);
}

TriggerSegmentId
UseOrnamentDialog::getId() const
{
    int ix = m_ornament->currentItem();

    for (Composition::triggersegmentcontaineriterator i =
                m_composition->getTriggerSegments().begin();
            i != m_composition->getTriggerSegments().end(); ++i) {

        if (ix == 0)
            return (*i)->getId();
        --ix;
    }

    return 0;
}

Mark
UseOrnamentDialog::getMark() const
{
    if (int(m_marks.size()) > m_mark->currentItem())
        return m_marks[m_mark->currentItem()];
    else
        return Marks::getTextMark(qstrtostr(m_text->text()));
}

bool
UseOrnamentDialog::getRetune() const
{
    return m_retune->isChecked();
}

std::string
UseOrnamentDialog::getTimeAdjust() const
{
    int option = m_adjustTime->currentItem();

    switch (option) {

    case 0:
        return BaseProperties::TRIGGER_SEGMENT_ADJUST_NONE;
    case 1:
        return BaseProperties::TRIGGER_SEGMENT_ADJUST_SYNC_START;
    case 2:
        return BaseProperties::TRIGGER_SEGMENT_ADJUST_SYNC_END;
    case 3:
        return BaseProperties::TRIGGER_SEGMENT_ADJUST_SQUISH;

    default:
        return BaseProperties::TRIGGER_SEGMENT_ADJUST_NONE;
    }
}

void
UseOrnamentDialog::slotMarkChanged(int i)
{
    if (i == 2) {
        m_text->setEnabled(true);
    } else {
        m_text->setEnabled(false);
    }
}

void
UseOrnamentDialog::slotOk()
{
    TDEConfig *config = kapp->config();
    config->setGroup(NotationViewConfigGroup);

    config->writeEntry("useornamentmark", strtoqstr(getMark()));
    config->writeEntry("useornamenttiming", strtoqstr(getTimeAdjust()));
    config->writeEntry("useornamentretune", m_retune->isChecked());
    config->writeEntry("useornamentlastornament", m_ornament->currentItem());

    accept();
}

}
#include "UseOrnamentDialog.moc"
