/***************************************************************************
*   Copyright (C) 2004 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*   Unai Garro (ugarro@users.sourceforge.net)                             *
*   Cyril Bosselut (bosselut@b1project.com)                               *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "headerlistview.h"

#include <tdemessagebox.h>
#include <tdeconfig.h>
#include <tdelocale.h>
#include <tdeglobal.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>

#include "backends/recipedb.h"
#include "dialogs/createelementdialog.h"
#include "dialogs/dependanciesdialog.h"

HeaderListView::HeaderListView( TQWidget *parent, RecipeDB *db ) : DBListViewBase( parent,db,db->unitCount() )
{
	setAllColumnsShowFocus( true );
	setDefaultRenameAction( TQListView::Reject );
}

void HeaderListView::init()
{
	connect( database, TQ_SIGNAL( ingGroupCreated( const Element & ) ), TQ_SLOT( checkCreateHeader( const Element & ) ) );
	connect( database, TQ_SIGNAL( ingGroupRemoved( int ) ), TQ_SLOT( removeHeader( int ) ) );
}

void HeaderListView::load( int /*limit*/, int /*offset*/ )
{
	ElementList headerList;
	database->loadIngredientGroups( &headerList );

	setTotalItems(headerList.count());

	for ( ElementList::const_iterator it = headerList.begin(); it != headerList.end(); ++it ) {
		createHeader( *it );
	}
}

void HeaderListView::checkCreateHeader( const Element &el )
{
	if ( handleElement(el.name) ) { //only create this header if the base class okays it
		createHeader(el);
	}
}


StdHeaderListView::StdHeaderListView( TQWidget *parent, RecipeDB *db, bool editable ) : HeaderListView( parent, db )
{
	addColumn( i18n( "Header" ) );

	TDEConfig * config = TDEGlobal::config();
	config->setGroup( "Advanced" );
	bool show_id = config->readBoolEntry( "ShowID", false );
	addColumn( i18n( "Id" ), show_id ? -1 : 0 );

	if ( editable ) {
		setRenameable( 0, true );

		TDEIconLoader *il = new TDEIconLoader;

		kpop = new TDEPopupMenu( this );
		kpop->insertItem( il->loadIcon( "document-new", TDEIcon::NoGroup, 16 ), i18n( "&Create" ), this, TQ_SLOT( createNew() ), CTRL + Key_C );
		kpop->insertItem( il->loadIcon( "edit-delete", TDEIcon::NoGroup, 16 ), i18n( "&Delete" ), this, TQ_SLOT( remove
			                  () ), Key_Delete );
		kpop->insertItem( il->loadIcon( "edit", TDEIcon::NoGroup, 16 ), i18n( "&Rename" ), this, TQ_SLOT( rename() ), CTRL + Key_R );
		kpop->polish();

		delete il;

		connect( this, TQ_SIGNAL( contextMenu( TDEListView *, TQListViewItem *, const TQPoint & ) ), TQ_SLOT( showPopup( TDEListView *, TQListViewItem *, const TQPoint & ) ) );
		connect( this, TQ_SIGNAL( doubleClicked( TQListViewItem*, const TQPoint &, int ) ), this, TQ_SLOT( modHeader( TQListViewItem*, const TQPoint &, int ) ) );
		connect( this, TQ_SIGNAL( itemRenamed( TQListViewItem*, const TQString &, int ) ), this, TQ_SLOT( saveHeader( TQListViewItem*, const TQString &, int ) ) );
	}
}

void StdHeaderListView::showPopup( TDEListView * /*l*/, TQListViewItem *i, const TQPoint &p )
{
	if ( i )
		kpop->exec( p );
}

void StdHeaderListView::createNew()
{
	CreateElementDialog * headerDialog = new CreateElementDialog( this, i18n("Header") );

	if ( headerDialog->exec() == TQDialog::Accepted ) {
		TQString result = headerDialog->newElementName();

		//check bounds first
		if ( checkBounds( result ) )
			database->createNewIngGroup( result );
	}
	delete headerDialog;
}

void StdHeaderListView::remove()
{
	// Find selected header item
	TQListViewItem * it = selectedItem();

	if ( it ) {
		int headerID = it->text( 1 ).toInt();

		ElementList recipeDependancies;
		database->findUseOfIngGroupInRecipes( &recipeDependancies, headerID );

		if ( recipeDependancies.isEmpty() )
			database->removeIngredientGroup( headerID );
		else { // need warning!	
			ListInfo info;
			info.list = recipeDependancies;
			info.name = i18n( "Recipes" );

			DependanciesDialog warnDialog( this, info );
			warnDialog.setCustomWarning( i18n("You are about to permanantly delete recipes from your database.") );
			if ( warnDialog.exec() == TQDialog::Accepted )
				database->removeIngredientGroup( headerID );
		}
	}
}

void StdHeaderListView::rename()
{
	TQListViewItem * item = currentItem();

	if ( item )
		HeaderListView::rename( item, 0 );
}

void StdHeaderListView::createHeader( const Element &header )
{
	createElement(new TQListViewItem( this, header.name, TQString::number( header.id ) ));
}

void StdHeaderListView::removeHeader( int id )
{
	TQListViewItem * item = findItem( TQString::number( id ), 1 );
	removeElement(item);
}

void StdHeaderListView::modHeader( TQListViewItem* i, const TQPoint & /*p*/, int c )
{
	if ( i )
		HeaderListView::rename( i, c );
}

void StdHeaderListView::saveHeader( TQListViewItem* i, const TQString &text, int /*c*/ )
{
	if ( !checkBounds( text ) ) {
		reload(ForceReload); //reset the changed text
		return ;
	}

	int existing_id = database->findExistingIngredientGroupByName( text );
	int header_id = i->text( 1 ).toInt();
	if ( existing_id != -1 && existing_id != header_id ) { //already exists with this label... merge the two
		switch ( KMessageBox::warningContinueCancel( this, i18n( "This header already exists.  Continuing will merge these two headers into one.  Are you sure?" ) ) ) {
		case KMessageBox::Continue: {
				database->modIngredientGroup( header_id, i->text( 0 ) );
				database->mergeIngredientGroups( header_id, existing_id );
				break;
			}
		default:
			reload(ForceReload);
			break;
		}
	}
	else {
		database->modIngredientGroup( header_id, i->text( 0 ) );
	}
}

bool StdHeaderListView::checkBounds( const TQString &header )
{
	if ( header.length() > uint(database->maxIngGroupNameLength()) ) {
		KMessageBox::error( this, TQString( i18n( "Header cannot be longer than %1 characters." ) ).arg( database->maxIngGroupNameLength() ) );
		return false;
	}
	else if ( header.stripWhiteSpace().isEmpty() )
		return false;

	return true;
}

#include "headerlistview.moc"
