/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#include "manager.h"
#include "entry.h"

#include <tqfile.h>

#include <kstandarddirs.h>
#include <kstaticdeleter.h>
#include <tdepopupmenu.h>
#include <tdemessagebox.h>
#include <tdelocale.h>

namespace KFTPCore {

namespace CustomCommands {

Manager *Manager::m_self = 0;
static KStaticDeleter<Manager> staticManagerDeleter;

Manager *Manager::self()
{
  if (!m_self) {
    staticManagerDeleter.setObject(m_self, new Manager());
  }
  
  return m_self;
}

Manager::Manager()
  : TQObject()
{
  // Populate the handlers list
  m_handlers["Raw"] = new Handlers::RawHandler();
  m_handlers["Substitute"] = new Handlers::SubstituteHandler();
  m_handlers["Regexp"] = new Handlers::RegexpHandler();
}

Manager::~Manager()
{
  if (m_self == this)
    staticManagerDeleter.setObject(m_self, 0, false);
  
  // Destroy the handlers
  delete static_cast<Handlers::RawHandler*>(m_handlers["Raw"]);
  delete static_cast<Handlers::SubstituteHandler*>(m_handlers["Substitute"]);
  delete static_cast<Handlers::RegexpHandler*>(m_handlers["Regexp"]);
  
  m_handlers.clear();
}

void Manager::load()
{
  TQString filename = locateLocal("appdata", "commands.xml");
  
  if (!TQFile::exists(filename)) {
    // Copy the default command set over
    TQFile source(locate("appdata", "commands.xml"));
    TQFile destination(filename);
    
    source.open(IO_ReadOnly);
    destination.open(IO_WriteOnly | IO_Truncate);
    
    destination.writeBlock(source.readAll());
    source.close();
    destination.close();
  }
  
  TQFile file(filename);
  if (!file.open(IO_ReadOnly))
    return;
  
  m_document.setContent(&file);
  file.close();
}

void Manager::parseEntries(TDEActionMenu *parentMenu, const TQDomNode &parentNode, KFTPSession::Session *session) const
{
  TQDomNode n = parentNode.firstChild();
  
  while (!n.isNull()) {
    if (n.isElement()) {
      TQDomElement e = n.toElement();
      TQString tagName = e.tagName();
      TQString name = e.attribute("name");
      
      if (tagName == "category") {
        TDEActionMenu *menu = new TDEActionMenu(name, "folder", parentMenu);
        parentMenu->insert(menu);
        
        // Recurse into this category
        parseEntries(menu, n, session);
      } else if (tagName == "entry") {
        Entry *entry = new Entry((Manager*) this, name);
        entry->setDescription(n.namedItem("description").toElement().text());
        entry->setIcon(e.attribute("icon"));
        entry->setCommand(n.namedItem("command").toElement().text());
        
        TQDomNode p = n.namedItem("params").firstChild();
        while (!p.isNull()) {
          TQDomElement pElement = p.toElement();
          
          if (pElement.tagName() == "param") {
            TQString typeString = pElement.attribute("type");
            Entry::ParameterType type = Entry::String;
            
            if (typeString == "String")
              type = Entry::String;
            else if (typeString == "Password")
              type = Entry::Password;
            else if (typeString == "Integer")
              type = Entry::Integer;
            
            entry->appendParameter(type, pElement.text());
          }
          
          p = p.nextSibling();
        }
        
        TQDomElement rElement = n.namedItem("response").toElement();
        entry->setResponseHandler(rElement.attribute("handler"), rElement);
        
        TQString displayString = rElement.attribute("display");
        Entry::DisplayType displayType = Entry::None;
        
        if (displayString == "None")
          displayType = Entry::None;
        else if (displayString == "Window")
          displayType = Entry::Window;
        else if (displayString == "MessageBox")
          displayType = Entry::MessageBox;
          
        entry->setDisplayType(displayType);
        
        // Create a new action
        EntryAction *action = new EntryAction(entry, session);
        connect(action, TQ_SIGNAL(activated()), this, TQ_SLOT(slotActionActivated()));
        
        parentMenu->insert(action);
      } else if (tagName == "separator") {
        parentMenu->popupMenu()->insertSeparator();
      } else {
        KMessageBox::error(0, i18n("Unknown tag while parsing custom site commands!"));
      }
    }
    
    n = n.nextSibling();
  }
}

Handlers::Handler *Manager::handler(const TQString &name) const
{
  if (m_handlers.contains(name))
    return m_handlers[name];
  
  return 0;
}

TDEActionMenu *Manager::categories(const TQString &name, KFTPSession::Session *session) const
{
  TDEActionMenu *actionMenu = new TDEActionMenu(name);
  parseEntries(actionMenu, m_document.documentElement(), session);
  
  return actionMenu;
}

void Manager::slotActionActivated()
{
  EntryAction *action = (EntryAction*) TQObject::sender();
  action->entryInfo()->execute(action->session());
}

}

}

#include "manager.moc"

