/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "itemeffectsmanager.h"
#include <kiconeffect.h>
#include <tdeapplication.h>
#include <tqobjectlist.h>
#include <tdeglobalsettings.h>
#include <tqclipboard.h>
#include <kurldrag.h>
#include <tdelocale.h>

#include "dolphin.h"
#include "dolphinstatusbar.h"

ItemEffectsManager::ItemEffectsManager()
{
    m_pixmapCopy = new TQPixmap();
}

ItemEffectsManager::~ItemEffectsManager()
{
    delete m_pixmapCopy;
    m_pixmapCopy = 0;

    m_highlightedURL = 0;
}

void ItemEffectsManager::zoomIn()
{
    Dolphin::mainWin().refreshViews();
}

void ItemEffectsManager::zoomOut()
{
    Dolphin::mainWin().refreshViews();
}

void ItemEffectsManager::activateItem(void* context)
{
    KFileItem* fileInfo = contextFileInfo(context);
    const KURL itemURL(fileInfo->url());
    if (m_highlightedURL == itemURL) {
        // the item is already highlighted
        return;
    }

    resetActivatedItem();

    const TQPixmap* itemPixmap = contextPixmap(context);
    if (itemPixmap != 0) {
        // remember the pixmap and item to be able to
        // restore it to the old state later
        *m_pixmapCopy = *itemPixmap;
        m_highlightedURL = itemURL;

        // apply an icon effect to the item below the mouse pointer
        TDEIconEffect iconEffect;
        TQPixmap pixmap = iconEffect.apply(*itemPixmap,
                                          TDEIcon::Desktop,
                                          TDEIcon::ActiveState);
        setContextPixmap(context, pixmap);
    }

    if (!Dolphin::mainWin().activeView()->hasSelection()) {
        DolphinStatusBar* statusBar = Dolphin::mainWin().activeView()->statusBar();
        statusBar->setMessage(statusBarText(fileInfo), DolphinStatusBar::Default);
    }
}

void ItemEffectsManager::resetActivatedItem()
{
    if (m_highlightedURL.isEmpty()) {
        return;
    }

    for (void* context = firstContext(); context != 0; context = nextContext(context)) {
        KURL itemURL(contextFileInfo(context)->url());
        if (itemURL == m_highlightedURL) {
            // the highlighted item has been found and is restored to the default state
            TDEIconEffect iconEffect;
            TQPixmap pixmap = iconEffect.apply(*m_pixmapCopy,
                                              TDEIcon::Desktop,
                                              TDEIcon::DefaultState);

            // TODO: KFileIconView does not emit any signal when the preview has been finished.
            // Hence check the size to prevent that a preview is hidden by restoring a
            // non-preview pixmap.
            const TQPixmap* highlightedPixmap = contextPixmap(context);
            const bool restore = (pixmap.width() == highlightedPixmap->width()) &&
                                 (pixmap.height() == highlightedPixmap->height());
            if (restore) {
                setContextPixmap(context, pixmap);
            }
            break;
        }
    }

    m_highlightedURL = 0;

    DolphinStatusBar* statusBar = Dolphin::mainWin().activeView()->statusBar();
    statusBar->clear();
}

void ItemEffectsManager::updateDisabledItems()
{
    if (!m_disabledItems.isEmpty()) {
        // restore all disabled items with their original pixmap
        for (void* context = firstContext(); context != 0; context = nextContext(context)) {
            const KFileItem* fileInfo = contextFileInfo(context);
            const KURL& fileURL = fileInfo->url();
            TQValueListIterator<DisabledItem> it = m_disabledItems.begin();
            while (it != m_disabledItems.end()) {
                if (fileURL == (*it).url) {
                    setContextPixmap(context, (*it).pixmap);
                }
                ++it;
            }
        }
        m_disabledItems.clear();
    }

    if (!Dolphin::mainWin().clipboardContainsCutData()) {
        return;
    }

    TQClipboard* clipboard = TQApplication::clipboard();
    TQMimeSource* data = clipboard->data();
    if (!KURLDrag::canDecode(data)) {
        return;
    }

    // The clipboard contains items, which have been cutted. Change the pixmaps of all those
    // items to the disabled state.
    KURL::List urls;
    KURLDrag::decode(data, urls);
    for (void* context = firstContext(); context != 0; context = nextContext(context)) {
        const KFileItem* fileInfo = contextFileInfo(context);
        const KURL& fileURL = fileInfo->url();
        for(KURL::List::ConstIterator it = urls.begin(); it != urls.end(); ++it) {
            if (fileURL == (*it)) {
                const TQPixmap* itemPixmap = contextPixmap(context);
                if (itemPixmap != 0) {
                    // remember old pixmap
                    DisabledItem disabledItem;
                    disabledItem.url = fileURL;
                    disabledItem.pixmap = *itemPixmap;
                    m_disabledItems.append(disabledItem);

                    TDEIconEffect iconEffect;
                    TQPixmap disabledPixmap = iconEffect.apply(*itemPixmap,
                                                              TDEIcon::Desktop,
                                                              TDEIcon::DisabledState);
                    setContextPixmap(context, disabledPixmap);
                }
                break;
            }
        }
    }
}

TQString ItemEffectsManager::statusBarText(KFileItem* fileInfo) const
{
    if (fileInfo->isDir()) {
        // KFileItem::getStatusBar() returns "MyDocuments/ Folder" as
        // status bar text for a folder 'MyDocuments'. This is adjusted
        // to "MyDocuments (Folder)" in Dolphin.
        return i18n("%1 (Folder)").arg(fileInfo->name());
    }

    return fileInfo->getStatusBarInfo();
}
