/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-07-09
 * Description : a tool to blur an image
 *
 * Copyright (C) 2004-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>

// KDE includes.

#include <tdeaboutdata.h>
#include <knuminput.h>
#include <tdeconfig.h>
#include <kcursor.h>
#include <tdelocale.h>
#include <tdeapplication.h>

// Digikam includes.

#include "ddebug.h"
#include "imageiface.h"
#include "dimggaussianblur.h"

// Local includes.

#include "imageeffect_blur.h"
#include "imageeffect_blur.moc"

namespace DigikamImagesPluginCore
{

ImageEffect_Blur::ImageEffect_Blur(TQWidget* parent)
        : Digikam::CtrlPanelDlg(parent, i18n("Apply Gaussian Blur on Photograph"),
                                "gaussianblur", false, true, true)
{
    setHelp("blursharpentool.anchor", "digikam");

    TQWidget *gboxSettings = new TQWidget(m_imagePreviewWidget);
    TQGridLayout* gridSettings = new TQGridLayout( gboxSettings, 1, 1, 0, spacingHint());
    TQLabel *label = new TQLabel(i18n("Smoothness:"), gboxSettings);

    m_radiusInput = new KIntNumInput(gboxSettings);
    m_radiusInput->setRange(0, 100, 1, true);
    m_radiusInput->setValue(0);
    TQWhatsThis::add( m_radiusInput, i18n("<p>A smoothness of 0 has no effect, "
                                         "1 and above determine the Gaussian blur matrix radius "
                                         "that determines how much to blur the image."));

    gridSettings->addMultiCellWidget(label, 0, 0, 0, 1);
    gridSettings->addMultiCellWidget(m_radiusInput, 1, 1, 0, 1);

    m_imagePreviewWidget->setUserAreaWidget(gboxSettings);
}

ImageEffect_Blur::~ImageEffect_Blur()
{
}

void ImageEffect_Blur::readUserSettings()
{
    TDEConfig* config = kapp->config();
    config->setGroup("gaussianblur Tool Dialog");
    m_radiusInput->setValue(config->readNumEntry("RadiusAjustment", 0));
}

void ImageEffect_Blur::writeUserSettings()
{
    TDEConfig* config = kapp->config();
    config->setGroup("gaussianblur Tool Dialog");
    config->writeEntry("RadiusAjustment", m_radiusInput->value());
    config->sync();
}

void ImageEffect_Blur::resetValues(void)
{
    m_radiusInput->blockSignals(true);
    m_radiusInput->setValue(0);
    m_radiusInput->blockSignals(false);
}

void ImageEffect_Blur::prepareEffect()
{
    m_radiusInput->setEnabled(false);

    Digikam::DImg img = m_imagePreviewWidget->getOriginalRegionImage();

    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>
                       (new Digikam::DImgGaussianBlur(&img, this, m_radiusInput->value()));
}

void ImageEffect_Blur::prepareFinal()
{
    m_radiusInput->setEnabled(false);

    Digikam::ImageIface iface(0, 0);
    uchar *data            = iface.getOriginalImage();
    int w                  = iface.originalWidth();
    int h                  = iface.originalHeight();
    bool sixteenBit        = iface.originalSixteenBit();
    bool hasAlpha          = iface.originalHasAlpha();
    Digikam::DImg orgImage = Digikam::DImg(w, h, sixteenBit, hasAlpha ,data);
    delete [] data;
    m_threadedFilter = dynamic_cast<Digikam::DImgThreadedFilter *>
                       (new Digikam::DImgGaussianBlur(&orgImage, this, m_radiusInput->value()));
}

void ImageEffect_Blur::putPreviewData(void)
{
    Digikam::DImg imDest = m_threadedFilter->getTargetImage();
    m_imagePreviewWidget->setPreviewImage(imDest);
}

void ImageEffect_Blur::putFinalData(void)
{
    Digikam::ImageIface iface(0, 0);
    Digikam::DImg imDest = m_threadedFilter->getTargetImage();
    iface.putOriginalImage(i18n("Gaussian Blur"), imDest.bits());
}

void ImageEffect_Blur::renderingFinished(void)
{
    m_radiusInput->setEnabled(true);
}

}  // NameSpace DigikamImagesPluginCore

