/*
 * Decompiled with CFR 0.152.
 */
package net.ripe.rpki.validator3.rrdp;

import com.google.common.hash.Hashing;
import fj.data.Either;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import net.ripe.rpki.commons.util.RepositoryObjectType;
import net.ripe.rpki.commons.validation.ValidationResult;
import net.ripe.rpki.validator3.domain.RpkiObjectUtils;
import net.ripe.rpki.validator3.domain.metrics.RrdpMetricsService;
import net.ripe.rpki.validator3.rrdp.DeltaElement;
import net.ripe.rpki.validator3.rrdp.DeltaInfo;
import net.ripe.rpki.validator3.rrdp.DeltaPublish;
import net.ripe.rpki.validator3.rrdp.DeltaWithdraw;
import net.ripe.rpki.validator3.rrdp.Notification;
import net.ripe.rpki.validator3.rrdp.RrdpClient;
import net.ripe.rpki.validator3.rrdp.RrdpException;
import net.ripe.rpki.validator3.rrdp.RrdpParser;
import net.ripe.rpki.validator3.rrdp.RrdpService;
import net.ripe.rpki.validator3.rrdp.SnapshotObject;
import net.ripe.rpki.validator3.storage.Storage;
import net.ripe.rpki.validator3.storage.Tx;
import net.ripe.rpki.validator3.storage.data.RpkiObject;
import net.ripe.rpki.validator3.storage.data.RpkiRepository;
import net.ripe.rpki.validator3.storage.data.validation.RpkiRepositoryValidationRun;
import net.ripe.rpki.validator3.storage.data.validation.ValidationCheck;
import net.ripe.rpki.validator3.storage.stores.RpkiObjects;
import net.ripe.rpki.validator3.storage.stores.RpkiRepositories;
import net.ripe.rpki.validator3.util.Hex;
import net.ripe.rpki.validator3.util.Sha256;
import net.ripe.rpki.validator3.util.Time;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Service;

@Service
@Profile(value={"!test"})
public class RrdpServiceImpl
implements RrdpService {
    private static final Logger log = LoggerFactory.getLogger(RrdpServiceImpl.class);
    private static final int PENDING_OBJECT_COMMIT_BATCH_SIZE_BYTES = 1000000;
    private static final Predicate<RepositoryObjectType> NO_MANIFESTS_PREDICATE = type -> type != RepositoryObjectType.Manifest;
    private static final Predicate<RepositoryObjectType> ONLY_MANIFESTS_PREDICATE = type -> type == RepositoryObjectType.Manifest;
    private final RrdpParser rrdpParser = new RrdpParser();
    private final RrdpClient rrdpClient;
    private final RpkiObjects rpkiObjects;
    private final RpkiRepositories rpkiRepositories;
    private final Storage storage;
    private final RrdpMetricsService rrdpMetrics;

    @Autowired
    public RrdpServiceImpl(RrdpClient rrdpClient, RpkiObjects rpkiObjects, RpkiRepositories rpkiRepositories, Storage storage, RrdpMetricsService rrdpMetrics) {
        this.rrdpClient = rrdpClient;
        this.rpkiObjects = rpkiObjects;
        this.rpkiRepositories = rpkiRepositories;
        this.storage = storage;
        this.rrdpMetrics = rrdpMetrics;
    }

    public boolean storeRepository(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun) {
        Pair timed = Time.timed(() -> {
            try {
                return this.doStoreRepository(rpkiRepository, validationRun);
            }
            catch (RrdpException e) {
                log.warn("Error retrieving RRDP repository at {}: " + e.getMessage(), (Object)rpkiRepository.getRrdpNotifyUri());
                ValidationCheck validationCheck = new ValidationCheck(rpkiRepository.getRrdpNotifyUri(), ValidationCheck.Status.ERROR, "rrdp.fetch", new String[]{e.getMessage()});
                validationRun.addCheck(validationCheck);
                validationRun.setFailed();
                return false;
            }
        });
        log.info("RRDP repository job for {} took {}ms", (Object)rpkiRepository.getRrdpNotifyUri(), timed.getRight());
        return (Boolean)timed.getLeft();
    }

    private boolean doStoreRepository(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun) {
        Notification notification = (Notification)this.rrdpClient.readStream(rpkiRepository.getRrdpNotifyUri(), arg_0 -> ((RrdpParser)this.rrdpParser).notification(arg_0));
        log.info("Repository {}: local serial is '{}', latest serial is {}", new Object[]{rpkiRepository.getRrdpNotifyUri(), rpkiRepository.getRrdpSerial(), notification.serial});
        AtomicBoolean changedObjects = new AtomicBoolean(false);
        if (notification.sessionId.equals(rpkiRepository.getRrdpSessionId())) {
            if (rpkiRepository.getRrdpSerial().compareTo(notification.serial) <= 0) {
                try {
                    List orderedDeltas = this.verifyAndOrderDeltaSerials(notification, rpkiRepository);
                    for (DeltaInfo deltaInfo : orderedDeltas) {
                        this.processDelta(rpkiRepository, validationRun, notification, deltaInfo, changedObjects);
                    }
                }
                catch (RrdpException e) {
                    log.info("Processing deltas failed {}, falling back to snapshot processing.", (Object)e.getMessage());
                    this.rrdpMetrics.update(rpkiRepository.getRrdpNotifyUri(), "rrdp.fetch.deltas");
                    String errorCode = e.getErrorCode() != null ? e.getErrorCode() : "rrdp.fetch.deltas";
                    ValidationCheck validationCheck = new ValidationCheck(rpkiRepository.getRrdpNotifyUri(), ValidationCheck.Status.WARNING, errorCode, new String[]{e.getMessage()});
                    validationRun.addCheck(validationCheck);
                    this.processSnapshot(rpkiRepository, validationRun, notification, changedObjects);
                }
            } else {
                log.info("Repository serial {} is ahead of serial in notification file {}, fetching the snapshot", (Object)rpkiRepository.getRrdpSessionId(), (Object)notification.sessionId);
                this.rrdpMetrics.update(rpkiRepository.getRrdpNotifyUri(), "rrdp.fetch.snapshot.local.ahead");
                this.processSnapshot(rpkiRepository, validationRun, notification, changedObjects);
            }
        } else {
            log.info("Repository has session id '{}' but the downloaded version has session id '{}', fetching the snapshot", (Object)rpkiRepository.getRrdpSessionId(), (Object)notification.sessionId);
            this.rrdpMetrics.update(rpkiRepository.getRrdpNotifyUri(), "rrdp.fetch.snapshot.new.session");
            this.processSnapshot(rpkiRepository, validationRun, notification, changedObjects);
        }
        return changedObjects.get();
    }

    private void processSnapshot(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun, Notification notification, AtomicBoolean changedObjects) {
        this.rrdpClient.processUsingTemporaryFile(notification.snapshotUri, Hashing.sha256(), (snapshotPath, snapshotHash) -> {
            if (!Arrays.equals(Hex.parse((String)notification.snapshotHash), snapshotHash.asBytes())) {
                throw new RrdpException("rrdp.wrong.snapshot.hash", "Hash of the snapshot file " + notification.snapshotUri + " is " + Hex.format((byte[])snapshotHash.asBytes()) + ", but notification file says " + notification.snapshotHash);
            }
            Long timedStoreSnapshot = Time.timed(() -> {
                int counter = 0;
                log.debug("Processing RRDP repository {} snapshot, except for manifests", (Object)rpkiRepository.getRrdpNotifyUri());
                counter += this.processDownloadedSnapshot(rpkiRepository, validationRun, notification, snapshotPath, NO_MANIFESTS_PREDICATE);
                log.debug("Processing RRDP repository {} snapshot, manifests only", (Object)rpkiRepository.getRrdpNotifyUri());
                this.storage.writeTx0(tx -> this.rpkiRepositories.update(tx, rpkiRepository));
                changedObjects.set((counter += this.processDownloadedSnapshot(rpkiRepository, validationRun, notification, snapshotPath, ONLY_MANIFESTS_PREDICATE)) > 0);
                log.info("Added (or updated locations for) {} new objects", (Object)counter);
            });
            log.info("Storing snapshot {} time {}ms", (Object)rpkiRepository.getRrdpNotifyUri(), (Object)timedStoreSnapshot);
            return null;
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private int processDownloadedSnapshot(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun, Notification notification, Path snapshotPath, Predicate<RepositoryObjectType> typePredicate) {
        try (FileInputStream in = new FileInputStream(snapshotPath.toFile());){
            AtomicInteger counter = new AtomicInteger(0);
            AtomicInteger pendingObjectsBytes = new AtomicInteger(0);
            ArrayList pendingObjects = new ArrayList(1000);
            Runnable commitPendingObjects = () -> {
                int count = this.storeSnapshotObjects(pendingObjects, validationRun);
                counter.addAndGet(count);
                pendingObjects.clear();
                pendingObjectsBytes.set(0);
            };
            this.rrdpParser.parseSnapshot((InputStream)in, snapshotInfo -> {
                if (!notification.sessionId.equals(snapshotInfo.getSessionId())) {
                    this.rrdpMetrics.update(notification.snapshotUri, "rrdp.wrong.snapshot.session");
                    throw new RrdpException("rrdp.wrong.snapshot.session", "Session id of the snapshot (" + snapshotInfo.getSessionId() + ") is not the same as in the notification file: " + notification.sessionId);
                }
                if (!notification.getSerial().equals(snapshotInfo.getSerial())) {
                    throw new RrdpException("rrdp.serial.mismatch", "Serial of the snapshot (" + snapshotInfo.getSerial() + ") is not the same as in the notification file: " + notification.serial);
                }
                rpkiRepository.setRrdpSessionId(snapshotInfo.getSessionId());
                rpkiRepository.setRrdpSerial(snapshotInfo.getSerial());
            }, snapshotObject -> {
                if (!typePredicate.test(RepositoryObjectType.parse((String)snapshotObject.getUri()))) {
                    return;
                }
                pendingObjects.add(snapshotObject);
                int bytes = pendingObjectsBytes.addAndGet(snapshotObject.estimatedSize());
                if (bytes > 1000000) {
                    commitPendingObjects.run();
                }
            });
            commitPendingObjects.run();
            int n = counter.get();
            return n;
        }
        catch (IOException e) {
            this.rrdpMetrics.update(notification.snapshotUri, "rrdp.corrupted.snapshot");
            throw new RrdpException("Couldn't read snapshot: ", (Throwable)e);
        }
    }

    private void processDelta(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun, Notification notification, DeltaInfo di, AtomicBoolean changedObjects) {
        this.rrdpClient.processUsingTemporaryFile(di.getUri(), Hashing.sha256(), (deltaPath, deltaHash) -> {
            if (!Arrays.equals(Hex.parse((String)di.getHash()), deltaHash.asBytes())) {
                this.rrdpMetrics.update(notification.snapshotUri, "rrdp.wrong.delta.hash");
                throw new RrdpException("rrdp.wrong.delta.hash", "Hash of the delta file " + di + " is " + Hex.format((byte[])deltaHash.asBytes()) + ", but notification file says " + di.getHash());
            }
            Long timedStoreDelta = Time.timed(() -> {
                int counter = 0;
                log.debug("Processing RRDP repository {} delta {}, except for manifests", (Object)rpkiRepository.getRrdpNotifyUri(), (Object)di.getSerial());
                counter += this.processDownloadedDelta(rpkiRepository, validationRun, notification, di, deltaPath, NO_MANIFESTS_PREDICATE);
                log.debug("Processing RRDP repository {} delta {}, manifests only", (Object)rpkiRepository.getRrdpNotifyUri(), (Object)di.getSerial());
                this.storage.writeTx0(tx -> this.rpkiRepositories.update(tx, rpkiRepository));
                changedObjects.set((counter += this.processDownloadedDelta(rpkiRepository, validationRun, notification, di, deltaPath, ONLY_MANIFESTS_PREDICATE)) > 0);
                log.info("Added, withdrew, or updated locations for {} new objects", (Object)counter);
            });
            log.info("Storing delta {} time {}ms", (Object)rpkiRepository.getRrdpNotifyUri(), (Object)timedStoreDelta);
            return null;
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private int processDownloadedDelta(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun, Notification notification, DeltaInfo di, Path deltaPath, Predicate<RepositoryObjectType> typePredicate) {
        try (FileInputStream in = new FileInputStream(deltaPath.toFile());){
            AtomicInteger counter = new AtomicInteger(0);
            AtomicInteger pendingObjectsBytes = new AtomicInteger(0);
            ArrayList pendingObjects = new ArrayList(1000);
            Runnable commitPendingObjects = () -> {
                this.storage.readTx0(tx -> this.verifyDeltaIsApplicable(tx, pendingObjects));
                this.storage.writeTx0(tx -> counter.addAndGet(this.storeDeltaObjects(tx, pendingObjects, validationRun)));
                pendingObjects.clear();
                pendingObjectsBytes.set(0);
            };
            this.rrdpParser.parseDelta((InputStream)in, deltaHeader -> {
                if (!notification.sessionId.equals(deltaHeader.getSessionId())) {
                    throw new RrdpException("rrdp.wrong.delta.session", "Session id of the delta (" + deltaHeader + ") is not the same as in the notification file: " + notification.sessionId);
                }
                if (!di.getSerial().equals(deltaHeader.getSerial())) {
                    throw new RrdpException("rrdp.serial.mismatch", "Serial of the delta (" + deltaHeader.getSerial() + ") is not the same as in the notification file: " + di);
                }
                rpkiRepository.setRrdpSerial(deltaHeader.getSerial());
            }, deltaElement -> {
                if (!typePredicate.test(RepositoryObjectType.parse((String)deltaElement.getUri()))) {
                    return;
                }
                pendingObjects.add(deltaElement);
                int bytes = pendingObjectsBytes.addAndGet(deltaElement.estimatedSize());
                if (bytes > 1000000) {
                    commitPendingObjects.run();
                }
            });
            commitPendingObjects.run();
            int n = counter.get();
            return n;
        }
        catch (IOException e) {
            throw new RrdpException("Error parsing delta (" + di + "): " + notification.sessionId, (Throwable)e);
        }
    }

    private List<DeltaInfo> verifyAndOrderDeltaSerials(Notification notification, RpkiRepository rpkiRepository) {
        List<DeltaInfo> orderedDeltas = notification.getDeltas().stream().filter(d -> d.getSerial().compareTo(rpkiRepository.getRrdpSerial()) > 0).sorted(Comparator.comparing(DeltaInfo::getSerial)).collect(Collectors.toList());
        if (orderedDeltas.isEmpty()) {
            if (!rpkiRepository.getRrdpSerial().equals(notification.serial)) {
                throw new RrdpException("rrdp.serial.mismatch", "The current serial is " + rpkiRepository.getRrdpSerial() + ", notification file serial is " + notification.serial + ", but the list of deltas is empty.");
            }
        } else {
            BigInteger earliestDeltaSerial = ((DeltaInfo)orderedDeltas.get(0)).getSerial();
            BigInteger latestDeltaSerial = orderedDeltas.get(orderedDeltas.size() - 1).getSerial();
            if (!notification.serial.equals(latestDeltaSerial)) {
                throw new RrdpException("rrdp.serial.mismatch", "The last delta serial is " + latestDeltaSerial + ", notification file serial is " + notification.serial);
            }
            if (earliestDeltaSerial.compareTo(notification.serial) > 0) {
                throw new RrdpException("rrdp.serial.mismatch", "The earliest available delta serial " + earliestDeltaSerial + " is later than notification file serial is " + notification.serial);
            }
            if (orderedDeltas.size() > 1) {
                for (int i = 0; i < orderedDeltas.size() - 1; ++i) {
                    BigInteger currentSerial = orderedDeltas.get(i).getSerial();
                    BigInteger nextSerial = orderedDeltas.get(i + 1).getSerial();
                    if (currentSerial.add(BigInteger.ONE).equals(nextSerial)) continue;
                    throw new RrdpException("rrdp.serial.mismatch", String.format("Serials of the deltas are not contiguous: found %d and %d after it", currentSerial, nextSerial));
                }
            }
        }
        return orderedDeltas;
    }

    int storeSnapshotObjects(List<SnapshotObject> snapshotObjects, RpkiRepositoryValidationRun validationRun) {
        AtomicInteger counter = new AtomicInteger();
        List converted = snapshotObjects.parallelStream().map(value -> RpkiObjectUtils.createRpkiObject((String)value.getUri(), (byte[])value.getContent())).collect(Collectors.toList());
        this.storage.writeTx0(tx -> converted.forEach(maybeRpkiObject -> this.storeSnapshotObject(tx, validationRun, maybeRpkiObject, counter)));
        return counter.get();
    }

    private void storeSnapshotObject(Tx.Write tx, RpkiRepositoryValidationRun validationRun, Either<ValidationResult, Pair<String, RpkiObject>> maybeRpkiObject, AtomicInteger counter) {
        if (maybeRpkiObject.isLeft()) {
            validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
        } else {
            Pair p = (Pair)maybeRpkiObject.right().value();
            RpkiObject object = (RpkiObject)p.getRight();
            String location = (String)p.getLeft();
            this.rpkiObjects.put(tx, object, location);
            counter.incrementAndGet();
        }
    }

    private int storeDeltaObjects(Tx.Write wtx, List<DeltaElement> deltaElements, RpkiRepositoryValidationRun validationRun) {
        AtomicInteger added = new AtomicInteger();
        AtomicInteger deleted = new AtomicInteger();
        deltaElements.forEach(deltaElement -> {
            if (deltaElement instanceof DeltaPublish) {
                if (this.applyDeltaPublish(validationRun, deltaElement.getUri(), (DeltaPublish)deltaElement, wtx)) {
                    added.incrementAndGet();
                }
            } else if (deltaElement instanceof DeltaWithdraw && this.applyDeltaWithdraw(validationRun, deltaElement.getUri(), (DeltaWithdraw)deltaElement, wtx)) {
                deleted.incrementAndGet();
            }
        });
        return added.get() + deleted.get();
    }

    private boolean applyDeltaWithdraw(RpkiRepositoryValidationRun validationRun, String uri, DeltaWithdraw deltaWithdraw, Tx.Write tx) {
        byte[] sha256 = deltaWithdraw.getHash();
        Optional maybeObject = this.rpkiObjects.findBySha256((Tx.Read)tx, sha256);
        if (maybeObject.isPresent()) {
            return true;
        }
        ValidationCheck validationCheck = new ValidationCheck(uri, ValidationCheck.Status.ERROR, "rrdp.withdraw.nonexistent.object", new String[]{Hex.format((byte[])sha256)});
        validationRun.addCheck(validationCheck);
        return false;
    }

    private void verifyDeltaIsApplicable(Tx.Read tx, List<DeltaElement> pendingObjects) {
        pendingObjects.forEach(deltaElement -> {
            if (deltaElement instanceof DeltaPublish) {
                ((DeltaPublish)deltaElement).getHash().ifPresent(sha -> this.checkObjectExists(deltaElement, "rrdp.replace.nonexistent.object", sha, tx));
            } else if (deltaElement instanceof DeltaWithdraw) {
                this.checkObjectExists(deltaElement, "rrdp.withdraw.nonexistent.object", ((DeltaWithdraw)deltaElement).getHash(), tx);
            }
        });
    }

    private void checkObjectExists(DeltaElement deltaElement, String errorCode, byte[] sha256, Tx.Read tx) {
        Optional objectByHash = this.rpkiObjects.findBySha256(tx, sha256);
        if (!objectByHash.isPresent()) {
            throw new RrdpException(errorCode, "Couldn't find an object with location '" + deltaElement.uri + "' with hash " + Hex.format((byte[])sha256));
        }
    }

    private boolean applyDeltaPublish(RpkiRepositoryValidationRun validationRun, String uri, DeltaPublish deltaPublish, Tx.Write tx) {
        if (deltaPublish.getHash().isPresent()) {
            byte[] sha256 = (byte[])deltaPublish.getHash().get();
            Optional existing = this.rpkiObjects.findBySha256((Tx.Read)tx, sha256);
            if (existing.isPresent()) {
                byte[] content = deltaPublish.getContent();
                Either maybeRpkiObject = RpkiObjectUtils.createRpkiObject((String)uri, (byte[])content);
                if (maybeRpkiObject.isLeft()) {
                    validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
                } else {
                    Pair p = (Pair)maybeRpkiObject.right().value();
                    RpkiObject object = (RpkiObject)p.getRight();
                    if (!Arrays.equals(object.getSha256(), sha256)) {
                        String location = (String)p.getLeft();
                        this.rpkiObjects.put(tx, object, location);
                        return true;
                    }
                }
            } else {
                ValidationCheck validationCheck = new ValidationCheck(uri, ValidationCheck.Status.ERROR, "rrdp.replace.nonexistent.object", new String[]{Hex.format((byte[])sha256)});
                validationRun.addCheck(validationCheck);
            }
        } else {
            byte[] content = deltaPublish.getContent();
            Either maybeRpkiObject = RpkiObjectUtils.createRpkiObject((String)uri, (byte[])content);
            if (maybeRpkiObject.isLeft()) {
                validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
            } else {
                Pair p = (Pair)maybeRpkiObject.right().value();
                RpkiObject object = (RpkiObject)p.getRight();
                Optional bySha256 = this.rpkiObjects.findBySha256((Tx.Read)tx, Sha256.hash((byte[])content));
                if (!bySha256.isPresent()) {
                    this.rpkiObjects.put(tx, object, (String)p.getLeft());
                    return true;
                }
            }
        }
        return false;
    }
}

