/*
 * Decompiled with CFR 0.152.
 */
package net.ripe.rpki.validator3.rrdp;

import com.google.common.hash.Hashing;
import com.google.common.hash.HashingInputStream;
import fj.data.Either;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.ripe.rpki.commons.validation.ValidationResult;
import net.ripe.rpki.validator3.domain.RpkiObjectUtils;
import net.ripe.rpki.validator3.rrdp.Delta;
import net.ripe.rpki.validator3.rrdp.DeltaElement;
import net.ripe.rpki.validator3.rrdp.DeltaInfo;
import net.ripe.rpki.validator3.rrdp.DeltaPublish;
import net.ripe.rpki.validator3.rrdp.DeltaWithdraw;
import net.ripe.rpki.validator3.rrdp.Notification;
import net.ripe.rpki.validator3.rrdp.RrdpClient;
import net.ripe.rpki.validator3.rrdp.RrdpException;
import net.ripe.rpki.validator3.rrdp.RrdpParser;
import net.ripe.rpki.validator3.rrdp.RrdpService;
import net.ripe.rpki.validator3.rrdp.Snapshot;
import net.ripe.rpki.validator3.storage.Storage;
import net.ripe.rpki.validator3.storage.Tx;
import net.ripe.rpki.validator3.storage.data.RpkiObject;
import net.ripe.rpki.validator3.storage.data.RpkiRepository;
import net.ripe.rpki.validator3.storage.data.validation.RpkiRepositoryValidationRun;
import net.ripe.rpki.validator3.storage.data.validation.ValidationCheck;
import net.ripe.rpki.validator3.storage.stores.RpkiObjects;
import net.ripe.rpki.validator3.storage.stores.RpkiRepositories;
import net.ripe.rpki.validator3.util.Hex;
import net.ripe.rpki.validator3.util.Sha256;
import net.ripe.rpki.validator3.util.Time;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Service;

@Service
@Profile(value={"!test"})
public class RrdpServiceImpl
implements RrdpService {
    private static final Logger log = LoggerFactory.getLogger(RrdpServiceImpl.class);
    private final RrdpParser rrdpParser = new RrdpParser();
    private final RrdpClient rrdpClient;
    private final RpkiObjects rpkiObjects;
    private final RpkiRepositories rpkiRepositories;
    private final Storage storage;
    private final ExecutorCompletionService<Either<ValidationResult, Pair<String, RpkiObject>>> asyncCreateObjects = new ExecutorCompletionService(Executors.newFixedThreadPool(Math.max(1, Runtime.getRuntime().availableProcessors() - 1)));

    @Autowired
    public RrdpServiceImpl(RrdpClient rrdpClient, RpkiObjects rpkiObjects, RpkiRepositories rpkiRepositories, Storage storage) {
        this.rrdpClient = rrdpClient;
        this.rpkiObjects = rpkiObjects;
        this.rpkiRepositories = rpkiRepositories;
        this.storage = storage;
    }

    public boolean storeRepository(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun) {
        try {
            return this.doStoreRepository(rpkiRepository, validationRun);
        }
        catch (RrdpException e) {
            log.warn("Error retrieving RRDP repository at {}: " + e.getMessage(), (Object)rpkiRepository.getRrdpNotifyUri());
            ValidationCheck validationCheck = new ValidationCheck(rpkiRepository.getRrdpNotifyUri(), ValidationCheck.Status.ERROR, "rrdp.fetch", new String[]{e.getMessage()});
            validationRun.addCheck(validationCheck);
            validationRun.setFailed();
            return false;
        }
    }

    private boolean doStoreRepository(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun) {
        Notification notification = (Notification)this.rrdpClient.readStream(rpkiRepository.getRrdpNotifyUri(), arg_0 -> ((RrdpParser)this.rrdpParser).notification(arg_0));
        log.info("Repository {}: local serial is '{}', latest serial is {}", new Object[]{rpkiRepository.getRrdpNotifyUri(), rpkiRepository.getRrdpSerial(), notification.serial});
        AtomicBoolean changedObjects = new AtomicBoolean(false);
        if (notification.sessionId.equals(rpkiRepository.getRrdpSessionId())) {
            if (rpkiRepository.getRrdpSerial().compareTo(notification.serial) <= 0) {
                try {
                    List<Delta> deltas = ((Stream)notification.deltas.stream().filter(d -> d.getSerial().compareTo(rpkiRepository.getRrdpSerial()) > 0).sorted(Comparator.comparing(DeltaInfo::getSerial)).parallel()).map(di -> this.readDelta(notification, di)).collect(Collectors.toList());
                    this.verifyDeltaSerials(deltas, notification, rpkiRepository);
                    deltas.forEach(d -> {
                        this.storage.readTx0(tx -> this.verifyDeltaIsApplicable(tx, d));
                        this.storage.writeTx0(tx -> {
                            this.storeDelta(tx, d, validationRun, rpkiRepository, changedObjects);
                            tx.afterCommit(() -> rpkiRepository.setRrdpSerial(rpkiRepository.getRrdpSerial().add(BigInteger.ONE)));
                        });
                    });
                }
                catch (RrdpException e) {
                    log.info("Processing deltas failed {}, falling back to snapshot processing.", (Object)e.getMessage());
                    String errorCode = e.getErrorCode() != null ? e.getErrorCode() : "rrdp.fetch.deltas";
                    ValidationCheck validationCheck = new ValidationCheck(rpkiRepository.getRrdpNotifyUri(), ValidationCheck.Status.WARNING, errorCode, new String[]{e.getMessage()});
                    validationRun.addCheck(validationCheck);
                    this.readSnapshot(rpkiRepository, validationRun, notification, changedObjects);
                }
            } else {
                log.info("Repository serial {} is ahead of serial in notification file {}, fetching the snapshot", (Object)rpkiRepository.getRrdpSessionId(), (Object)notification.sessionId);
                this.readSnapshot(rpkiRepository, validationRun, notification, changedObjects);
            }
        } else {
            log.info("Repository has session id '{}' but the downloaded version has session id '{}', fetching the snapshot", (Object)rpkiRepository.getRrdpSessionId(), (Object)notification.sessionId);
            this.readSnapshot(rpkiRepository, validationRun, notification, changedObjects);
        }
        return changedObjects.get();
    }

    private void readSnapshot(RpkiRepository rpkiRepository, RpkiRepositoryValidationRun validationRun, Notification notification, AtomicBoolean changedObjects) {
        AtomicReference hashingStream = new AtomicReference();
        Pair timedSnapshot = Time.timed(() -> (Snapshot)this.rrdpClient.readStream(notification.snapshotUri, is -> {
            hashingStream.set(new HashingInputStream(Hashing.sha256(), is));
            return this.rrdpParser.snapshot((InputStream)hashingStream.get());
        }));
        byte[] snapshotHash = ((HashingInputStream)hashingStream.get()).hash().asBytes();
        if (!Arrays.equals(Hex.parse((String)notification.snapshotHash), snapshotHash)) {
            throw new RrdpException("rrdp.wrong.snapshot.hash", "Hash of the snapshot file " + notification.snapshotUri + " is " + Hex.format((byte[])snapshotHash) + ", but notification file says " + notification.snapshotHash);
        }
        log.info("Downloading/hashing/parsing snapshot time {}ms", timedSnapshot.getRight());
        Long timedStoreSnapshot = Time.timed(() -> this.storage.writeTx0(tx -> {
            this.storeSnapshot(tx, (Snapshot)timedSnapshot.getLeft(), validationRun, changedObjects);
            rpkiRepository.setRrdpSessionId(notification.sessionId);
            rpkiRepository.setRrdpSerial(notification.serial);
            this.rpkiRepositories.update(tx, rpkiRepository);
        }));
        log.info("Storing snapshot time {}ms", (Object)timedStoreSnapshot);
    }

    private Delta readDelta(Notification notification, DeltaInfo di) {
        Delta d;
        byte[] deltaBody = this.rrdpClient.getBody(di.getUri());
        byte[] deltaHash = Sha256.hash((byte[])deltaBody);
        if (!Arrays.equals(Hex.parse((String)di.getHash()), deltaHash)) {
            throw new RrdpException("rrdp.wrong.delta.hash", "Hash of the delta file " + di + " is " + Hex.format((byte[])deltaHash) + ", but notification file says " + di.getHash());
        }
        try {
            d = this.rrdpParser.delta((InputStream)new ByteArrayInputStream(deltaBody));
        }
        catch (Exception e) {
            throw new RrdpException("Error parsing delta (" + di + "): " + notification.sessionId, (Throwable)e);
        }
        if (!d.getSessionId().equals(notification.sessionId)) {
            throw new RrdpException("rrdp.wrong.delta.session", "Session id of the delta (" + di + ") is not the same as in the notification file: " + notification.sessionId);
        }
        return d;
    }

    private void verifyDeltaSerials(List<Delta> orderedDeltas, Notification notification, RpkiRepository rpkiRepository) {
        if (orderedDeltas.isEmpty()) {
            if (!rpkiRepository.getRrdpSerial().equals(notification.serial)) {
                throw new RrdpException("rrdp.serial.mismatch", "The current serial is " + rpkiRepository.getRrdpSerial() + ", notification file serial is " + notification.serial + ", but the list of deltas is empty.");
            }
        } else {
            BigInteger earliestDeltaSerial = orderedDeltas.get(0).getSerial();
            BigInteger latestDeltaSerial = orderedDeltas.get(orderedDeltas.size() - 1).getSerial();
            if (!notification.serial.equals(latestDeltaSerial)) {
                throw new RrdpException("rrdp.serial.mismatch", "The last delta serial is " + latestDeltaSerial + ", notification file serial is " + notification.serial);
            }
            if (earliestDeltaSerial.compareTo(notification.serial) > 0) {
                throw new RrdpException("rrdp.serial.mismatch", "The earliest available delta serial " + earliestDeltaSerial + " is later than notification file serial is " + notification.serial);
            }
            if (orderedDeltas.size() > 1) {
                for (int i = 0; i < orderedDeltas.size() - 1; ++i) {
                    BigInteger currentSerial = orderedDeltas.get(i).getSerial();
                    BigInteger nextSerial = orderedDeltas.get(i + 1).getSerial();
                    if (currentSerial.add(BigInteger.ONE).equals(nextSerial)) continue;
                    throw new RrdpException("rrdp.serial.mismatch", String.format("Serials of the deltas are not contiguous: found %d and %d after it", currentSerial, nextSerial));
                }
            }
        }
    }

    void storeSnapshot(Tx.Write tx, Snapshot snapshot, RpkiRepositoryValidationRun validationRun, AtomicBoolean changedObjects) {
        AtomicInteger counter = new AtomicInteger();
        AtomicInteger workCounter = new AtomicInteger(0);
        int threshold = 10;
        snapshot.asMap().forEach((uri, value) -> {
            byte[] content = value.content;
            byte[] sha256 = Sha256.hash((byte[])content);
            Optional existing = this.rpkiObjects.findBySha256((Tx.Read)tx, sha256);
            if (existing.isPresent()) {
                this.rpkiObjects.addLocation(tx, ((RpkiObject)existing.get()).key(), uri);
            } else {
                if (workCounter.get() > 10) {
                    this.storeSnapshotObject(tx, validationRun, counter);
                    workCounter.decrementAndGet();
                }
                this.asyncCreateObjects.submit(() -> RpkiObjectUtils.createRpkiObject((String)uri, (byte[])content));
                workCounter.incrementAndGet();
            }
        });
        while (workCounter.getAndDecrement() > 0) {
            this.storeSnapshotObject(tx, validationRun, counter);
        }
        changedObjects.set(counter.get() > 0);
        log.info("Added (or updated locations for) {} new objects", (Object)counter.get());
    }

    private void storeSnapshotObject(Tx.Write tx, RpkiRepositoryValidationRun validationRun, AtomicInteger counter) {
        try {
            Either maybeRpkiObject = (Either)this.asyncCreateObjects.take().get();
            if (maybeRpkiObject.isLeft()) {
                validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
            } else {
                Pair p = (Pair)maybeRpkiObject.right().value();
                RpkiObject object = (RpkiObject)p.getRight();
                String location = (String)p.getLeft();
                this.rpkiObjects.put(tx, object, location);
                counter.incrementAndGet();
            }
        }
        catch (Exception e) {
            log.error("Something strange happened here", (Throwable)e);
        }
    }

    private void storeDelta(Tx.Write wtx, Delta delta, RpkiRepositoryValidationRun validationRun, RpkiRepository rpkiRepository, AtomicBoolean changedObjectsCounter) {
        AtomicInteger added = new AtomicInteger();
        AtomicInteger deleted = new AtomicInteger();
        delta.asMap().forEach((uri, deltaElement) -> {
            if (deltaElement instanceof DeltaPublish) {
                if (this.applyDeltaPublish(validationRun, uri, (DeltaPublish)deltaElement, wtx)) {
                    added.incrementAndGet();
                    changedObjectsCounter.set(true);
                }
            } else if (deltaElement instanceof DeltaWithdraw && this.applyDeltaWithdraw(validationRun, uri, (DeltaWithdraw)deltaElement, wtx)) {
                deleted.incrementAndGet();
                changedObjectsCounter.set(true);
            }
        });
        log.info("Repository {}: added (or updated locations for) {} new objects, delete (or removed locations) for {} objects", new Object[]{rpkiRepository.getRrdpNotifyUri(), added.get(), deleted.get()});
    }

    private boolean applyDeltaWithdraw(RpkiRepositoryValidationRun validationRun, String uri, DeltaWithdraw deltaWithdraw, Tx.Write tx) {
        byte[] sha256 = deltaWithdraw.getHash();
        Optional maybeObject = this.rpkiObjects.findBySha256((Tx.Read)tx, sha256);
        if (maybeObject.isPresent()) {
            return true;
        }
        ValidationCheck validationCheck = new ValidationCheck(uri, ValidationCheck.Status.ERROR, "rrdp.withdraw.nonexistent.object", new String[]{Hex.format((byte[])sha256)});
        validationRun.addCheck(validationCheck);
        return false;
    }

    private void verifyDeltaIsApplicable(Tx.Read tx, Delta d) {
        d.asMap().forEach((uri, deltaElement) -> {
            if (deltaElement instanceof DeltaPublish) {
                ((DeltaPublish)deltaElement).getHash().ifPresent(sha -> this.checkObjectExists(deltaElement, "rrdp.replace.nonexistent.object", sha, tx));
            } else if (deltaElement instanceof DeltaWithdraw) {
                this.checkObjectExists(deltaElement, "rrdp.withdraw.nonexistent.object", ((DeltaWithdraw)deltaElement).getHash(), tx);
            }
        });
    }

    private void checkObjectExists(DeltaElement deltaElement, String errorCode, byte[] sha256, Tx.Read tx) {
        Optional objectByHash = this.rpkiObjects.findBySha256(tx, sha256);
        if (!objectByHash.isPresent()) {
            throw new RrdpException(errorCode, "Couldn't find an object with location '" + deltaElement.uri + "' with hash " + Hex.format((byte[])sha256));
        }
    }

    private boolean applyDeltaPublish(RpkiRepositoryValidationRun validationRun, String uri, DeltaPublish deltaPublish, Tx.Write tx) {
        if (deltaPublish.getHash().isPresent()) {
            byte[] sha256 = (byte[])deltaPublish.getHash().get();
            Optional existing = this.rpkiObjects.findBySha256((Tx.Read)tx, sha256);
            if (existing.isPresent()) {
                byte[] content = deltaPublish.getContent();
                Either maybeRpkiObject = RpkiObjectUtils.createRpkiObject((String)uri, (byte[])content);
                if (maybeRpkiObject.isLeft()) {
                    validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
                } else {
                    Pair p = (Pair)maybeRpkiObject.right().value();
                    RpkiObject object = (RpkiObject)p.getRight();
                    if (!Arrays.equals(object.getSha256(), sha256)) {
                        String location = (String)p.getLeft();
                        this.rpkiObjects.put(tx, object, location);
                        return true;
                    }
                }
            } else {
                ValidationCheck validationCheck = new ValidationCheck(uri, ValidationCheck.Status.ERROR, "rrdp.replace.nonexistent.object", new String[]{Hex.format((byte[])sha256)});
                validationRun.addCheck(validationCheck);
            }
        } else {
            byte[] content = deltaPublish.getContent();
            Either maybeRpkiObject = RpkiObjectUtils.createRpkiObject((String)uri, (byte[])content);
            if (maybeRpkiObject.isLeft()) {
                validationRun.addChecks((ValidationResult)maybeRpkiObject.left().value());
            } else {
                Pair p = (Pair)maybeRpkiObject.right().value();
                RpkiObject object = (RpkiObject)p.getRight();
                Optional bySha256 = this.rpkiObjects.findBySha256((Tx.Read)tx, Sha256.hash((byte[])content));
                if (!bySha256.isPresent()) {
                    this.rpkiObjects.put(tx, object, (String)p.getLeft());
                    return true;
                }
            }
        }
        return false;
    }
}

