// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/views/examples/examples_main_proc.h"

#include <memory>
#include <string>

#include "base/base_switches.h"
#include "base/bind.h"
#include "base/command_line.h"
#include "base/feature_list.h"
#include "base/files/file_path.h"
#include "base/i18n/icu_util.h"
#include "base/lazy_instance.h"
#include "base/memory/ptr_util.h"
#include "base/path_service.h"
#include "base/power_monitor/power_monitor.h"
#include "base/power_monitor/power_monitor_device_source.h"
#include "base/run_loop.h"
#include "base/test/scoped_run_loop_timeout.h"
#include "base/test/task_environment.h"
#include "base/test/test_discardable_memory_allocator.h"
#include "base/test/test_timeouts.h"
#include "build/build_config.h"
#include "build/chromeos_buildflags.h"
#include "components/viz/common/features.h"
#include "components/viz/host/host_frame_sink_manager.h"
#include "components/viz/service/display_embedder/server_shared_bitmap_manager.h"
#include "components/viz/service/frame_sinks/frame_sink_manager_impl.h"
#include "mojo/core/embedder/embedder.h"
#include "ui/base/ime/init/input_method_initializer.h"
#include "ui/base/resource/resource_bundle.h"
#include "ui/base/ui_base_paths.h"
#include "ui/compositor/compositor_switches.h"
#include "ui/compositor/test/in_process_context_factory.h"
#include "ui/compositor/test/test_context_factories.h"
#include "ui/display/screen.h"
#include "ui/gfx/font_util.h"
#include "ui/gfx/image/image.h"
#include "ui/gl/gl_switches.h"
#include "ui/gl/init/gl_factory.h"
#include "ui/views/buildflags.h"
#include "ui/views/examples/example_base.h"
#include "ui/views/examples/examples_window.h"
#include "ui/views/test/desktop_test_views_delegate.h"
#include "ui/views/widget/any_widget_observer.h"
#include "ui/views/widget/widget.h"

#if defined(USE_AURA)
#include "ui/aura/env.h"
#include "ui/wm/core/wm_state.h"
#endif

#if BUILDFLAG(IS_CHROMEOS_ASH)
#include "ui/views/examples/examples_views_delegate_chromeos.h"
#endif

#if BUILDFLAG(ENABLE_DESKTOP_AURA)
#include "ui/views/widget/desktop_aura/desktop_screen.h"
#endif

#if defined(OS_WIN)
#include "ui/base/win/scoped_ole_initializer.h"
#include "ui/views/examples/examples_skia_gold_pixel_diff.h"
#endif

#if defined(USE_OZONE)
#include "ui/base/ui_base_features.h"
#include "ui/ozone/public/ozone_platform.h"
#endif

namespace views {
namespace examples {

base::LazyInstance<base::TestDiscardableMemoryAllocator>::DestructorAtExit
    g_discardable_memory_allocator = LAZY_INSTANCE_INITIALIZER;

ExamplesExitCode ExamplesMainProc(bool under_test) {
#if defined(OS_WIN)
  ui::ScopedOleInitializer ole_initializer;
#endif

  base::CommandLine* command_line = base::CommandLine::ForCurrentProcess();

  if (CheckCommandLineUsage())
    return ExamplesExitCode::kSucceeded;

  // Disabling Direct Composition works around the limitation that
  // InProcessContextFactory doesn't work with Direct Composition, causing the
  // window to not render. See http://crbug.com/936249.
  command_line->AppendSwitch(switches::kDisableDirectComposition);

  // Disable skia renderer to use GL instead.
  std::string disabled =
      command_line->GetSwitchValueASCII(switches::kDisableFeatures);
  if (!disabled.empty())
    disabled += ",";
  disabled += features::kUseSkiaRenderer.name;
  command_line->AppendSwitchASCII(switches::kDisableFeatures, disabled);

  base::FeatureList::InitializeInstance(
      command_line->GetSwitchValueASCII(switches::kEnableFeatures),
      command_line->GetSwitchValueASCII(switches::kDisableFeatures));

  if (under_test)
    command_line->AppendSwitch(switches::kEnablePixelOutputInTests);

  mojo::core::Init();

#if defined(USE_OZONE)
  if (features::IsUsingOzonePlatform()) {
    ui::OzonePlatform::InitParams params;
    params.single_process = true;
    ui::OzonePlatform::InitializeForGPU(params);
  }
#endif

  gl::init::InitializeGLOneOff();

  // Viz depends on the task environment to correctly tear down.
  base::test::TaskEnvironment task_environment(
      base::test::TaskEnvironment::MainThreadType::UI);

  // The ContextFactory must exist before any Compositors are created.
  auto context_factories =
      std::make_unique<ui::TestContextFactories>(under_test);
  context_factories->SetUseTestSurface(false);

  base::i18n::InitializeICU();

  ui::RegisterPathProvider();

  base::FilePath ui_test_pak_path;
  CHECK(base::PathService::Get(ui::UI_TEST_PAK, &ui_test_pak_path));
  ui::ResourceBundle::InitSharedInstanceWithPakPath(ui_test_pak_path);

  base::FilePath views_examples_resources_pak_path;
  CHECK(base::PathService::Get(base::DIR_MODULE,
                               &views_examples_resources_pak_path));
  ui::ResourceBundle::GetSharedInstance().AddDataPackFromPath(
      views_examples_resources_pak_path.AppendASCII(
          "views_examples_resources.pak"),
      ui::SCALE_FACTOR_100P);

  base::DiscardableMemoryAllocator::SetInstance(
      g_discardable_memory_allocator.Pointer());

  base::PowerMonitor::Initialize(
      std::make_unique<base::PowerMonitorDeviceSource>());

  gfx::InitializeFonts();

#if defined(USE_AURA)
  std::unique_ptr<aura::Env> env = aura::Env::CreateInstance();
  aura::Env::GetInstance()->set_context_factory(
      context_factories->GetContextFactory());
#endif
  ui::InitializeInputMethodForTesting();

  ExamplesExitCode compare_result = ExamplesExitCode::kSucceeded;

  {
#if BUILDFLAG(IS_CHROMEOS_ASH)
    ExamplesViewsDelegateChromeOS views_delegate;
#else
    views::DesktopTestViewsDelegate views_delegate;
#if defined(USE_AURA)
    wm::WMState wm_state;
#endif
#endif
#if BUILDFLAG(ENABLE_DESKTOP_AURA)
    std::unique_ptr<display::Screen> desktop_screen =
        views::CreateDesktopScreen();
#endif

    base::RunLoop run_loop(base::RunLoop::Type::kNestableTasksAllowed);

#if defined(OS_WIN)
    ExamplesSkiaGoldPixelDiff pixel_diff;
    views::AnyWidgetObserver widget_observer{
        views::test::AnyWidgetTestPasskey()};

    // If this app isn't a test, it shouldn't timeout.
    auto disable_timeout =
        std::make_unique<base::test::ScopedDisableRunLoopTimeout>();

    if (under_test) {
      pixel_diff.Init("ViewsExamples");
      widget_observer.set_shown_callback(
          base::BindRepeating(&ExamplesSkiaGoldPixelDiff::OnExamplesWindowShown,
                              base::Unretained(&pixel_diff)));
      // Enable the timeout since we're not running in a test.
      disable_timeout.reset();
    }
#else
    base::test::ScopedDisableRunLoopTimeout disable_timeout;
#endif

    views::examples::ShowExamplesWindow(run_loop.QuitClosure());

    run_loop.Run();

#if defined(OS_WIN)
    compare_result = pixel_diff.get_result();
#endif

    ui::ResourceBundle::CleanupSharedInstance();
  }

  ui::ShutdownInputMethod();

#if defined(USE_AURA)
  env.reset();
#endif

  return compare_result;
}

}  // namespace examples
}  // namespace views
