export const description = `
TODO:

- Start a pipeline statistics query in all possible encoders:
    - queryIndex {in, out of} range for GPUQuerySet
    - GPUQuerySet {valid, invalid, device mismatched}
    - x ={render pass, compute pass} encoder
`;

import { makeTestGroup } from '../../../../../common/framework/test_group.js';
import { kQueryTypes } from '../../../../capability_info.js';
import { ValidationTest } from '../../validation_test.js';

import { createQuerySetWithType } from './common.js';

export const g = makeTestGroup(ValidationTest);

g.test('occlusion_query,query_type')
  .desc(
    `
Tests that set occlusion query set with all types in render pass descriptor:
- type {occlusion (control case), pipeline statistics, timestamp}
- {undefined} for occlusion query set in render pass descriptor
  `
  )
  .paramsSubcasesOnly(u => u.combine('type', [undefined, ...kQueryTypes]))
  .fn(async t => {
    const type = t.params.type;

    if (type) {
      await t.selectDeviceForQueryTypeOrSkipTestCase(type);
    }

    const querySet = type === undefined ? undefined : createQuerySetWithType(t, type, 1);

    const encoder = t.createEncoder('render pass', { occlusionQuerySet: querySet });
    encoder.encoder.beginOcclusionQuery(0);
    encoder.encoder.endOcclusionQuery();
    encoder.validateFinish(type === 'occlusion');
  });

g.test('occlusion_query,invalid_query_set')
  .desc(
    `
Tests that begin occlusion query with a invalid query set that failed during creation.
  `
  )
  .paramsSubcasesOnly(u => u.combine('querySetState', ['valid', 'invalid'] as const))
  .fn(t => {
    const occlusionQuerySet = t.createQuerySetWithState(t.params.querySetState);

    const encoder = t.createEncoder('render pass', { occlusionQuerySet });
    encoder.encoder.beginOcclusionQuery(0);
    encoder.encoder.endOcclusionQuery();
    encoder.validateFinishAndSubmitGivenState(t.params.querySetState);
  });

g.test('occlusion_query,query_index')
  .desc(
    `
Tests that begin occlusion query with query index:
- queryIndex {in, out of} range for GPUQuerySet
  `
  )
  .paramsSubcasesOnly(u => u.combine('queryIndex', [0, 2]))
  .fn(t => {
    const occlusionQuerySet = createQuerySetWithType(t, 'occlusion', 2);

    const encoder = t.createEncoder('render pass', { occlusionQuerySet });
    encoder.encoder.beginOcclusionQuery(t.params.queryIndex);
    encoder.encoder.endOcclusionQuery();
    encoder.validateFinish(t.params.queryIndex < 2);
  });

g.test('timestamp_query,query_type_and_index')
  .desc(
    `
Tests that write timestamp to all types of query set on all possible encoders:
- type {occlusion, pipeline statistics, timestamp}
- queryIndex {in, out of} range for GPUQuerySet
- x= {non-pass, compute, render} encoder
  `
  )
  .params(u =>
    u
      .combine('encoderType', ['non-pass', 'compute pass', 'render pass'] as const)
      .combine('type', kQueryTypes)
      .beginSubcases()
      .expand('queryIndex', p => (p.type === 'timestamp' ? [0, 2] : [0]))
  )
  .fn(async t => {
    const { encoderType, type, queryIndex } = t.params;

    await t.selectDeviceForQueryTypeOrSkipTestCase(type);

    const count = 2;
    const querySet = createQuerySetWithType(t, type, count);

    const encoder = t.createEncoder(encoderType);
    encoder.encoder.writeTimestamp(querySet, queryIndex);
    encoder.validateFinish(type === 'timestamp' && queryIndex < count);
  });

g.test('timestamp_query,invalid_query_set')
  .desc(
    `
Tests that write timestamp to a invalid query set that failed during creation:
- x= {non-pass, compute, render} enconder
  `
  )
  .paramsSubcasesOnly(u =>
    u
      .combine('encoderType', ['non-pass', 'compute pass', 'render pass'] as const)
      .combine('querySetState', ['valid', 'invalid'] as const)
  )
  .fn(async t => {
    const { encoderType, querySetState } = t.params;
    await t.selectDeviceForQueryTypeOrSkipTestCase('timestamp');

    const querySet = t.createQuerySetWithState(querySetState, {
      type: 'timestamp',
      count: 2,
    });

    const encoder = t.createEncoder(encoderType);
    encoder.encoder.writeTimestamp(querySet, 0);
    encoder.validateFinish(querySetState !== 'invalid');
  });

g.test('timestamp_query,device_mismatch')
  .desc('Tests writeTimestamp cannot be called with a query set created from another device')
  .paramsSubcasesOnly(u =>
    u
      .combine('encoderType', ['non-pass', 'compute pass', 'render pass'] as const)
      .combine('mismatched', [true, false])
  )
  .unimplemented();
