// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_BROWSER_MOCK_EXTERNAL_PROVIDER_H_
#define EXTENSIONS_BROWSER_MOCK_EXTERNAL_PROVIDER_H_

#include <map>
#include <memory>
#include <string>
#include <utility>

#include "base/files/file_path.h"
#include "base/macros.h"
#include "extensions/browser/external_provider_interface.h"
#include "extensions/common/extension_id.h"
#include "extensions/common/manifest.h"

namespace base {
class Version;
}

namespace extensions {

class MockExternalProvider : public ExternalProviderInterface {
 public:
  MockExternalProvider(VisitorInterface* visitor, Manifest::Location location);
  ~MockExternalProvider() override;

  void UpdateOrAddExtension(const ExtensionId& id,
                            const std::string& version,
                            const base::FilePath& path);
  void UpdateOrAddExtension(std::unique_ptr<ExternalInstallInfoFile> info);
  void RemoveExtension(const ExtensionId& id);

  // ExternalProviderInterface implementation:
  void VisitRegisteredExtension() override;
  bool HasExtension(const std::string& id) const override;
  bool GetExtensionDetails(
      const std::string& id,
      Manifest::Location* location,
      std::unique_ptr<base::Version>* version) const override;
  bool IsReady() const override;
  void ServiceShutdown() override {}

  int visit_count() const { return visit_count_; }
  void set_visit_count(int visit_count) { visit_count_ = visit_count; }

 private:
  using DataMap =
      std::map<ExtensionId, std::unique_ptr<ExternalInstallInfoFile>>;
  DataMap extension_map_;
  Manifest::Location location_;
  VisitorInterface* visitor_;

  // visit_count_ tracks the number of calls to VisitRegisteredExtension().
  // Mutable because it must be incremented on each call to
  // VisitRegisteredExtension(), which must be a const method to inherit
  // from the class being mocked.
  mutable int visit_count_;

  DISALLOW_COPY_AND_ASSIGN(MockExternalProvider);
};

}  // namespace extensions

#endif  // EXTENSIONS_BROWSER_MOCK_EXTERNAL_PROVIDER_H_
