/*
 * Decompiled with CFR 0.152.
 */
package liquibase.command.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.module.ModuleDescriptor;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import liquibase.Scope;
import liquibase.command.AbstractCommandStep;
import liquibase.command.CommandArgumentDefinition;
import liquibase.command.CommandBuilder;
import liquibase.command.CommandDefinition;
import liquibase.command.CommandResultsBuilder;
import liquibase.configuration.ConfigurationValueConverter;
import liquibase.configuration.ConfiguredValue;
import liquibase.configuration.LiquibaseConfiguration;
import liquibase.exception.UnexpectedLiquibaseException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

public class LpmCommandStep
extends AbstractCommandStep {
    public static final String[] COMMAND_NAME = new String[]{"lpm"};
    public static final CommandArgumentDefinition<Boolean> DOWNLOAD_ARG;
    public static final CommandArgumentDefinition<String> LPM_HOME_ARG;
    private static final String LPM_BINARY_NAME = "lpm";
    private static final String DOCS_URL = "http://docs.liquibase.com/LPM";
    private static final String LPM_ERROR_PREFIX = "ERROR: Liquibase Package Manager (LPM)";
    private static final String LPM_DOWNLOAD_PAGE_URL = "https://api.github.com/repos/liquibase/liquibase-package-manager/releases/latest";

    @Override
    public List<Class<?>> requiredDependencies() {
        return new ArrayList();
    }

    @Override
    public String[][] defineCommandNames() {
        return new String[][]{COMMAND_NAME};
    }

    @Override
    public void adjustCommandDefinition(CommandDefinition commandDefinition) {
        commandDefinition.setShortDescription("Initialize and update Liquibase Package Manager (LPM)");
        commandDefinition.setLongDescription("Download, install, and manage Liquibase packages using the Liquibase Package Manager (LPM)");
    }

    @Override
    public void run(CommandResultsBuilder resultsBuilder) throws Exception {
        String lpmHome = this.validateAndResolveLpmHome(resultsBuilder);
        String lpmExecutable = this.buildLpmExecutablePath(lpmHome);
        boolean download = resultsBuilder.getCommandScope().getArgumentValue(DOWNLOAD_ARG);
        if (download) {
            this.downloadOrUpgradeLpm(lpmHome, lpmExecutable);
            return;
        }
        this.checkForLpmInstallation(lpmHome, lpmExecutable);
        ArrayList<String> lpmCommand = new ArrayList<String>();
        String lpmArgsStr = (String)((Object)Scope.getCurrentScope().get((Enum)Scope.Attr.lpmArgs, String.class));
        if (lpmArgsStr != null && !lpmArgsStr.trim().isEmpty()) {
            String[] lpmArgs = lpmArgsStr.trim().split("\\s+");
            Collections.addAll(lpmCommand, lpmArgs);
        } else {
            lpmCommand.add("--help");
        }
        if ((lpmCommand.contains("--help") || lpmCommand.contains("--h")) && lpmCommand.size() == 1) {
            this.printHelpMessage();
        }
        this.runLpmWithCommands(lpmExecutable, lpmCommand);
    }

    private void runLpmWithCommands(String lpmExecutable, List<String> commands) {
        ArrayList<String> lpmExecArgs = new ArrayList<String>();
        lpmExecArgs.add(lpmExecutable);
        lpmExecArgs.addAll(commands);
        try {
            Scope.getCurrentScope().getLog(this.getClass()).fine("Executing LPM command: " + String.join((CharSequence)" ", lpmExecArgs));
            Process process = new ProcessBuilder(lpmExecArgs).redirectErrorStream(true).start();
            try (InputStream stdOut = process.getInputStream();){
                List lines = IOUtils.readLines((InputStream)stdOut, (Charset)StandardCharsets.UTF_8);
                for (String line : lines) {
                    Scope.getCurrentScope().getUI().sendMessage(line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode != 0) {
                throw new UnexpectedLiquibaseException("LPM command failed with exit code: " + exitCode);
            }
        }
        catch (IOException | InterruptedException e) {
            throw new UnexpectedLiquibaseException("Failed to execute LPM command: " + e.getMessage(), e);
        }
    }

    private void downloadOrUpgradeLpm(String lpmHome, String lpmExecutable) {
        this.validateLpmHomeNotEmpty(lpmHome);
        String latestVersion = this.getLatestLpmVersion();
        File lpmExecutableFile = new File(lpmExecutable);
        if (lpmExecutableFile.exists()) {
            String currentVersion = this.getCurrentLpmVersion(lpmExecutable);
            if (currentVersion != null && this.isNewerVersion(latestVersion, currentVersion)) {
                Scope.getCurrentScope().getUI().sendMessage("Upgrading LPM from version " + currentVersion + " to " + latestVersion);
                this.installLpm(lpmHome, latestVersion);
                Scope.getCurrentScope().getUI().sendMessage("LPM has been upgraded to version " + latestVersion + " successfully.");
            } else {
                Scope.getCurrentScope().getUI().sendMessage("LPM is already up to date (version " + (currentVersion != null ? currentVersion : "unknown") + ").");
            }
        } else {
            Scope.getCurrentScope().getUI().sendMessage("Installing LPM version " + latestVersion + "...");
            this.ensureDirectoryExists(lpmHome);
            this.installLpm(lpmHome, latestVersion);
            Scope.getCurrentScope().getUI().sendMessage("LPM has been installed successfully (version " + latestVersion + ").");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String getCurrentLpmVersion(String lpmExecutable) {
        try {
            Process process = new ProcessBuilder(lpmExecutable, "--version").start();
            try (InputStream inputStream = process.getInputStream();){
                Pattern versionPattern;
                Matcher matcher;
                String output = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
                if (output != null && !output.trim().isEmpty() && (matcher = (versionPattern = Pattern.compile("(?:version\\s+)?v?(\\d+\\.\\d+\\.\\d+(?:\\.\\d+)?)")).matcher(output)).find()) {
                    String string = matcher.group(1);
                    return string;
                }
            }
            process.waitFor();
            return null;
        }
        catch (Exception e) {
            Scope.getCurrentScope().getLog(this.getClass()).fine("Could not get current LPM version: " + e.getMessage());
        }
        return null;
    }

    boolean isNewerVersion(String version1, String version2) {
        if (version1 == null || version2 == null) {
            return version1 != null;
        }
        try {
            String cleanVersion1 = this.cleanVersionString(version1);
            String cleanVersion2 = this.cleanVersionString(version2);
            ModuleDescriptor.Version v1 = ModuleDescriptor.Version.parse(cleanVersion1);
            ModuleDescriptor.Version v2 = ModuleDescriptor.Version.parse(cleanVersion2);
            return v1.compareTo(v2) > 0;
        }
        catch (IllegalArgumentException e) {
            Scope.getCurrentScope().getLog(this.getClass()).fine("Failed to parse versions: " + e.getMessage());
            return false;
        }
    }

    private String cleanVersionString(String version) {
        if (version == null) {
            return "0";
        }
        Pattern pattern = Pattern.compile("(\\d+(?:\\.\\d+)*)");
        Matcher matcher = pattern.matcher(version);
        return matcher.find() ? matcher.group(1) : "0";
    }

    private String validateAndResolveLpmHome(CommandResultsBuilder resultsBuilder) {
        LpmHomeResolution resolution = this.resolveLpmHomeFromSources(resultsBuilder);
        return this.validateLpmHomePath(resolution.path, resolution.configuredValue);
    }

    private LpmHomeResolution resolveLpmHomeFromSources(CommandResultsBuilder resultsBuilder) {
        String lpmHome = resultsBuilder.getCommandScope().getArgumentValue(LPM_HOME_ARG);
        ConfiguredValue<String> configuredValue = null;
        if (lpmHome == null || lpmHome.trim().isEmpty()) {
            configuredValue = Scope.getCurrentScope().getSingleton(LiquibaseConfiguration.class).getCurrentConfiguredValue(ConfigurationValueConverter.STRING, null, "liquibase.home");
            lpmHome = configuredValue.getValue();
        }
        if (lpmHome == null || lpmHome.trim().isEmpty()) {
            throw new UnexpectedLiquibaseException("LPM home directory is not configured. Cannot determine where to install LPM.");
        }
        return new LpmHomeResolution(lpmHome, configuredValue);
    }

    private String validateLpmHomePath(String lpmHome, ConfiguredValue<String> configuredValue) {
        Path lpmHomePath = Paths.get(lpmHome, new String[0]).toAbsolutePath();
        File lpmHomeDir = lpmHomePath.toFile();
        if (lpmHomeDir.exists()) {
            this.validateExistingDirectory(lpmHomePath, lpmHomeDir, configuredValue);
        }
        return lpmHomePath.toString();
    }

    private void validateExistingDirectory(Path lpmHomePath, File lpmHomeDir, ConfiguredValue<String> configuredValue) {
        if (!lpmHomeDir.isDirectory()) {
            String source = this.getConfigurationSource(configuredValue);
            throw new UnexpectedLiquibaseException(this.formatLpmErrorMessage("path '%s' set by %s is not a directory", lpmHomePath, source));
        }
        if (!lpmHomeDir.canRead() || !lpmHomeDir.canWrite()) {
            String source = this.getConfigurationSource(configuredValue);
            throw new UnexpectedLiquibaseException(this.formatLpmErrorMessage("was not installed at path of '%s' set by %s", lpmHomePath, source));
        }
    }

    private String getConfigurationSource(ConfiguredValue<String> configuredValue) {
        if (configuredValue == null || configuredValue.getProvidedValue() == null) {
            return "default configuration";
        }
        String provider = configuredValue.getProvidedValue().getProvider().getClass().getSimpleName();
        if (provider.contains("Environment")) {
            return "LIQUIBASE_LPM_HOME environment variable";
        }
        if (provider.contains("SystemProperty")) {
            return "liquibase.lpm.home system property";
        }
        if (provider.contains("CommandLine")) {
            return "--lpm-home command line argument";
        }
        if (provider.contains("PropertiesFile")) {
            return "liquibase.lpm.home in properties file";
        }
        if (provider.contains("Default")) {
            return "default configuration";
        }
        return "configuration";
    }

    private void checkForLpmInstallation(String lpmHome, String lpmExecutable) {
        this.validateLpmHomeNotEmpty(lpmHome);
        Scope.getCurrentScope().getLog(this.getClass()).fine("Checking for LPM at " + lpmExecutable);
        File lpmExecutableFile = new File(lpmExecutable);
        if (!lpmExecutableFile.exists() || !lpmExecutableFile.isFile()) {
            Scope.getCurrentScope().getUI().sendMessage("LPM not found at '" + lpmExecutable + "'. Installing LPM automatically...");
            this.ensureDirectoryExists(lpmHome);
            this.installLpm(lpmHome);
        } else {
            Scope.getCurrentScope().getLog(this.getClass()).info("LPM found at " + lpmExecutable);
        }
    }

    private void installLpm(String lpmHome) {
        ConfiguredValue<String> lpmVersionProperty = Scope.getCurrentScope().getSingleton(LiquibaseConfiguration.class).getCurrentConfiguredValue(ConfigurationValueConverter.STRING, null, "liquibase.lpmVersion");
        String version = lpmVersionProperty != null && lpmVersionProperty.getValue() != null ? lpmVersionProperty.getValue() : this.getLatestLpmVersion();
        this.installLpm(lpmHome, version);
    }

    private void installLpm(String lpmHome, String version) {
        String platform = this.determinePlatform();
        String lpmUrl = String.format("https://github.com/liquibase/liquibase-package-manager/releases/download/v%s/lpm-%s-%s.zip", version, version, platform);
        Scope.getCurrentScope().getUI().sendMessage("Downloading LPM version " + version + " for " + platform + " from " + lpmUrl);
        Path tempFile = null;
        try {
            tempFile = Files.createTempFile("lpm-download-", ".zip", new FileAttribute[0]);
            FileUtils.copyURLToFile((URL)new URL(lpmUrl), (File)tempFile.toFile());
            this.unzipLpm(tempFile, lpmHome);
            Scope.getCurrentScope().getUI().sendMessage("LPM installation completed successfully.");
        }
        catch (Exception e) {
            throw new UnexpectedLiquibaseException("Failed to download or install LPM: " + e.getMessage(), e);
        }
        finally {
            if (tempFile != null) {
                try {
                    Files.deleteIfExists(tempFile);
                }
                catch (IOException e) {
                    Scope.getCurrentScope().getLog(this.getClass()).warning("Could not delete temporary LPM installer: " + String.valueOf(tempFile));
                }
            }
        }
    }

    private String getLatestLpmVersion() {
        String fallbackVersion = "0.2.10";
        try {
            Scope.getCurrentScope().getLog(this.getClass()).fine("Fetching latest LPM version from GitHub API");
            String response = IOUtils.toString((URI)new URI(LPM_DOWNLOAD_PAGE_URL), (Charset)StandardCharsets.UTF_8);
            String version = this.parseVersionFromGitHubResponse(response);
            if (version != null) {
                Scope.getCurrentScope().getLog(this.getClass()).info("Using latest LPM version: " + version);
                return version;
            }
        }
        catch (Exception e) {
            Scope.getCurrentScope().getLog(this.getClass()).warning("Failed to fetch latest LPM version from GitHub: " + e.getMessage() + ", using fallback version");
        }
        Scope.getCurrentScope().getLog(this.getClass()).info("Using fallback LPM version: 0.2.10");
        return "0.2.10";
    }

    private String parseVersionFromGitHubResponse(String jsonResponse) {
        Pattern pattern = Pattern.compile("\"tag_name\"\\s*:\\s*\"v?([^\"]+)\"");
        Matcher matcher = pattern.matcher(jsonResponse);
        if (matcher.find()) {
            String version = matcher.group(1);
            if (version.startsWith("v")) {
                version = version.substring(1);
            }
            return version;
        }
        return null;
    }

    private String determinePlatform() {
        String osName = System.getProperty("os.name").toLowerCase();
        String osArch = System.getProperty("os.arch").toLowerCase();
        if (osName.contains("win")) {
            return "windows";
        }
        if (osName.contains("mac") || osName.contains("darwin")) {
            if (osArch.contains("aarch64") || osArch.contains("arm")) {
                return "darwin-arm64";
            }
            return "darwin";
        }
        if (osName.contains("linux")) {
            if (osArch.contains("s390") || osArch.contains("zarch")) {
                return "s390x";
            }
            if (osArch.contains("aarch64") || osArch.contains("arm64")) {
                return "linux-arm64";
            }
            return "linux";
        }
        Scope.getCurrentScope().getLog(this.getClass()).warning("Unknown OS: " + osName + ", defaulting to linux platform");
        return "linux";
    }

    private void unzipLpm(Path lpmZip, String lpmHome) {
        Scope.getCurrentScope().getLog(this.getClass()).info("Unzipping LPM to " + lpmHome);
        if (!Files.exists(lpmZip, new LinkOption[0])) {
            throw new UnexpectedLiquibaseException("LPM zip file not found: " + String.valueOf(lpmZip));
        }
        try (FileInputStream fis = new FileInputStream(lpmZip.toFile());
             ZipInputStream zis = new ZipInputStream(fis);){
            ZipEntry ze;
            boolean foundExecutable = false;
            while ((ze = zis.getNextEntry()) != null) {
                String fileName = ze.getName();
                if (ze.isDirectory()) continue;
                File newFile = new File(lpmHome, fileName);
                String canonicalDestPath = new File(lpmHome).getCanonicalPath();
                String canonicalFilePath = newFile.getCanonicalPath();
                if (!canonicalFilePath.startsWith(canonicalDestPath + File.separator)) {
                    throw new UnexpectedLiquibaseException("Entry is outside of the target dir: " + fileName);
                }
                Scope.getCurrentScope().getLog(this.getClass()).fine("Extracting: " + fileName);
                File parentDir = newFile.getParentFile();
                if (parentDir != null && !parentDir.exists() && !parentDir.mkdirs()) {
                    throw new UnexpectedLiquibaseException("Could not create directory: " + parentDir.getAbsolutePath());
                }
                try (FileOutputStream fos = new FileOutputStream(newFile);){
                    zis.transferTo(fos);
                }
                if (fileName.toLowerCase().contains(LPM_BINARY_NAME) && !fileName.contains(".") || fileName.endsWith(".exe")) {
                    newFile.setExecutable(true);
                    foundExecutable = true;
                    Scope.getCurrentScope().getLog(this.getClass()).info("Made executable: " + newFile.getAbsolutePath());
                }
                zis.closeEntry();
            }
            if (!foundExecutable) {
                Scope.getCurrentScope().getLog(this.getClass()).warning("No LPM executable found in the extracted files");
            }
        }
        catch (IOException e) {
            throw new UnexpectedLiquibaseException("Failed to extract LPM archive: " + e.getMessage(), e);
        }
    }

    private String buildLpmExecutablePath(String lpmHome) {
        String extension = System.getProperty("os.name").toLowerCase().contains("win") ? ".exe" : "";
        return lpmHome + File.separator + LPM_BINARY_NAME + extension;
    }

    private void validateLpmHomeNotEmpty(String lpmHome) {
        if (lpmHome == null || lpmHome.trim().isEmpty()) {
            throw new UnexpectedLiquibaseException("LPM home directory is not configured. Cannot determine where to install LPM.");
        }
    }

    private void ensureDirectoryExists(String directoryPath) {
        File directory = new File(directoryPath);
        if (!directory.exists() && !directory.mkdirs()) {
            throw new UnexpectedLiquibaseException("Cannot create LPM home directory: " + directoryPath);
        }
        if (directory.exists() && !directory.canWrite()) {
            throw new UnexpectedLiquibaseException("Unable to write to LPM home directory: " + directoryPath);
        }
    }

    private String formatLpmErrorMessage(String messageTemplate, Object ... args) {
        String message = String.format(messageTemplate, args);
        return String.format("%s %s. Please check this path for access, or edit your property value. Learn more at %s", LPM_ERROR_PREFIX, message, DOCS_URL);
    }

    private void printHelpMessage() {
        Scope.getCurrentScope().getUI().sendMessage("Initialize and update Liquibase Package Manager (LPM)\nUsage: liquibase lpm [OPTIONS]\nDownload, install, and manage Liquibase packages using the Liquibase Package\nManager (LPM)\n      --download[=PARAM]   Download and install LPM binary\n                           DEFAULT: false\n                           (defaults file: 'liquibase.command.download' OR\n                             'liquibase.command.lpm.download' , environment\n                             variable: 'LIQUIBASE_COMMAND_DOWNLOAD' OR\n                             'LIQUIBASE_COMMAND_LPM_DOWNLOAD')\n  -h, --help               Show this help message and exit\n      --lpm-home=PARAM     Directory where LPM is installed\n                           (defaults file: 'liquibase.command.lpmHome' OR\n                             'liquibase.command.lpm.lpmHome' , environment\n                             variable: 'LIQUIBASE_COMMAND_LPM_HOME' OR\n                             'LIQUIBASE_COMMAND_LPM_LPM_HOME')\nEach argument contains the corresponding 'configuration key' in parentheses. As\nan alternative to passing values on the command line, these keys can be used as\na basis for configuration settings in other locations.\nAvailable configuration locations, in order of priority:\n- Command line arguments (argument name in --help)\n- Java system properties (configuration key listed above)\n- Environment values (env variable listed above)\n- Defaults file (configuration key OR argument name)");
    }

    static {
        CommandBuilder builder = new CommandBuilder(new String[][]{COMMAND_NAME});
        DOWNLOAD_ARG = builder.argument("download", Boolean.class).description("Download and install LPM binary").defaultValue(false).build();
        LPM_HOME_ARG = builder.argument("lpmHome", String.class).description("Directory where LPM is installed").build();
    }

    private static class LpmHomeResolution {
        final String path;
        final ConfiguredValue<String> configuredValue;

        LpmHomeResolution(String path, ConfiguredValue<String> configuredValue) {
            this.path = path;
            this.configuredValue = configuredValue;
        }
    }
}

