/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.oauthbearer.internals.secured;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.security.auth.login.AppConfigurationEntry;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.security.oauthbearer.JwtRetriever;
import org.apache.kafka.common.security.oauthbearer.JwtRetrieverException;
import org.apache.kafka.common.security.oauthbearer.internals.secured.ConfigurationUtils;
import org.apache.kafka.common.security.oauthbearer.internals.secured.HttpRequestFormatter;
import org.apache.kafka.common.security.oauthbearer.internals.secured.JaasOptionsUtils;
import org.apache.kafka.common.security.oauthbearer.internals.secured.JwtResponseParser;
import org.apache.kafka.common.security.oauthbearer.internals.secured.Retry;
import org.apache.kafka.common.security.oauthbearer.internals.secured.UnretryableException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpJwtRetriever
implements JwtRetriever {
    private static final Logger log = LoggerFactory.getLogger(HttpJwtRetriever.class);
    private static final Set<Integer> UNRETRYABLE_HTTP_CODES = new HashSet<Integer>();
    private final HttpRequestFormatter requestFormatter;
    private SSLSocketFactory sslSocketFactory;
    private URL tokenEndpointUrl;
    private long loginRetryBackoffMs;
    private long loginRetryBackoffMaxMs;
    private Integer loginConnectTimeoutMs;
    private Integer loginReadTimeoutMs;

    public HttpJwtRetriever(HttpRequestFormatter requestFormatter) {
        this.requestFormatter = Objects.requireNonNull(requestFormatter);
    }

    @Override
    public void configure(Map<String, ?> configs, String saslMechanism, List<AppConfigurationEntry> jaasConfigEntries) {
        ConfigurationUtils cu = new ConfigurationUtils(configs, saslMechanism);
        JaasOptionsUtils jou = new JaasOptionsUtils(saslMechanism, jaasConfigEntries);
        this.tokenEndpointUrl = cu.validateUrl("sasl.oauthbearer.token.endpoint.url");
        if (jou.shouldCreateSSLSocketFactory(this.tokenEndpointUrl)) {
            this.sslSocketFactory = jou.createSSLSocketFactory();
        }
        this.loginRetryBackoffMs = cu.validateLong("sasl.login.retry.backoff.ms");
        this.loginRetryBackoffMaxMs = cu.validateLong("sasl.login.retry.backoff.max.ms");
        this.loginConnectTimeoutMs = cu.validateInteger("sasl.login.connect.timeout.ms", false);
        this.loginReadTimeoutMs = cu.validateInteger("sasl.login.read.timeout.ms", false);
    }

    @Override
    public String retrieve() throws JwtRetrieverException {
        String responseBody;
        String requestBody = this.requestFormatter.formatBody();
        Retry<String> retry = new Retry<String>(this.loginRetryBackoffMs, this.loginRetryBackoffMaxMs);
        Map<String, String> headers = this.requestFormatter.formatHeaders();
        try {
            responseBody = retry.execute(() -> {
                HttpURLConnection con = null;
                try {
                    con = (HttpURLConnection)this.tokenEndpointUrl.openConnection();
                    if (this.sslSocketFactory != null && con instanceof HttpsURLConnection) {
                        ((HttpsURLConnection)con).setSSLSocketFactory(this.sslSocketFactory);
                    }
                    String string = HttpJwtRetriever.post(con, headers, requestBody, this.loginConnectTimeoutMs, this.loginReadTimeoutMs);
                    return string;
                }
                catch (IOException e) {
                    throw new ExecutionException(e);
                }
                finally {
                    if (con != null) {
                        con.disconnect();
                    }
                }
            });
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof JwtRetrieverException) {
                throw (JwtRetrieverException)e.getCause();
            }
            throw new KafkaException(e.getCause());
        }
        JwtResponseParser responseParser = new JwtResponseParser();
        return responseParser.parseJwt(responseBody);
    }

    public static String post(HttpURLConnection con, Map<String, String> headers, String requestBody, Integer connectTimeoutMs, Integer readTimeoutMs) throws IOException, UnretryableException {
        HttpJwtRetriever.handleInput(con, headers, requestBody, connectTimeoutMs, readTimeoutMs);
        return HttpJwtRetriever.handleOutput(con);
    }

    private static void handleInput(HttpURLConnection con, Map<String, String> headers, String requestBody, Integer connectTimeoutMs, Integer readTimeoutMs) throws IOException, UnretryableException {
        log.debug("handleInput - starting post for {}", (Object)con.getURL());
        con.setRequestMethod("POST");
        con.setRequestProperty("Accept", "application/json");
        if (headers != null) {
            for (Map.Entry<String, String> header : headers.entrySet()) {
                con.setRequestProperty(header.getKey(), header.getValue());
            }
        }
        con.setRequestProperty("Cache-Control", "no-cache");
        if (requestBody != null) {
            con.setRequestProperty("Content-Length", String.valueOf(requestBody.length()));
            con.setDoOutput(true);
        }
        con.setUseCaches(false);
        if (connectTimeoutMs != null) {
            con.setConnectTimeout(connectTimeoutMs);
        }
        if (readTimeoutMs != null) {
            con.setReadTimeout(readTimeoutMs);
        }
        log.debug("handleInput - preparing to connect to {}", (Object)con.getURL());
        con.connect();
        if (requestBody != null) {
            try (OutputStream os = con.getOutputStream();){
                ByteArrayInputStream is = new ByteArrayInputStream(requestBody.getBytes(StandardCharsets.UTF_8));
                log.debug("handleInput - preparing to write request body to {}", (Object)con.getURL());
                HttpJwtRetriever.copy(is, os);
            }
        }
    }

    static String handleOutput(HttpURLConnection con) throws IOException {
        int responseCode = con.getResponseCode();
        log.debug("handleOutput - responseCode: {}", (Object)responseCode);
        String responseBody = null;
        String errorResponseBody = null;
        try (InputStream is = con.getInputStream();){
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            log.debug("handleOutput - preparing to read response body from {}", (Object)con.getURL());
            HttpJwtRetriever.copy(is, os);
            responseBody = os.toString(StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            try (InputStream is2 = con.getErrorStream();){
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                log.debug("handleOutput - preparing to read error response body from {}", (Object)con.getURL());
                HttpJwtRetriever.copy(is2, os);
                errorResponseBody = os.toString(StandardCharsets.UTF_8);
            }
            catch (Exception e2) {
                log.warn("handleOutput - error retrieving error information", (Throwable)e2);
            }
            log.warn("handleOutput - error retrieving data", (Throwable)e);
        }
        if (responseCode == 200 || responseCode == 201) {
            log.debug("handleOutput - responseCode: {}, error response: {}", (Object)responseCode, errorResponseBody);
            if (responseBody == null || responseBody.isEmpty()) {
                throw new IOException(String.format("The token endpoint response was unexpectedly empty despite response code %d from %s and error message %s", responseCode, con.getURL(), HttpJwtRetriever.formatErrorMessage(errorResponseBody)));
            }
            return responseBody;
        }
        log.warn("handleOutput - error response code: {}, error response body: {}", (Object)responseCode, (Object)HttpJwtRetriever.formatErrorMessage(errorResponseBody));
        if (UNRETRYABLE_HTTP_CODES.contains(responseCode)) {
            throw new UnretryableException(new IOException(String.format("The response code %s and error response %s was encountered reading the token endpoint response; will not attempt further retries", responseCode, HttpJwtRetriever.formatErrorMessage(errorResponseBody))));
        }
        throw new IOException(String.format("The unexpected response code %s and error message %s was encountered reading the token endpoint response", responseCode, HttpJwtRetriever.formatErrorMessage(errorResponseBody)));
    }

    static void copy(InputStream is, OutputStream os) throws IOException {
        int b;
        byte[] buf = new byte[4096];
        while ((b = is.read(buf)) != -1) {
            os.write(buf, 0, b);
        }
    }

    static String formatErrorMessage(String errorResponseBody) {
        if (errorResponseBody == null || errorResponseBody.trim().isEmpty()) {
            return "{}";
        }
        ObjectMapper mapper = new ObjectMapper();
        try {
            JsonNode rootNode = mapper.readTree(errorResponseBody);
            if (!rootNode.at("/error").isMissingNode()) {
                return String.format("{%s - %s}", rootNode.at("/error"), rootNode.at("/error_description"));
            }
            if (!rootNode.at("/errorCode").isMissingNode()) {
                return String.format("{%s - %s}", rootNode.at("/errorCode"), rootNode.at("/errorSummary"));
            }
            return errorResponseBody;
        }
        catch (Exception e) {
            log.warn("Error parsing error response", (Throwable)e);
            return String.format("{%s}", errorResponseBody);
        }
    }

    static {
        UNRETRYABLE_HTTP_CODES.add(400);
        UNRETRYABLE_HTTP_CODES.add(401);
        UNRETRYABLE_HTTP_CODES.add(402);
        UNRETRYABLE_HTTP_CODES.add(403);
        UNRETRYABLE_HTTP_CODES.add(404);
        UNRETRYABLE_HTTP_CODES.add(405);
        UNRETRYABLE_HTTP_CODES.add(406);
        UNRETRYABLE_HTTP_CODES.add(407);
        UNRETRYABLE_HTTP_CODES.add(409);
        UNRETRYABLE_HTTP_CODES.add(410);
        UNRETRYABLE_HTTP_CODES.add(411);
        UNRETRYABLE_HTTP_CODES.add(412);
        UNRETRYABLE_HTTP_CODES.add(413);
        UNRETRYABLE_HTTP_CODES.add(414);
        UNRETRYABLE_HTTP_CODES.add(415);
        UNRETRYABLE_HTTP_CODES.add(501);
        UNRETRYABLE_HTTP_CODES.add(505);
    }
}

