#!/bin/sh
# Script to install a catalog in the centralized SGML catalog
# Send any comments to Eric Bischoff <eric@caldera.de>
# This program is under GPL license. See LICENSE file for details.

# Set help message
SGML_HELP_MESSAGE="\
Usage: `basename $0` [<option>] <action>
where <option> is:
  -d|--delegate:			Use DELEGATE instead of CATALOG
and where <action> is:
  -a|--add <centralized> <ordinary>:	Declare ordinary catalog in the centralized catalog
  -r|--remove <centralized> <ordinary>:	Remove ordinary catalog from the centralized catalog
  -h, --help:				Print this help message and exit
  -v, --version:			Print the version number and exit
"

# Set version message
SGML_VERSION_MESSAGE="sgml-common version 0.7.1 (install-catalog version 1.0)"

# Set type of pointer
SGML_POINTER="CATALOG"

# Set action to be performed
SGML_ACTION=""

# Set catalogs
SGML_CENTRALIZED=""
SGML_ORDINARY=""

# Process options
case $1 in
   -d|--delegate) SGML_POINTER="DELEGATE"
                shift 1
                ;;
esac

# Process actions
case $1 in
   -a|--add)    SGML_ACTION="addition"
		SGML_CENTRALIZED=$2
		SGML_ORDINARY=$3
		;;
   -r|--remove)	if [ -z "$3" -o "$3" = "--version" ]
		then
		  echo "install-catalog: Old syntax; doing nothing"
		  exit 0
		fi
   		SGML_ACTION="removal"
		SGML_CENTRALIZED=$2
		SGML_ORDINARY=$3
		;;
   -h|--help)	echo $SGML_HELP_MESSAGE
		exit 0
		;;
   -v|--version) echo $SGML_VERSION_MESSAGE
	 	exit 0
		;;
   --install)   echo "install-catalog: Old syntax; doing nothing"
		exit 0
		;;
   *)		echo $SGML_HELP_MESSAGE >&2
		exit 1
		;;
esac

# Check that the super catalog can be created and changed and deleted
if [ ! -w /etc/sgml ]
then
  echo "`basename $0`: unable to write in /etc/sgml." >&2
  exit 2
fi
case $SGML_ACTION in
   addition)
	if [ -e /etc/sgml/catalog -a ! -w /etc/sgml/catalog ]
	then
	  echo "`basename $0`: can not modify \"/etc/sgml/catalog\"." >&2
	  exit 2
	fi
	;;
   removal)
	if [ ! -w /etc/sgml/catalog ]
	then
	  echo "`basename $0`: can not modify \"/etc/sgml/catalog\"." >&2
	  exit 2
	fi
	;;
esac

# Check that the centralized catalog can be created, changed and deleted
if [ -z "$SGML_CENTRALIZED" ]
then
  echo $SGML_HELP_MESSAGE >&2
  exit 1
fi
case $SGML_ACTION in
   addition)
	if [ -e $SGML_CENTRALIZED -a ! -w $SGML_CENTRALIZED ]
	then
	  echo "`basename $0`: can not modify \"$SGML_CENTRALIZED\"." >&2
	  exit 2
	fi
	;;
   removal)
	if [ ! -w $SGML_CENTRALIZED ]
	then
	  echo "`basename $0`: can not modify \"$SGML_CENTRALIZED\"." >&2
	  exit 2
	fi
	;;
esac

# Check that we have at least one ordinary package to process
if [ -z "$SGML_ORDINARY" ]
then
  echo $SGML_HELP_MESSAGE >&2
  exit 1
fi
case $SGML_ACTION in
   addition)
	if [ ! -s $SGML_ORDINARY ]
	then
	  echo "`basename $0`: \"$SGML_ORDINARY\" does not exist or is empty." >&2
	  exit 2
	fi
	;;
esac

# Installation or removal of pointers
case $SGML_ACTION in
    addition)
	echo "`basename $0`: addition of $SGML_ORDINARY in $SGML_CENTRALIZED"
	if grep -q $SGML_ORDINARY $SGML_CENTRALIZED 2>/dev/null
	then
	  echo "Warning: $SGML_ORDINARY is already installed in the centralized catalog $SGML_CENTRALIZED" >&2
	else
	  echo "$SGML_POINTER $SGML_ORDINARY" >> $SGML_CENTRALIZED
	fi
	grep -q $SGML_CENTRALIZED /etc/sgml/catalog 2>/dev/null
	if [ $? -ne 0 ]
	then
	  echo "`basename $0`: addition of $SGML_CENTRALIZED in /etc/sgml/catalog"
	  echo "$SGML_POINTER $SGML_CENTRALIZED" >> /etc/sgml/catalog
	fi
	;;
   removal)
	echo "`basename $0`: removal of $SGML_ORDINARY from $SGML_CENTRALIZED"
	if grep -q $SGML_ORDINARY $SGML_CENTRALIZED 2>/dev/null
	then
          sed -e "\:$SGML_POINTER $SGML_ORDINARY:d" < $SGML_CENTRALIZED > ${SGML_CENTRALIZED}.new
          mv ${SGML_CENTRALIZED}.new $SGML_CENTRALIZED
	else
	  echo "Warning: $SGML_ORDINARY was not found in the centralized catalog $SGML_CENTRALIZED" >&2
	fi
	if [ ! -s $SGML_CENTRALIZED ]
	then
	  rm $SGML_CENTRALIZED
	  echo "`basename $0`: removal of $SGML_CENTRALIZED from /etc/sgml/catalog"
          sed -e "\:$SGML_POINTER $SGML_CENTRALIZED:d" < /etc/sgml/catalog > /etc/sgml/catalog.new
          mv /etc/sgml/catalog.new /etc/sgml/catalog
	fi
	;;
esac

exit 0
